/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.managed;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IEndpointReference;
import org.eclipse.higgins.icard.ITokenService;
import org.eclipse.higgins.icard.InvalidClaimException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.ReadOnlyObjectException;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.provider.cardspace.common.EndpointReference;
import org.eclipse.higgins.icard.provider.cardspace.common.ManagedCard;
import org.eclipse.higgins.icard.provider.cardspace.common.TokenService;
import org.eclipse.higgins.icard.provider.cardspace.common.CredentialDescriptor;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.DateConvertor;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.IdASContext;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.INode;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.NoSuchNodeException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Implementation of managed CardSpace-interoperable ICard stored within IdAS
 * context
 */
public class IdASBasedManagedCard extends ManagedCard {
	private Log log = LogFactory.getLog(IdASBasedManagedCard.class);

	INode card_ = null;
	
	private boolean editMode = false;;

	public IdASBasedManagedCard() {
	}

	/**
	 * @param provider
	 * @param card
	 * @throws Exception
	 */
	public IdASBasedManagedCard(IdASBasedManagedCardProvider provider, INode card) throws Exception {
		if (provider == null)
			throw new IllegalArgumentException("Parameter \"provider\" is null");
		if (card == null)
			throw new IllegalArgumentException("Parameter \"card\" is null");
		description_ = "IdAS-based CardSpace-interoperable managed I-Card";
		provider_ = provider;
		card_ = card;
		initFromDS(card_);
	}

	/**
	 * @param provider
	 * @param card
	 * @throws Exception
	 */
	public IdASBasedManagedCard(IdASBasedManagedCardProvider provider, IContext context, Element card) throws Exception {
		if (provider == null)
			throw new IllegalArgumentException("Parameter \"provider\" is null");
		if (card == null)
			throw new IllegalArgumentException("Parameter \"card\" is null");
		description_ = "IdAS-based CardSpace-interoperable managed I-Card";
		provider_ = provider;
		initFromXML(card);
		try {
			try {
				card_ = context.getNode(id_.toString());
			} catch (NoSuchNodeException e) {
				card_ = context.addNode(IdASContext.ICARD_ManagedInformationCard, id_.toString());
			}
			saveCard();
			context.applyUpdates();
		}
		catch(Exception e) {
			try {
				context.cancelUpdates();
			}
			catch(Exception e1) {
				log.error(e1);
			}
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/**
	 * @param card
	 * @throws Exception
	 */
	public void initFromDS(INode card) throws Exception {
		initIsSelfIssued(card);
		initLanguage(card);
		initCardId(card);
		initCardVersion(card);
		initCardName(card);
		initCardImage(card);
		initCardImageType(card);
		initIssuer(card);
		initIssuerName(card);
		initTimeIssued(card);
		initTimeExpires(card);
		initSupportedTokenTypes(card);
		initClaimTypes(card);
		initTokenServiceList(card);
		initRequireAppliesTo(card);
		initPrivacyNotice(card);
		initHashSalt(card);
		initTimeLastUpdated(card);
		initMasterKey(card);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initIsSelfIssued(INode card) throws IdASException, CardException, ClassCastException {
		isSelfIssued_ = false;
		Boolean issi = (Boolean) getSimpleValueData(card, IdASContext.ICARD_isSelfIssued);
		if (issi != null && issi.booleanValue() == true)
			throw new CardException("Can not init managed card with Node storing data of self-issued (Personal) card");
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initLanguage(INode card) throws IdASException, CardException, ClassCastException {
		language_ = (String) getSimpleValueData(card, IdASContext.ICARD_language);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardId(INode card) throws IdASException, CardException, ClassCastException {
		String id = (String)getSimpleValueData(card, IdASContext.HIGGINS_uniqueIdentifier);
		try {
			id_ = new URI(id);
		} catch (URISyntaxException e) {
			log.error(e);
			throw new CardException(e);
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardVersion(INode card) throws IdASException, CardException, ClassCastException {
		version_ = (String) getSimpleValueData(card, IdASContext.ICARD_version);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardName(INode card) throws IdASException, CardException, ClassCastException {
		name_ = (String) getSimpleValueData(card, IdASContext.ICARD_name);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardImage(INode card) throws IdASException, CardException, ClassCastException {
		image_ = (byte[]) getSimpleValueData(card, IdASContext.ICARD_image);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardImageType(INode card) throws IdASException, CardException, ClassCastException {
		imageMimeType_ = (String) getSimpleValueData(card, IdASContext.ICARD_imageMimeType);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initIssuer(INode card) throws IdASException, CardException, ClassCastException {
		issuer_ = (String) getSimpleValueData(card, IdASContext.ICARD_issuer);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initIssuerName(INode card) throws IdASException, CardException, ClassCastException {
		issuerName_ = (String) getSimpleValueData(card, IdASContext.ICARD_issuerName);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initTimeIssued(INode card) throws IdASException, CardException {
		timeIssued_ = null;
		Object obj = getSimpleValueData(card, IdASContext.ICARD_timeIssued);
		if (obj != null) {
			if (obj instanceof Date)
				timeIssued_ = (Date) obj;
			else
				timeIssued_ = DateConvertor.parse(obj.toString());
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initTimeExpires(INode card) throws IdASException, CardException {
		timeExpires_ = null;
		Object obj = getSimpleValueData(card, IdASContext.ICARD_expiredTime);
		if (obj != null) {
			if (obj instanceof Date)
				timeExpires_ = (Date) obj;
			else
				timeExpires_ = DateConvertor.parse(obj.toString());
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initSupportedTokenTypes(INode card) throws IdASException, CardException {
		supportedTokenTypes_ = new ArrayList();
		IAttribute attr = card.getAttribute(IdASContext.ICARD_supportedTokenType);
		Iterator itr = attr.getValues();
		while (itr.hasNext()) {
			IAttributeValue val = (IAttributeValue) itr.next();
			if (val.isSimple()) {
				ISimpleAttrValue sv = (ISimpleAttrValue) val;
				if (sv != null) {
					Object obj = sv.getData();
					if (obj != null) {
						if (obj instanceof URI)
							supportedTokenTypes_.add(obj);
						else {
							try {
								supportedTokenTypes_.add(new URI(obj.toString()));
							} catch (URISyntaxException e) {
								log.error(e);
								throw new CardException(e);
							}
						}
					}
				}
			} else
				throw new CardException("Attribute " + IdASContext.ICARD_supportedTokenType + " contains non simple value");
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initClaimTypes(INode card) throws IdASException, CardException {
		claimTypes_ = new ArrayList();
		IAttribute attr = card.getAttribute(IdASContext.ICARD_supportedClaimType);
		Iterator itr = attr.getValues();
		while (itr.hasNext()) {
			IAttributeValue val = (IAttributeValue) itr.next();
			if (val.isSimple() == false) {
				IComplexAttrValue cv = (IComplexAttrValue) val;
				if (cv != null) {
					URI type = (URI) getSimpleValueData(cv, IdASContext.ICARD_claimTypeType);
					String name = (String) getSimpleValueData(cv, IdASContext.ICARD_claimTypeName);
					String description = (String) getSimpleValueData(cv, IdASContext.ICARD_claimTypeDescription);
					ClaimType ct = new ClaimType(type, name, description);
					claimTypes_.add(ct);
				}
			}
		}
	}

	/**
	 * @param card
	 * @throws ClassCastException
	 * @throws IdASException
	 * @throws CardException
	 * @throws IllegalArgumentException
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */
	private void initTokenServiceList(INode card) throws ClassCastException, IdASException, CardException, IllegalArgumentException,
			ParserConfigurationException, SAXException, IOException {
		tokenServices_ = new ArrayList();
		IAttribute attr = card.getAttribute(IdASContext.ICARD_tokenService);
		Iterator itr = attr.getValues();
		while (itr.hasNext()) {
			IAttributeValue val = (IAttributeValue) itr.next();
			if (val.isSimple() == false) {
				IComplexAttrValue cv = (IComplexAttrValue)val;
				URI epAddress = (URI) getSimpleValueData(cv, IdASContext.ICARD_endpoint_address);
				String epMetadata = (String) getSimpleValueData(cv, IdASContext.ICARD_endpoint_metadata);
				String epIdentity = (String) getSimpleValueData(cv, IdASContext.ICARD_endpoint_identity);
				String ucHint = (String) getSimpleValueData(cv, IdASContext.ICARD_credential_displayCredentialHint);
				String ucCredential = (String) getSimpleValueData(cv, IdASContext.ICARD_credential_credential);
				EndpointReference endpoint = new EndpointReference(epAddress, epMetadata, epIdentity);
				CredentialDescriptor credential = new CredentialDescriptor(ucHint, ucCredential);
				TokenService tc = new TokenService(endpoint, credential);
				tokenServices_.add(tc);
			}
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initRequireAppliesTo(INode card) throws IdASException, CardException {
		requireAppliesTo_ = null;
		Object obj = getSimpleValueData(card, IdASContext.ICARD_requireAppliesTo);
		if (obj != null) {
			if (obj instanceof Boolean)
				requireAppliesTo_ = (Boolean) obj;
			else
				requireAppliesTo_ = Boolean.valueOf(obj.toString());
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 * @throws ClassCastException
	 */
	private void initPrivacyNotice(INode card) throws IdASException, CardException, ParserConfigurationException, SAXException, IOException,
			ClassCastException {
		privacyNotice_ = null;
		String obj = (String) getSimpleValueData(card, IdASContext.ICARD_privacyNotice);
		if (obj != null) {
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			dbf.setIgnoringComments(true);
			dbf.setIgnoringElementContentWhitespace(true);
			dbf.setNamespaceAware(true);
			dbf.setValidating(false);
			DocumentBuilder db = dbf.newDocumentBuilder();
			StringReader sisMetadata = new StringReader(obj);
			InputSource isMetadata = new InputSource(sisMetadata);
			Document mdDoc = db.parse(isMetadata);
			privacyNotice_ = mdDoc.getDocumentElement();
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initHashSalt(INode card) throws IdASException, CardException, ClassCastException {
		hashSalt_ = (byte[]) getSimpleValueData(card, IdASContext.ICARD_hashSalt);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initTimeLastUpdated(INode card) throws IdASException, CardException {
		timeLastUpdated_ = null;
		Object obj = getSimpleValueData(card, IdASContext.ICARD_timeLastUpdated);
		if (obj != null) {
			if (obj instanceof Date)
				timeLastUpdated_ = (Date) obj;
			else
				timeLastUpdated_ = DateConvertor.parse(obj.toString());
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initMasterKey(INode card) throws IdASException, CardException, ClassCastException {
		masterKey_ = (byte[]) getSimpleValueData(card, IdASContext.ICARD_masterKey);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setName(java.lang.String)
	 */
	public void setName(String name) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (name == null || name.trim().length() == 0)
			throw new CardException("Couldn't set empty card name.");
		String oldName = name_;
		try {
			name_ = name.trim();
			saveCardName();
			saveTimeLastUpdated();
		} catch (Exception e) {
			name_ = oldName;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setImage(byte[], java.lang.String)
	 */
	public void setImage(byte[] image, String imageMimeType) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (image == null)
			throw new CardException("Couldn't set null card image.");
		if (imageMimeType == null)
			throw new CardException("Couldn't set null card image MIME type.");
		byte[] oldImage = image_;
		String oldImageType = imageMimeType_;
		image_ = image;
		imageMimeType_ = imageMimeType;
		try {
			saveCardImage();
			saveCardImageType();
			saveTimeLastUpdated();
		} catch (Exception e) {
			image_ = oldImage;
			imageMimeType_ = oldImageType;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setIssuerName(java.lang.String)
	 */
	public void setIssuerName(String name) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (name == null || name.trim().length() == 0)
			throw new CardException("Couldn't set empty card issuer name.");
		String oldName = issuerName_;
		try {
			issuerName_ = name.trim();
			saveIssuerName();
			saveTimeLastUpdated();
		} catch (Exception e) {
			issuerName_ = oldName;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setExpiredTime(java.util.Date)
	 */
	public void setTimeExpires(Date date) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		Date oldDate_ = timeExpires_; 
		try {
			timeExpires_ = date;
			saveTimeExpires();
			saveTimeLastUpdated();
		} catch (Exception e) {
			timeExpires_ = oldDate_;
			log.error(e);
			throw new CardException(e);
		}
	}

	/**
	 * @throws Exception
	 */
	private void saveCard() throws Exception {
		saveIsSelfIssued();
		saveLanguage();
		saveCardId();
		saveCardVersion();
		saveCardName();
		saveCardImage();
		saveCardImageType();
		saveIssuer();
		saveIssuerName();
		saveTimeIssued();
		saveTimeExpires();
		saveSupportedTokenTypes();
		saveClaimTypes();
		saveTokenServiceList();
		saveRequireAppliesTo();
		savePrivacyNotice();
		saveHashSalt();
		saveTimeLastUpdated();
		saveMasterKey();
	}

	/**
	 * @throws IdASException
	 */
	private void saveIsSelfIssued() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_isSelfIssued);
		attr.addSimpleValue(null, Boolean.valueOf(false));
	}

	/**
	 * @throws IdASException
	 */
	private void saveLanguage() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_language);
		attr.addSimpleValue(null, language_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardId() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.HIGGINS_uniqueIdentifier);
		attr.addSimpleValue(null, id_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardVersion() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_version);
		attr.addSimpleValue(null, version_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardName() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_name);
		attr.addSimpleValue(null, name_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardImage() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_image);
		attr.addSimpleValue(null, image_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardImageType() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_imageMimeType);
		attr.addSimpleValue(null, imageMimeType_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveIssuer() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_issuer);
		attr.addSimpleValue(null, issuer_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveIssuerName() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_issuerName);
		attr.addSimpleValue(null, issuerName_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveTimeIssued() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_timeIssued);
		attr.addSimpleValue(null, timeIssued_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveTimeExpires() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_expiredTime);
		attr.addSimpleValue(null, timeExpires_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveSupportedTokenTypes() throws IdASException {
		IAttribute attr = card_.getAttribute(IdASContext.ICARD_supportedTokenType);
		Iterator olds = attr.getValues();
		while (olds.hasNext()) {
			IAttributeValue old = (IAttributeValue) olds.next();
			old.remove();
		}
		Iterator it = supportedTokenTypes_.iterator();
		while (it.hasNext()) {
			URI uri = (URI) it.next();
			attr.addSimpleValue(null, uri);
		}
	}

	/**
	 * @throws IdASException
	 */
	private void saveClaimTypes() throws IdASException {
		IAttribute attr = card_.getAttribute(IdASContext.ICARD_supportedClaimType);
		Iterator olds = attr.getValues();
		while (olds.hasNext()) {
			IAttributeValue old = (IAttributeValue) olds.next();
			old.remove();
		}
		Iterator it = claimTypes_.iterator();
		while (it.hasNext()) {
			IClaimType claimType = (IClaimType) it.next();
			IComplexAttrValue val = attr.addComplexValue(null);
			val.getSingleValuedAttribute(IdASContext.ICARD_claimTypeType).addSimpleValue(null, claimType.getType());
			val.getSingleValuedAttribute(IdASContext.ICARD_claimTypeDescription).addSimpleValue(null, claimType.getDescription());
			val.getSingleValuedAttribute(IdASContext.ICARD_claimTypeName).addSimpleValue(null, claimType.getDisplayName());
		}
	}
	
	
	/**
	 * @throws IdASException 
	 * @throws TransformerException 
	 */
	private void saveTokenServiceList() throws IdASException, TransformerException {
		IAttribute attr = card_.getAttribute(IdASContext.ICARD_tokenService);
		Iterator olds = attr.getValues();
		while (olds.hasNext()) {
			IAttributeValue old = (IAttributeValue) olds.next();
			old.remove();
		}
		Iterator it = tokenServices_.iterator();
		while (it.hasNext()) {
			ITokenService tokenService = (ITokenService) it.next();
			IComplexAttrValue val = attr.addComplexValue(null);

			IEndpointReference er = tokenService.getEndpointReference();
			if (er != null) {
				val.getSingleValuedAttribute(IdASContext.ICARD_endpoint_address).addSimpleValue(null, er.getAddress());
				String metadataStr = XMLUtils.elementToString(er.getMetadata());
				val.getSingleValuedAttribute(IdASContext.ICARD_endpoint_metadata).addSimpleValue(null, metadataStr);
				String identityStr = XMLUtils.elementToString(er.getIdentity());
				val.getSingleValuedAttribute(IdASContext.ICARD_endpoint_identity).addSimpleValue(null, identityStr);
			}
			ICredentialDescriptor uc = tokenService.getUserCredential();
			if (uc != null) {
				val.getSingleValuedAttribute(IdASContext.ICARD_credential_displayCredentialHint).addSimpleValue(null, uc.getDisplayCredentialHint());
				String credentialStr = XMLUtils.elementToString(uc.asXML());
				val.getSingleValuedAttribute(IdASContext.ICARD_credential_credential).addSimpleValue(null, credentialStr);
			}
		}
	}

	/**
	 * @throws IdASException
	 */
	private void saveRequireAppliesTo() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_requireAppliesTo);
		attr.addSimpleValue(null, requireAppliesTo_);
	}

	/**
	 * @throws IdASException
	 * @throws TransformerException
	 */
	private void savePrivacyNotice() throws IdASException, TransformerException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_privacyNotice);
		if (privacyNotice_ != null) {
			attr.addSimpleValue(null, XMLUtils.elementToString(privacyNotice_));
		}
		else
			attr.addSimpleValue(null, null);
	}

	/**
	 * @throws IdASException
	 */
	private void saveHashSalt() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_hashSalt);
		attr.addSimpleValue(null, hashSalt_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveTimeLastUpdated() throws IdASException {
		timeLastUpdated_ = new Date();
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_timeLastUpdated);
		attr.addSimpleValue(null, timeLastUpdated_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveMasterKey() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_masterKey);
		attr.addSimpleValue(null, masterKey_);
	}

	/**
	 * @param val
	 * @param attribure
	 * @return
	 * @throws CardException
	 */
	private Object getSimpleValueData(IComplexAttrValue val, URI attribure) throws CardException {
		try {
			ISingleValuedAttribute attr = val.getSingleValuedAttribute(attribure);
			return getSimpleValueData(attr);
		} catch (IdASException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/**
	 * @param subj
	 * @param attribure
	 * @return
	 * @throws CardException
	 */
	private Object getSimpleValueData(INode subj, URI attribure) throws CardException {
		try {
			ISingleValuedAttribute attr = subj.getSingleValuedAttribute(attribure);
			return getSimpleValueData(attr);
		} catch (IdASException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/**
	 * @param attr
	 * @return
	 * @throws IdASException
	 * @throws CardException
	 */
	private Object getSimpleValueData(ISingleValuedAttribute attr) throws IdASException, CardException {
		if (attr != null) {
			IAttributeValue v = attr.getValue();
			if (v != null) {
				if (v.isSimple()) {
					ISimpleAttrValue sv = (ISimpleAttrValue) v;
					return sv.getData();
				} else
					throw new CardException("Attribute " + (attr.getAttrID() != null ? attr.getAttrID().toString() : "") + " contains non simple value");
			}
		}
		return null;
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	public INode getCardSubject() {
		return card_;
	}

	public void applyUpdates() throws InvalidStateException, CardException {
		if (!editMode) {
			throw new InvalidStateException("Not in edit mode.");
		} else {
			try {
				card_.getContext().applyUpdates();
				editMode = false;
			} catch (IdASException e) {
				log.error(e);
				throw new CardException(e.getMessage());
			}
		}
	}

	public void beginUpdates() throws InvalidStateException {
		if (editMode) {
			throw new InvalidStateException("Already in edit mode.");
		} else {
			editMode = true;
		}
	}

	public void cancelUpdates() throws InvalidStateException {
		if (!editMode) {
			throw new InvalidStateException("Not in edit mode.");
		} else {
			try {
				card_.getContext().cancelUpdates();
				editMode = false;
			} catch (IdASException e) {
				log.error(e);
				throw new InvalidStateException(e.getMessage());
			}
		}
	}

	public boolean isEditMode() {
		return editMode;
	}

	public boolean isEditable() {
		return false;
	}

	public IClaim createClaim(String type) throws InvalidTypeException, ReadOnlyObjectException {
		throw new ReadOnlyObjectException();
	}

	public IClaim setClaim(IClaim copyFrom) throws InvalidClaimException, InvalidTypeException, ReadOnlyObjectException {
		throw new ReadOnlyObjectException();
	}

	public boolean isClaimsRetrieved() {
		// TODO Need to implement!!!!
		return false;
	}

}
