/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.personal;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;

import org.apache.axiom.om.util.UUIDGenerator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.configuration.api.IConfigurableComponent;
import org.eclipse.higgins.configuration.api.ISettingDescriptor;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.ITemplateValue;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCardProvider;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.IPCardManagedBean;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.IUserCredentials;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.PCardEntity;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.UsernamePasswordCredentials;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.PersonalCardTemplateHelper;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.registry.IConfiguration;
import org.eclipse.higgins.registry.MapConfiguraton;
import org.eclipse.higgins.registry.RegistryConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class CardProvider extends PersonalCardProvider implements ICardProvider, IConfigurableComponent {
	private Log log = LogFactory.getLog(CardProvider.class);

	private static byte[] defaultImage = null;

	private static final String DEFAULT_IMAGE_FILE = "defaultImage.file";

	private static final String DEFAULT_IMAGE_TYPE = "defaultImage.mimeType";

	private String id_ = "org.eclipse.higgins.icard.provider.cardspace.personal.db";

	private String name_ = "CardSpace-compatable personal I-Card Provider Plug-in";

	private IConfiguration config = null;

	private IPCardManagedBean mbean = null;

	private String defaultImageFileType = null;

	private String defaultImageFileName = null;


	/**
	 * @return
	 */
	private HashMap convertConfigurationToMap(IConfiguration config) {
		HashMap ht = new HashMap();
		Iterator itr = config.getPropertyNames();
		while (itr.hasNext()) {
			String key = itr.next().toString();
			Object value = config.getProperty(key);
			ht.put(key, value);
		}
		return ht;
	}

	public void configure(Map mapGlobalSettings, String strComponentName, Map mapComponentSettings,
			ISettingDescriptor componentDescriptor, ISettingDescriptor globalDescriptor) throws Exception {
		log.info("Initializing provider " + getID() + " from registry configuration as IConfigurableComponent.");
		config = new MapConfiguraton(mapComponentSettings);
		init(mapComponentSettings);
		log.info("Provider " + getID() + " was initialized from registry configuration as IConfigurableComponent.");
	}

	/**
	 * Initialize provider from registry configuration
	 * 
	 * @throws CardException
	 */
	private synchronized void initFromRegistryConfig() throws CardException {
		if (mbean == null) {
			try {
				log.info("Initializing provider " + getID() + " from registry configuration.");
				HashMap configMap = convertConfigurationToMap(getConfiguration());
				init(configMap);
				log.info("Provider " + getID() + " was initialized from registry configuration.");
			} catch (Exception e) {
				log.error(e, e);
				throw new CardException(e);
			}
		}
	}

	private synchronized void init(Map config) throws Exception {
		defaultImageFileType = (String) config.get(DEFAULT_IMAGE_TYPE);
		defaultImageFileName = (String) config.get(DEFAULT_IMAGE_FILE);
		String managedBeanClassName = (String) config.get("managedbean.classname");
		if (managedBeanClassName == null)
			throw new CardException("Can not find \"managedbean.classname\" configuration property");
		Class cls = Class.forName(managedBeanClassName);
		mbean = (IPCardManagedBean)cls.newInstance();
		mbean.init(config);
	}
	
	
	public IPCardManagedBean getManagedBean() throws CardException {
		if (mbean == null) {
			initFromRegistryConfig();
		}
		return mbean;
	}

	public void setID(String id) throws Exception {
		id_ = id;
	}

	public Iterator getICards(CallbackHandler authHandler) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		List cards = new ArrayList();
		IUserCredentials userCrds = authenticate(authHandler);
		List lst = getManagedBean().getPCards(userCrds);
		for (int i = 0, j = lst.size(); i < j; i++) {
			PCardEntity crd = (PCardEntity) lst.get(i);
			PCard card = new PCard(this, userCrds, crd);
			cards.add(card);
		}
		return cards.iterator();
	}

	public ICard importCard(CallbackHandler authHandler, String filename) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (filename == null)
			throw new CardException("Parameter \"filename\" is null");
		FileInputStream fis;
		try {
			fis = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		// TODO need to separate import of different file types
		if (filename.toUpperCase().endsWith(".CRDS")) {
			importFromCrdsFile(authHandler, fis);
			return null;
		}
		else
			throw new CardException("Unsupported file extension.");
	}

	private void importFromCrdsFile(CallbackHandler authHandler, InputStream is) throws CardException {
		Document cardBackup;
		PasswordCallback pc = new PasswordCallback("Enter password for imported backup file: ", false);
		Callback[] callbacks = new Callback[] { pc };
		try {
			authHandler.handle(callbacks);
			String password = new String(pc.getPassword());
			pc.clearPassword();
			cardBackup = CardCryptography.decrypt(is, password);
		} catch (Exception e) {
			log.error(e, e);
			throw new CardException(e.getMessage());
		}
		List cardsList = new ArrayList();
		NodeList newCards = cardBackup.getDocumentElement().getChildNodes();
		IUserCredentials userCreds = authenticate(authHandler);
		for (int i = 0, j = newCards.getLength(); i < j; i++) {
			Node nd = newCards.item(i);
			if (nd.getNodeType() == Node.ELEMENT_NODE) {
				Element elm = (Element) nd;
				try {
					PCard crd = new PCard(this, userCreds, elm);
					cardsList.add(crd);
				} catch (Exception e) {
					log.error(e, e);
				}
			}
		}
	}

	public void deleteCard(CallbackHandler authHandler, ICard card) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		IUserCredentials userCreds = authenticate(authHandler);
		getManagedBean().deletePCard(card.getID(), userCreds);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getID()
	 */
	public String getID() {
		return id_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getName()
	 */
	public String getName() {
		return name_;
	}

	public String getDescription() {
		return "JDBC-based CardSpace-interoperable personal I-Card provider";
	}

	public ICard importICard(CallbackHandler authHandler, IElement element) throws CardException {
		try {
			Element crd = (Element) element.getAs(Element.class);
			IUserCredentials userCrds = authenticate(authHandler);
			PCard pc = new PCard(this, userCrds, crd);
			return pc;
		} catch (CardException e) {
			throw e;
		} catch (Exception e) {
			log.error(e, e);
			throw new CardException(e);
		}
	}

	protected ICard getICardByID(CallbackHandler authHandler, String cardID) throws CardException {
		IUserCredentials userCreds = authenticate(authHandler);
		PCardEntity crd = getManagedBean().getPCard(cardID, userCreds);
		if (crd != null)
			return new PCard(this, userCreds, crd);
		else
			return null;
	}

	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid) throws CardException {
		if (this.getID().equals(cuid.getProviderID()) == false) {
			log.debug("ProviderID of requested CUID (" + cuid.getProviderID() + ") is not equal to this providerID ("
					+ this.getID() + ").");
			return null;
		}
		ICard crd = getICardByID(authHandler, cuid.getCardID());
		if (crd == null) {
			log.debug("Can not find any card with ID = " + cuid.getCardID());
			return null;
		} else {
			String issuer = cuid.getIssuer();
			if (issuer != null && issuer.equals(crd.getIssuer()))
				return crd;
			else {
				log.debug("Wrong issuer for card with ID = " + cuid.getCardID() + ". Requested issuer = " + issuer
						+ ", real issuer = " + crd.getIssuer());
				return null;
			}
		}
	}

	public boolean canCreateCard(CallbackHandler authHandler, String id, Properties props) {
		if (authHandler == null) {
			log.info("Parameter \"authHandler\" is null.");
			return false;
		}
		if (id == null) {
			log.info("Parameter \"id\" is null.");
			return false;
		}
		try {
			IUserCredentials userCreds = authenticate(authHandler);
			PCardEntity crd = getManagedBean().getPCard(id, userCreds);
			if (crd != null) {
				log.info("Can not create card. Card with ID = " + id + " already exists.");
				return false;
			} else
				return true;
		} catch (CardException e) {
			log.error(e, e);
			return false;
		}
	}

	public ICard createCard(CallbackHandler authHandler, String id, Properties props) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (id == null)
			throw new CardException("Parameter \"id\" is null");
		URI cardID = null;
		try {
			cardID = new URI(id);
		} catch (URISyntaxException e) {
			log.error(e, e);
			throw new CardException(e);
		}
		IUserCredentials userCreds = authenticate(authHandler);
		return new PCard(this, userCreds, cardID, null, null, null, null);
		 
	}

	public String getPictureMIMEType(byte[] picure) {
		return "image/jpeg";
	}

	public String getDefaultPictureMIMEType() {
		return defaultImageFileType;
	}

	public byte[] getDefaultImage() {
		if (defaultImage == null) {
			return readDefaultImage();
		}
		return defaultImage;
	}

	private synchronized byte[] readDefaultImage() {
		if (defaultImage != null) {
			return defaultImage;
		}
		if (defaultImageFileName != null) {
			File file = new File(defaultImageFileName);
			if (file.exists() && file.isFile()) {
				int size;
				long fileLength = file.length();
				if (fileLength <= Integer.MAX_VALUE)
					size = (int) fileLength;
				else {
					log.error("IdASBasedPersonalCardProvider.getDefaultImage - too long default image file size.");
					return null;
				}
				int offset = 0;
				int len = size;
				FileInputStream fis = null;
				try {
					fis = new FileInputStream(file);
					int count = 0;
					byte[] image = new byte[size];
					do {
						count = fis.read(image, offset, len);
						if (count != len) {
							offset += count;
							len -= count;
						}
					} while (count != -1);
					defaultImage = image;
				} catch (IOException e) {
					log.error(e, e);
				} finally {
					try {
						if (fis != null)
							fis.close();
					} catch (IOException e) {
						log.error(e, e);
					}
				}
			} else {
				log.error("IdASBasedPersonalCardProvider.getDefaultImage - default image file " + defaultImageFileName
						+ " does not exist.");
			}
		} else {
			log.error("IdASBasedPersonalCardProvider.getDefaultImage - can not get the name of default image.");
		}
		return defaultImage;
	}

	public ICard createCard(CallbackHandler authHandler, ICardTemplate template) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		String cardName = null;
		ITemplateValue nameTV = template.getTemplateValueByID(PersonalCardTemplateHelper.CARD_NAME);
		if (nameTV != null)
			cardName = nameTV.getValue();
		byte[] picture = null;
		String pictureType = null;
		ITemplateValue pictureTV = template.getTemplateValueByID(PersonalCardTemplateHelper.CARD_PICTURE);
		if (pictureTV != null) {
			String picStr = pictureTV.getValue();
			if (picStr != null) {
				picture = CardCryptography.decodeBase64(picStr);
				pictureType = getPictureMIMEType(picture);
			}
		}
		HashMap dirtyClaims = PersonalCardTemplateHelper.getDirtyClaimsFromCardTemplate(template);
		IUserCredentials userCreds = authenticate(authHandler);
		getManagedBean(); // calls to read config
		if (picture == null) {
			picture = getDefaultImage();
			pictureType = getDefaultPictureMIMEType();
		}
		PCard card = new PCard(this, userCreds, URI.create(UUIDGenerator.getUUID()), cardName, dirtyClaims, picture, pictureType);
		return card;
	}

	public ICardTemplate[] getCardCreationTemplates(CallbackHandler authHandler) {
		return new ICardTemplate[] { PersonalCardTemplateHelper.getCardTemplate(this) };
	}

	public ISettingDescriptor getComponentDescriptor() {
		// TODO Auto-generated method stub
		return null;
	}

	public IConfiguration getConfiguration() {
		if (config == null) {
			initConfiguration();
		}
		return config;
	}

	private synchronized void initConfiguration() {
		if (config == null) {
			try {
				config = ICardRegistry.getInstance().getConfiguration(this);
			} catch (RegistryConfigurationException e) {
				log.error(e, e);
			}
		}
	}

	
	private IUserCredentials authenticate(CallbackHandler authHandler) throws CardException {
		NameCallback nc = new NameCallback("User id: ");
		PasswordCallback pc = new PasswordCallback("Password: ", false);
		Callback[] callbacks = new Callback[] { nc, pc };
		try {
			authHandler.handle(callbacks);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		String userName = nc.getName();
		if (userName == null || userName.trim().length() == 0)
			throw new CardException("Empty user name.");
		String pass = (pc != null && pc.getPassword() != null) ? new String(pc.getPassword()) : "";
		return new UsernamePasswordCredentials(userName, pass);
	}

	
}
