/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Patrick R Wardrop (IBM Corporation)
 *    Bruce Rich (IBM Corporation)
 *******************************************************************************/ 
package org.eclipse.higgins.cardstore.schemas._2005._05.identity.impl;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;

import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.security.auth.callback.PasswordCallback;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.higgins.cardstore.Constants;
import org.eclipse.higgins.cardstore.LocalElementNames;
import org.eclipse.higgins.cardstore.exceptions.ExpectedObjectNotPresent;
import org.eclipse.higgins.cardstore.exceptions.StoreDecryptionException;
import org.eclipse.higgins.cardstore.exceptions.StoreEncryptionException;
import org.eclipse.higgins.cardstore.exceptions.UnsupportedObjectModel;
import org.eclipse.higgins.cardstore.logging.Log;
import org.eclipse.higgins.cardstore.logging.LogHelper;
import org.eclipse.higgins.cardstore.resources.CardStoreMessages;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IEncryptedStore;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IRoamingStore;
import org.eclipse.higgins.cardstore.utils.Base64Utility;
import org.eclipse.higgins.cardstore.utils.HexDumper;
import org.eclipse.higgins.cardstore.utils.MessageHelper;
import org.eclipse.higgins.cardstore.utils.XmlUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

public class EncryptedStore implements IEncryptedStore {
	public static String copyright = Constants.FULL_COPYRIGHT;
	
	private static Log _log = LogHelper.getLogger(EncryptedStore.class);
	
	private static final String STORE_KEY_TYPE = "AES";
	
	private static final String STORE_CIPHER = "AES/CBC/PKCS5Padding";
	
	private static final String UTF8 = "UTF-8";
	
	private static final String UTF16LE = "UTF-16LE";
	
    private static byte[] ENC_KEY_ENTROPY =  { 
    	(byte)0xd9, (byte)0x59, (byte)0x7b, (byte)0x26, 
    	(byte)0x1e, (byte)0xd8, (byte)0xb3, (byte)0x44, 
    	(byte)0x93, (byte)0x23, (byte)0xb3, (byte)0x96, 
    	(byte)0x85, (byte)0xde, (byte)0x95, (byte)0xfc };
    
    private static byte[] SIG_KEY_ENTROPY =         {
    	(byte)0xc4, (byte)0x01, (byte)0x7b, (byte)0xf1, 
    	(byte)0x6b, (byte)0xad, (byte)0x2f, (byte)0x42, 
    	(byte)0xaf, (byte)0xf4, (byte)0x97, (byte)0x7d, 
    	(byte)0x4,  (byte)0x68, (byte)0x3,  (byte)0xdb};    
    
    private static byte[] BYTE_ORDER_MARK = {
    	(byte)0xef, (byte)0xbb, (byte) 0xbf};
    	
	private byte[] _byteOrderMark = null;
	
	private byte[] _crdsBytes = null;
	
	private byte[] _storeSalt = null;
	
	private Document _rootDocument = null;
	
	private Element _encryptedStoreElement = null;
	
	private IRoamingStore _roamingStore = null;
	
	private static SecureRandom _randomGenerator = null;
	
	public Object getParent()
	{
		return getRootDocument();
	}
	
	public void setParent(Object parent)
	{
		// no-op necessary
	}
	
	/**
	 * Construct an empty EncryptedStore
	 */
	public EncryptedStore()
	{
		this._byteOrderMark = BYTE_ORDER_MARK;
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setNamespaceAware(true);
		try {
			DocumentBuilder db = dbf.newDocumentBuilder();
			this._rootDocument = db.newDocument();
			Element roamingStoreElement = this._rootDocument.createElementNS(Constants.NSURI_IDENTITY_2005_05,	LocalElementNames.IDENTITY_ROAMING_STORE);
			_rootDocument.appendChild(roamingStoreElement);
			_roamingStore = new RoamingStore();
			try {
				_roamingStore.fromXml(roamingStoreElement);
				((RoamingStore)_roamingStore).setParent(this);
			} catch (Exception e) {
				_log.error(null, e);
			}			
 
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}
	
    public EncryptedStore(byte[] crdsBytes) throws UnsupportedObjectModel, ExpectedObjectNotPresent
    {
    	_crdsBytes = crdsBytes;
    	
    	_byteOrderMark = retrieveByteOrderMark(_crdsBytes);
    	
    	Element encryptedStoreElement = retrieveStoreXmlMessage(_crdsBytes);
    	
    	this.fromXml(encryptedStoreElement);
    }	
	
	private Element retrieveDecryptXmlStore(char[] password) throws StoreDecryptionException {
		String method = "retrieveDecryptXmlStore";
		_log.enterMethod(method);
		
		Element result = null; 

		if (_encryptedStoreElement != null)
		{
			Element encryptedData = XmlUtils.retrieveFirstChildMatchOfType(
										_encryptedStoreElement, 
										LocalElementNames.XENC_ENCRYPTED_DATA, 
										Constants.NSURI_XENC_2001_04);
			if (encryptedData != null)
			{
				Element cipherData = XmlUtils.retrieveFirstChildMatchOfType(
						encryptedData, 
						LocalElementNames.XENC_CIPHER_DATA, 
						Constants.NSURI_XENC_2001_04);
				
				if (cipherData != null)
				{
					Element cipherValue = XmlUtils.retrieveFirstChildMatchOfType(
							cipherData, 
							LocalElementNames.XENC_CIPHER_VALUE, 
							Constants.NSURI_XENC_2001_04);
					
					if (cipherValue != null)
					{
						String base64CipherValue = XmlUtils.retrieveAllTextFromChildTextNodes(cipherValue);
						
						if (base64CipherValue != null && base64CipherValue.length() > 0)
						{							
							_log.trace(method, "base64 encoded cipher value retrieved is: " + base64CipherValue);
							try {
								byte[] roamingStoreBytes = decrypt(base64CipherValue, password, getStoreSalt());
																
								if (roamingStoreBytes != null)
								{
									DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
									dbf.setNamespaceAware(true);
									DocumentBuilder db = dbf.newDocumentBuilder();
									Document doc = db.parse(new ByteArrayInputStream(roamingStoreBytes));
									
									if (doc != null)
									{
										result = doc.getDocumentElement();
									}
								}
								
							} catch (Exception e) {
								throw new StoreDecryptionException(MessageHelper.getMessage(CardStoreMessages.encrypted_store_could_not_be_decrypted, null), e); 
						}
						} else {
							throw new StoreDecryptionException(MessageHelper.getMessage(CardStoreMessages.encrypted_store_could_not_be_decrypted_ciphervalue_no_data, null));
						}
					} else {
						throw new StoreDecryptionException(MessageHelper.getMessage(CardStoreMessages.encrypted_store_could_not_be_decrypted_ciphervalue_not_present, null));						
					}					
				} else {
					throw new StoreDecryptionException(MessageHelper.getMessage(CardStoreMessages.encrypted_store_could_not_be_decrypted_cipherdata_not_present, null));
				}
			} else {
				throw new StoreDecryptionException(MessageHelper.getMessage(CardStoreMessages.encrypted_store_could_not_be_decrypted_encrypteddata_not_present, null));
			}
			
		}
		
		if (result == null)
		{
			throw new StoreDecryptionException(MessageHelper.getMessage(CardStoreMessages.encrypted_store_could_not_be_decrypted, null));
		}
		
		_log.exitMethod(method, result);
		return result;
	}

	/**
	 * This method will walk the document and retrieve the base64 encoded salt and decode it into
	 * a byte array.
	 * 
	 * @param encryptedStore
	 * @return returns a byte array of the salt that is base64 decoded; null if no salt is present.
	 */
	private byte[] retrieveStoreSalt(Element encryptedStore) {
		String method = "retrieveStoreSalt";
		_log.enterMethod(method, new Object[]{encryptedStore});
		
		byte[] result = null;
		
		if (encryptedStore != null)
		{
			Element storeSaltElement = XmlUtils.retrieveFirstChildMatchOfType(encryptedStore, LocalElementNames.IDENTITY_STORE_SALT, Constants.NSURI_IDENTITY_2005_05);
			
			if (storeSaltElement != null)
			{
				String s = XmlUtils.retrieveAllTextFromChildTextNodes(storeSaltElement);
				
				if (s != null && s.length() > 0)
				{
					result = Base64Utility.decode(s);
				}
			}
		}
				
		_log.exitMethod(method, result);
		return result;
	}

	private Element retrieveStoreXmlMessage(byte[] bytes) {
		String method = "retrieveStoreXmlMessage";
		_log.enterMethod(method, new Object[]{bytes});
				
		Element result = null;
		
		if (bytes != null)
		{
			Document doc = null;
			
			try {
				DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
				dbf.setNamespaceAware(true);
				DocumentBuilder db = dbf.newDocumentBuilder();
				doc = db.parse(new ByteArrayInputStream(bytes));
			} catch (Exception e) {
				_log.error(null, e);
			}
			
			result = doc.getDocumentElement();			
		}
		
		_log.exitMethod(method, result);
		return result;
	}

	/**
	 * This method will extract the byte order mark from the first three
	 * bytes of the CRDS file.
	 * 
	 * @param crdsBytes
	 * @return three bytes; if byte order marker not present null is returned.
	 */
	private static byte[] retrieveByteOrderMark(byte[] crdsBytes) {
		String method = "retrieveByteOrderMark";
		_log.enterMethod(method, new Object[]{crdsBytes});
		
		byte[] result = null;
		
		//
		// We are going to get the first three bytes of the CRDS file.
		// Also, we are going to check if the next two bytes/characters are the
		// characters '<' and '?'.
		//
		
		if (crdsBytes != null && crdsBytes.length > 5)
		{
			result = new byte[3];
			int i = 0;
			for(i = 0; i < 3; i++)
			{
				result[i] = crdsBytes[i];
			}
						
			//
			// checking that the next two characters are <?
			// if not, we assume the byte order mark isn't
			// present.
			//
			if (crdsBytes[i] != 60 || crdsBytes[i+1] != 63)
			{
				_log.trace(method, "the first three bytes weren't the byte order mark.");
				_log.warn(MessageHelper.getMessage(CardStoreMessages.first_three_bytes_were_not_byte_order_mark, null));
				result = null;
			}
		}
		
		_log.trace(method, "the byte order mark return is: " + HexDumper.byteArrayToHexString(result));
						
		_log.exitMethod(method, result);
		return result;
	}
	

	public byte[] getByteOrderMark() {
		if (_byteOrderMark == null)
		{
			// lazy retrieval of byte order mark. This laziness
			// should be changed if this byte order mark is
			// used frequently.
			_byteOrderMark = retrieveByteOrderMark(_crdsBytes); 
		}
		
		return _byteOrderMark;
	}

	public byte[] getStoreSalt() {
		return _storeSalt;
	}	
	
	private byte[] decrypt(String ciphertext, char[] password, byte[] salt) throws StoreDecryptionException	
	{
		byte[] result = null;
		
		try {
			byte[] ct = Base64Utility.decode(ciphertext); // raw cipher text value 
			byte[] iv = new byte[16]; // initialization vector
			byte[] ic = new byte[32]; // integrity code (signature)
			byte[] da = new byte[ ct.length - iv.length - ic.length]; // the encrypted data
			
	        System.arraycopy(ct, 0, iv, 0, 16);
	        System.arraycopy(ct, 16, ic, 0, 32);
	        System.arraycopy(ct, 48, da, 0, da.length);
	        
	        byte keysBytes[][] = makeKeys(password, salt);
	        
	        Key encKey = new SecretKeySpec(keysBytes[0], STORE_KEY_TYPE);
	        
	        Cipher cipher = Cipher.getInstance(STORE_CIPHER);
			IvParameterSpec ips = new IvParameterSpec(iv);
			cipher.init(Cipher.DECRYPT_MODE, encKey, ips);
			
			byte[] out = cipher.doFinal(da);
					
			// remove the byte order mark (if present)
			if (out[0]==BYTE_ORDER_MARK[0] 
			      && out[1]==BYTE_ORDER_MARK[1] 
			      && out[2]==BYTE_ORDER_MARK[2]) {
				result = new byte[out.length-3];
				System.arraycopy(out, 3, result, 0, out.length-3);
			} else {
				result = new byte[out.length];
				System.arraycopy(out, 0, result, 0, out.length);
			}
		
			//
			// Do integrity check
			//
			byte[] last_block = new byte[16];			
			System.arraycopy(out, out.length - 16, last_block, 0, 16);
			
			if(!isSignatureValid(iv, last_block, keysBytes[1], ic))
			{			
				throw new StoreDecryptionException(
						MessageHelper.getMessage(CardStoreMessages.encrypted_store_integrity_check_failed, null)); 
			} 
		} catch (Exception e) {
			throw new StoreDecryptionException(e);
		}
		
		return result;
	}
	
	private byte[] encrypt(byte[] cleartext, char[] password, byte[] salt) throws StoreEncryptionException
	{
		byte[] result = null;
		
		try {
			byte[] in = new byte[BYTE_ORDER_MARK.length + cleartext.length];
			
			// put the BOM before the cleartext
			System.arraycopy(BYTE_ORDER_MARK, 0, in, 0, BYTE_ORDER_MARK.length);
			System.arraycopy(cleartext, 0, in, BYTE_ORDER_MARK.length, cleartext.length);
	
			// generate a new initialization vector
			byte[] iv = new byte[16];
			generateRandomBytes(iv);
			
	        byte keysBytes[][] = makeKeys(password, salt);
	        
	        Key encKey = new SecretKeySpec(keysBytes[0], STORE_KEY_TYPE);
	        
	        Cipher cipher = Cipher.getInstance(STORE_CIPHER);
			IvParameterSpec ips = new IvParameterSpec(iv);
			cipher.init(Cipher.ENCRYPT_MODE, encKey, ips);
			
			byte[] out = cipher.doFinal(in);

			byte[] last_block = new byte[16];			
			System.arraycopy(in, in.length - last_block.length, last_block, 0, last_block.length);
			
			// get integrity code
			byte[] ic = calculateHashedIntegrityCode(iv, last_block, keysBytes[1]);
            result = new byte[iv.length + ic.length + out.length];
			
			System.arraycopy(iv, 0, result, 0, iv.length);
			System.arraycopy(ic, 0, result, iv.length, ic.length);
			System.arraycopy(out, 0, result, iv.length + ic.length, out.length);
			
		} catch (Exception e) {
			throw new StoreEncryptionException(e);
		}
		
		return result;
	}
	
	/**
	 * This method will generate a random 16 byte salt.
	 * 
	 * @param the array to store; the array should have been initialized already (i.e., shouldn't be null)
	 * @return random 16 byte salt
	 * @throws NoSuchAlgorithmException
	 */
	private static void generateRandomBytes(byte[] b) throws NoSuchAlgorithmException
	{
		if (b == null)
		{
			return;
		}
		
		if (_randomGenerator == null)
		{
			_randomGenerator = SecureRandom.getInstance("SHA1PRNG");
		}
		
		_randomGenerator.nextBytes(b);				
	}

	/**
	 * Returns true if the calculated integrity code matches the one given with the
	 * encrypted store.
	 * 
	 * @param iv initialization vector
	 * @param last_block 16 bytes to use to calculate integrity code.
	 * @param ik integrity key (SHA256(iv + iv + PKCS5-derived-key))
	 * @param ic the given integrity code from store - used to check calculated integrity code against.
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	private boolean isSignatureValid(byte[] iv, byte[] last_block, byte[] ik, byte[] ic) throws NoSuchAlgorithmException {		
        byte[] icCalc = calculateHashedIntegrityCode(iv, last_block, ik);
        
		return Arrays.equals(ic, icCalc);
	}
	
	/**
	 * Calculate the integrity code
	 * @param iv initialization vector
	 * @param last_block 16 bytes to use to calculate integrity code.
	 * @param ik integrity key (SHA256(iv + iv + PKCS5-derived-key))
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	private byte[] calculateHashedIntegrityCode(byte[] iv, byte[] last_block, byte[] ik) throws NoSuchAlgorithmException
	{
		MessageDigest sha256 = MessageDigest.getInstance("SHA-256");
		sha256.reset();
		byte[] dataBytes = new byte[iv.length + ik.length + last_block.length];
        System.arraycopy(iv, 0, dataBytes, 0, iv.length);
        System.arraycopy(ik, 0, dataBytes, iv.length, ik.length);
        System.arraycopy(last_block, 0, dataBytes, iv.length + ik.length, last_block.length);
        sha256.update(dataBytes);
        return sha256.digest();
	}
	
	/**
	 * This method will convert the given char[] into a byte[] WITHOUT
	 * using java.lang.String. java.lang.String is not used for
	 * security reasons. 
	 * 
	 * @param chars
	 * @return
	 */
	private byte[] convertCharArrayToByteArray(char[] chars)
	{
		if (chars == null)
		{
			return null;
		}
		
		byte[] newbytes8 = null;
		Charset utf8char = Charset.forName(UTF16LE);
		CharsetEncoder utf8ncd = utf8char.newEncoder();
		CharBuffer buf8 = CharBuffer.allocate(chars.length);
		buf8.put(chars).position(0);
		//try {
		ByteBuffer bb1;
		try {
			bb1 = utf8ncd.encode(buf8);
			newbytes8 = new byte[bb1.limit()-bb1.position()];
			bb1.get(newbytes8);
		} catch (CharacterCodingException cce) {
			throw new RuntimeException(cce.getMessage(), cce);
		}
		return newbytes8;
	}

	/**
	 * This method will return an array of byte arrays that contains two sets of key data.  The first key should be
	 * used for encryption/decryption and the second key should be used for signture creation and validation.
	 * 
	 * @param password
	 * @param salt
	 * @return Key array containing two keys.
	 * @throws UnsupportedEncodingException
	 * @throws NoSuchAlgorithmException
	 */
	public byte[][] makeKeys(char[] password, byte[] salt) throws UnsupportedEncodingException, NoSuchAlgorithmException {				
		MessageDigest sha256 = MessageDigest.getInstance("SHA-256");
		
		sha256.update(convertCharArrayToByteArray(password));
		sha256.update(salt);
		byte[] digestBytes = sha256.digest();
        for (int i = 1; i < 1000; i++) {
        	sha256.update(digestBytes);
            digestBytes = sha256.digest();
        }
        
        sha256.reset();

        // mix with the entropy for enc key        
        byte[] encKeyBytes = new byte[ENC_KEY_ENTROPY.length + digestBytes.length];
        System.arraycopy(ENC_KEY_ENTROPY, 0, encKeyBytes, 0, ENC_KEY_ENTROPY.length);
        System.arraycopy(digestBytes, 0, encKeyBytes, ENC_KEY_ENTROPY.length ,digestBytes.length);
        sha256.update(encKeyBytes);
        
        byte[] ek = sha256.digest();

        sha256.reset();
        
        // mix with the entropy for sig key
        byte[] sigKeyBytes = new byte[SIG_KEY_ENTROPY.length + digestBytes.length];
        System.arraycopy(SIG_KEY_ENTROPY, 0, sigKeyBytes, 0, SIG_KEY_ENTROPY.length);
        System.arraycopy(digestBytes, 0, sigKeyBytes, SIG_KEY_ENTROPY.length ,digestBytes.length);
        sha256.update(sigKeyBytes);
        
        byte[] ik = sha256.digest();
		
		byte result[][] = new byte[2][];
		result[0] = ek;
		result[1] = ik;
		
        return result;
	}

	public synchronized IRoamingStore getRoamingStore(PasswordCallback passwdCallback) throws StoreDecryptionException {
		Element roamingStoreElement = null;
		
		if (_roamingStore == null && passwdCallback != null 
				&& passwdCallback.getPassword() != null)
		{
			roamingStoreElement = retrieveDecryptXmlStore(passwdCallback.getPassword());			
		}	
		
		if (_roamingStore == null && roamingStoreElement != null);
		{
			_roamingStore = new RoamingStore();
			try {
				_roamingStore.fromXml(roamingStoreElement);
				((RoamingStore)_roamingStore).setParent(this);
			} catch (Exception e) {
				_log.error(null, e);
			}			
		}
				
		return _roamingStore;
	}

	public synchronized void fromXml(Object obj) throws UnsupportedObjectModel, ExpectedObjectNotPresent {		
		_encryptedStoreElement = XmlUtils.retrieveElementFromObject(obj);
		if (_encryptedStoreElement != null)
		{			
			if (!XmlUtils.isElementType(_encryptedStoreElement, LocalElementNames.IDENTITY_ENCRYPTED_STORE, Constants.NSURI_IDENTITY_2005_05))
			{
				_encryptedStoreElement = null;
				throw new ExpectedObjectNotPresent("The expected object " + Constants.NSURI_IDENTITY_2005_05 + ":" + LocalElementNames.IDENTITY_ENCRYPTED_STORE + " was not present.");
			} else {
				_rootDocument = _encryptedStoreElement.getOwnerDocument();
				
				_storeSalt = retrieveStoreSalt(_encryptedStoreElement);								
			}			
		}
	}

	public Object toXml() {
		//
		// The toXml(javax.security.auth.callback.PasswordCallback) method should always 
		// be called.
		//
		throw new UnsupportedOperationException(
				MessageHelper.getMessage(CardStoreMessages.encrypted_store_to_xml_w_password_should_be_called, null));
	}
	
	/**
	 * Returns the bytes that should be written our to a CRDS file.  The first three bytes with be a byte order mark followed by
	 * the EncryptedStore XML.
	 * 
	 * @param passwdCallback The password is required to encrypt the store
	 * @return the byte array to be written out to a CRDS file.
	 */
	public synchronized byte[] toXml(PasswordCallback passwdCallback) throws StoreEncryptionException
	{
		try {
			byte[] salt = new byte[16];
							
			generateRandomBytes(salt);
			
			String roamingStoreString = XmlUtils.getString((Node)_roamingStore.toXml());
			
			if (roamingStoreString != null && roamingStoreString.length() > 0)
			{
				byte[] encryptedRoamingStoreBytes = encrypt(roamingStoreString.getBytes("UTF-8"), passwdCallback.getPassword(), salt);
				_storeSalt = salt;
				_encryptedStoreElement = createEncryptedStoreElement(encryptedRoamingStoreBytes, salt);	
				
				if (_encryptedStoreElement != null)
				{
					String encryptedStoreString = XmlUtils.getString(_encryptedStoreElement);
										
					byte[] encryptedStoreElementBytes = encryptedStoreString.getBytes("UTF-8");
					_crdsBytes = new byte[BYTE_ORDER_MARK.length + encryptedStoreElementBytes.length];
					System.arraycopy(BYTE_ORDER_MARK, 0, _crdsBytes, 0, BYTE_ORDER_MARK.length);
					System.arraycopy(encryptedStoreElementBytes, 0, _crdsBytes, BYTE_ORDER_MARK.length, encryptedStoreElementBytes.length);
				}
			}
		} catch (Exception e) {
			throw new StoreEncryptionException(e);
		}		
		
		return _crdsBytes;
	}

	private Element createEncryptedStoreElement(byte[] encryptedRoamingStoreBytes, byte[] salt) throws ParserConfigurationException {
		_rootDocument = XmlUtils.newDocument();
		
		Element encryptedStoreElement = _rootDocument.createElementNS(Constants.NSURI_IDENTITY_2005_05, LocalElementNames.IDENTITY_ENCRYPTED_STORE);
		encryptedStoreElement.setAttribute("xmlns", Constants.NSURI_IDENTITY_2005_05);
		_rootDocument.appendChild(encryptedStoreElement);
		
		Element storeSaltElement = _rootDocument.createElementNS(Constants.NSURI_IDENTITY_2005_05, LocalElementNames.IDENTITY_STORE_SALT);
		encryptedStoreElement.appendChild(storeSaltElement);
		Text storeSaltText = _rootDocument.createTextNode(Base64Utility.encode(salt));
		storeSaltElement.appendChild(storeSaltText);
		
		Element encryptedDataElement = _rootDocument.createElementNS(Constants.NSURI_XENC_2001_04, LocalElementNames.XENC_ENCRYPTED_DATA);
		encryptedDataElement.setAttribute("xmlns", Constants.NSURI_XENC_2001_04);
		encryptedStoreElement.appendChild(encryptedDataElement);
		
		Element cipherDataElement = _rootDocument.createElementNS(Constants.NSURI_XENC_2001_04, LocalElementNames.XENC_CIPHER_DATA);
		encryptedDataElement.appendChild(cipherDataElement);
		
		Element cipherValueElement = _rootDocument.createElementNS(Constants.NSURI_XENC_2001_04, LocalElementNames.XENC_CIPHER_VALUE);
		cipherDataElement.appendChild(cipherValueElement);
		
		Text cipherValueText = _rootDocument.createTextNode(Base64Utility.encode(encryptedRoamingStoreBytes));
		cipherValueElement.appendChild(cipherValueText);
		
		return encryptedStoreElement;
	}
	
	public Document getRootDocument()
	{
		if (_rootDocument == null)
		{
			try {
				_rootDocument = XmlUtils.newDocument();
			} catch (ParserConfigurationException e) {
				_log.error(null, e);
			}
		}
		
		return _rootDocument;
	}

	public static void main(String[] args) throws Exception
	{
		boolean go = true;
		int x = 0;
		while(go)
		{
			{		
				x++;
				{
					EncryptedStore es = null;
					{ //READ
						PasswordCallback pc = new PasswordCallback("What is your password: ", true);				
						pc.setPassword((new String("passw0rd")).toCharArray());		
						File crdsFile = new File("c:/temp/mycards.crds");
						byte[] crdsBytes = new byte[(int)crdsFile.length()];		
						FileInputStream fis = new FileInputStream(crdsFile);
						fis.read(crdsBytes);		
						
						long timeStart = System.currentTimeMillis();
						es = new EncryptedStore(crdsBytes);
						IRoamingStore rs = es.getRoamingStore(pc);
						long timeStop = System.currentTimeMillis();
						
						System.out.println("total time (decrypt): " + (timeStop - timeStart));						
					}
					{ //WRITE
						PasswordCallback pc = new PasswordCallback("What is your password: ", true);						
						pc = new PasswordCallback("What is your password: ", true);				
						pc.setPassword((new String("mercury1")).toCharArray());		
						
						long timeStart = System.currentTimeMillis();
						byte[] esBytes = es.toXml(pc);
						long timeStop = System.currentTimeMillis();
						
						System.out.println("total time (encrypt): " + (timeStop - timeStart));	
						
						FileOutputStream fos = new FileOutputStream(new File("c:/temp/new_mycards.crds"));
						fos.write(esBytes);
						fos.flush();
						fos.close();
					}
					{ //READ
						PasswordCallback pc = new PasswordCallback("What is your password: ", true);				
						pc.setPassword((new String("mercury1")).toCharArray());		
						File crdsFile = new File("c:/temp/new_mycards.crds");
						byte[] crdsBytes = new byte[(int)crdsFile.length()];		
						FileInputStream fis = new FileInputStream(crdsFile);
						fis.read(crdsBytes);		
						
						long timeStart = System.currentTimeMillis();
						es = new EncryptedStore(crdsBytes);
						IRoamingStore rs = es.getRoamingStore(pc);
						long timeStop = System.currentTimeMillis();
						
						System.out.println("total time (decrypt2): " + (timeStop - timeStart));						
					}
					
				}
			}
			
			{		
				EncryptedStore es = null;
				{ //READ
					PasswordCallback pc = new PasswordCallback("What is your password: ", true);				
					pc.setPassword((new String("ibmibmibm")).toCharArray());		
					File crdsFile = new File("c:/temp/tonycards.crds");
					byte[] crdsBytes = new byte[(int)crdsFile.length()];		
					FileInputStream fis = new FileInputStream(crdsFile);
					fis.read(crdsBytes);		
					
					long timeStart = System.currentTimeMillis();
					es = new EncryptedStore(crdsBytes);
					
					IRoamingStore rs = es.getRoamingStore(pc);
					long timeStop = System.currentTimeMillis();
					
					System.out.println("total time (decrypt): " + (timeStop - timeStart));					
				}
				
				{ //WRITE
					PasswordCallback pc = new PasswordCallback("What is your password: ", true);						
					pc = new PasswordCallback("What is your password: ", true);				
					pc.setPassword((new String("mercury2")).toCharArray());		
					
					long timeStart = System.currentTimeMillis();
					byte[] esBytes = es.toXml(pc);
					long timeStop = System.currentTimeMillis();
					
					System.out.println("total time (encrypt): " + (timeStop - timeStart));	
					
					FileOutputStream fos = new FileOutputStream(new File("c:/temp/new_tonycards.crds"));
					fos.write(esBytes);
					fos.flush();
					fos.close();
				}

				{ //READ
					PasswordCallback pc = new PasswordCallback("What is your password: ", true);				
					pc.setPassword((new String("mercury2")).toCharArray());		
					File crdsFile = new File("c:/temp/new_tonycards.crds");
					byte[] crdsBytes = new byte[(int)crdsFile.length()];		
					FileInputStream fis = new FileInputStream(crdsFile);
					fis.read(crdsBytes);		
					
					long timeStart = System.currentTimeMillis();
					es = new EncryptedStore(crdsBytes);
					
					IRoamingStore rs = es.getRoamingStore(pc);
					long timeStop = System.currentTimeMillis();
					
					System.out.println("total time (decrypt2): " + (timeStop - timeStart));					
				}

				
			}

			if (x > 1)
			{
				go = false;
			}
		}

		
		
		
		System.out.println("Done.");
				
	}

	public void setRoamingStore(IRoamingStore roamingStore) {
		_roamingStore = roamingStore;		
	}
	
}
