/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Patrick R Wardrop (IBM Corporation)
 *******************************************************************************/ 
package org.eclipse.higgins.cardstore.utils;

import java.io.StringWriter;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.eclipse.higgins.cardstore.Constants;
import org.eclipse.higgins.cardstore.exceptions.ExpectedObjectNotPresent;
import org.eclipse.higgins.cardstore.exceptions.UnsupportedObjectModel;
import org.eclipse.higgins.cardstore.schemas.XMLSerializer;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

public class XmlUtils {
	public static String copyright = Constants.FULL_COPYRIGHT;
	
	public static Element retrieveElementFromObject(Object obj) throws UnsupportedObjectModel
	{
		Element result = null;
		
		if (obj != null && obj instanceof Node)
		{
			Node n = (Node)obj;
			if (n.getNodeType() == Node.ELEMENT_NODE)
			{
				result = (Element)n;
			} else if (n.getNodeType() == Node.DOCUMENT_NODE)
			{
				result = ((Document)n).getDocumentElement();
			}			
		}
		
		if (result == null)
		{
			throw new UnsupportedObjectModel("This implementation only supports " + Node.class.getPackage().getName() + " objects.");				
		}		
				
		return result;
	}
	
	public static boolean isElementType(Element el, String nodeName, String namespaceUri)
	{
		boolean result = false;
		
		result = ((el != null)
				&& nodeName != null
				&& namespaceUri != null
				&& el.getNamespaceURI() != null
				&& el.getNamespaceURI().equals(namespaceUri)
				&& el.getNodeName() != null
				&& el.getNodeName().equals(nodeName));
		
		return result;
	}
	
	public static int numberOfChildElementsOfType(Element parent, String nodeName, String namespaceUri)
	{
		
		int result = 0;
		
		if (parent == null || nodeName == null || namespaceUri == null)
		{
			return result;
		}
		
		NodeList nl = parent.getChildNodes();
		for(int i = 0; i < nl.getLength(); i++)
		{
			Node n = nl.item(i);
			
			if (n.getNodeType() == Node.ELEMENT_NODE)
			{
				if (isElementType((Element)n, nodeName, namespaceUri))
				{
					result++;
				}
			}
		}
		
		return result;
	}
	
	public static Element retrieveFirstChildMatchOfType(Element parent, String nodeName, String namespaceUri)
	{
		Element result = null;
		
		if (parent == null || nodeName == null || namespaceUri == null)
		{
			return result;
		}
		
		NodeList nl = parent.getChildNodes();
		for(int i = 0; i < nl.getLength(); i++)
		{
			Node n = nl.item(i);
			
			if (n.getNodeType() == Node.ELEMENT_NODE)
			{
				if (isElementType((Element)n, nodeName, namespaceUri))
				{
					result = (Element)n;
					break;
				}
			}
		}
		
		return result;
	}
	
	public static String retrieveAllTextFromChildTextNodes(Element parent)
	{
		StringBuffer result = new StringBuffer();
		
		if (parent != null)
		{
			NodeList nl = parent.getChildNodes();
			int len = nl.getLength();
			
			for(int i = 0; i < len; i++)
			{
				Node n = nl.item(i);
				if (n != null && n.getNodeType() == Node.TEXT_NODE)
				{
					Text text = (Text)n;
					result.append( text.getData());
				}
			}
		}
		
		return result.toString();
	}	
	
	public static void buildObjectFromXml(Object parentObj, XMLSerializer obj, Element parent, String nodeName, String namespaceUri) throws UnsupportedObjectModel, ExpectedObjectNotPresent
	{		
		Element childEl = XmlUtils.retrieveFirstChildMatchOfType(
				parent, 
				nodeName, 
				namespaceUri);
		
		if (childEl != null)
		{			
			obj.fromXml(childEl);
			obj.setParent(parentObj);
		} else {
			obj = null;
		}
		
	}
	
	public static Document getDocumentRootFromParent(XMLSerializer child)
	{
			Object parent = child.getParent();
			
			if (parent == null)
			{
				return null;
			}
			
			if (parent instanceof Document)
			{
				return (Document)parent;
			} else if (parent instanceof XMLSerializer)
			{
				return getDocumentRootFromParent((XMLSerializer)parent);
			}			
		
		return null;
	}
	
	/**
	 * This method will return a java.lang.String representation of the XML given.  
	 * This method should not be used for any code that does signature functionality.
	 * 
	 * @param node
	 * @return
	 * @throws TransformerException 
	 */
	public static String getString(Node node) throws TransformerException
	{
		TransformerFactory tf = TransformerFactory.newInstance();
		Transformer t = tf.newTransformer();
		
		StringWriter sw = new StringWriter();

        t.transform(new DOMSource(node), new StreamResult(sw));		
		
        return sw.toString();
	}
	
	public static Document newDocument() throws ParserConfigurationException
	{
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setNamespaceAware(true);
		DocumentBuilder db;		
		db = dbf.newDocumentBuilder();
		return db.newDocument();		
	}

}
