/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Anthony Bussani - Initial API and implementation
 *     Li Tie		   - Add PIN code support
 *******************************************************************************/

package org.eclipse.higgins.icard.provider.securestorage;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import org.eclipse.higgins.cardstore.utils.Base64Utility;
import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.AuthenticationRequiredException;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IInformationCardExtension;
import org.eclipse.higgins.icard.InvalidClaimException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.ReadOnlyObjectException;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.IPinCodeCredential;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.common.ClaimValue;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCard;
import org.eclipse.higgins.icard.common.utils.CardContext;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.SelfIssuedCardClaims;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * @author bus
 * 
 */
public class CardStorePersonalCard extends PersonalCard {


    public CardStorePersonalCard() throws IOException, Exception {
		description_ = "CardStore CardSpace-interoperable personal I-Card";
		provider_ = null;
		isSelfIssued_ = true;
		issuer_ = ISSUER_SELF;
		issuerName_ = "Self";
    }

    public CardStorePersonalCard(CardStorePersonalCard card2copy) {
    	id_ = card2copy.id_;
    	name_ = card2copy.name_;
    	image_ = card2copy.image_;
    	imageMimeType_ = card2copy.imageMimeType_;
    	description_ = card2copy.description_;
		provider_ = card2copy.provider_;
		isSelfIssued_ = card2copy.isSelfIssued_;
		issuer_ = card2copy.issuer_;
		issuerName_ = card2copy.issuerName_;
		timeLastUpdated_ = new Date();
		rawMasterKey_ = card2copy.rawMasterKey_;
    }

    public CardStorePersonalCard(ICardProvider provider, Element element)
	    throws IOException, Exception {
		if (provider == null)
		    throw new IllegalArgumentException("Parameter \"provider\" is null");
		if (element == null)
		    throw new IllegalArgumentException("Parameter \"element\" is null");
		description_ = "CardStore CardSpace-interoperable personal I-Card";
		provider_ = provider;
		super.initFromXML(element);
    }

    public void setIId(URI id) {
    	id_ = id;
    }

    public void setVersion(String version) {
    	version_ = version;
    }

    public void setClaimList(List claims) throws CardException {
    	super.setClaims(claims);
    }

    public void setPinCode(IPinCodeCredential pinCodeCredential)
	    throws CardException {
    }

    public void applyUpdates() throws InvalidStateException, CardException {
    }

    public void beginUpdates() throws InvalidStateException {
    }

    public void cancelUpdates() throws InvalidStateException {
    }

    public boolean isEditMode() {
	return false;
    }

    public boolean isEditable() {
	return false;
    }

    public void setImage(byte[] newImage, String newImageType)
	    throws CardException {
	this.image_ = newImage;
	this.imageMimeType_ = newImageType;
    }

    public void setIssuerName(String name) throws CardException {
	this.issuerName_ = name;
    }

    public void setName(String newName) throws CardException {
	this.name_ = newName;
    }

    public void setTimeIssued(Date date) throws CardException {
	this.timeIssued_ = date;
    }

    public void setTimeExpires(Date date) throws CardException {
	this.timeExpires_ = date;
    }
    
    public void setTimeLastUpdate(Date date) throws CardException {
	this.timeLastUpdated_ = date;
    }

    public IClaim createClaim(String type) throws InvalidTypeException,
	    ReadOnlyObjectException {
	throw new ReadOnlyObjectException();
    }

    public IClaim setClaim(IClaim copyFrom) throws InvalidClaimException,
	    InvalidTypeException, ReadOnlyObjectException {
	throw new ReadOnlyObjectException();
    }

    protected void retrieveClaims(ICredential credential) throws 
    		AuthenticationRequiredException, AuthenticationException, CardException {
    	//If credential not NULL, then decrypt the master key and claim values     	
    	if(credential != null && (credential instanceof IPinCodeCredential)){
    		IPinCodeCredential ipc = (IPinCodeCredential) credential;
    		byte[] pinCode = ipc.getPinCode(); //From a UTF-8 encoded String
    		
    		try {
    			decrypt(pinCode);
			} catch (Exception e) {
				throw new CardException("Cannot decrypt the PIN protected card", e);
			} 
    	} 

		ArrayList claimValueList = new ArrayList();
		ArrayList suppportedTypes = SelfIssuedCardClaims.getSupportedClaimTypeList();
		int size = suppportedTypes.size();
		try {
			for (int i = 0; i < size; i++) {
				Object obj = suppportedTypes.get(i);
				if (obj instanceof ClaimType == false)
					throw new CardException("Instance of " + ClaimType.class.getName() + " expected instead of " + obj.getClass().getName());
				ClaimType cType = (ClaimType) obj;
				String type = cType.getType();
				String val = (dirtyClaimValueMap_.containsKey(type)) ? (String)dirtyClaimValueMap_.get(type) : "";
				ClaimValue cValue = new ClaimValue(cType, val, this);
				claimValueList.add(cValue);
			}
		} catch (CardException e) {
			log.error(e, e);
			throw new CardException(e);
		}
		super.setClaims(claimValueList);
    }

    public void setPinCode(String pinCode) throws CardException {
    }

    public void setSupportedTokenTypeList(List supportedTokenTypes) {
	supportedTokenTypes_ = supportedTokenTypes;
    }

    public void setClaimsTypeList(List claimType) {
	claimTypes_ = claimType;
    }

    /**
     * @param key
     *                256 bit random number
     */
    public void setMasterKey(byte key[]) {
    	rawMasterKey_ = key;
    }

//    public Element toXML(Document doc) throws CardException {
//	Element oldRoot = super.toXML(doc);
//
//	Element newRoot = doc.createElementNS(CardContext.IC_NS,
//		"RoamingInformationCard");
//	// Copy the attributes from InformationCard to RoamingInformationCard
//	NamedNodeMap attrs = oldRoot.getAttributes();
//	for (int i = 0; i < attrs.getLength(); i++) {
//	    Attr attr2 = (Attr) doc.importNode(attrs.item(i), true);
//	    newRoot.getAttributes().setNamedItem(attr2);
//	}
//
//	// add metaData node
//	Element metaData = doc.createElementNS(CardContext.IC_NS,
//		"InformationCardMetaData");
//	// Move all the children to MetaData
//	while (oldRoot.hasChildNodes()) {
//	    metaData.appendChild(oldRoot.getFirstChild());
//	}
//
//	// add privateData node
//	Element privateData = doc.createElementNS(CardContext.IC_NS,
//		"InformationCardPrivateData");
//	Element masterKey = doc.createElementNS(CardContext.IC_NS, "MasterKey");
//	if (masterKey_ == null) {
//	    // Add a master key: 256 bit random number
//	    byte key[] = new byte[256 / 8];
//	    masterKey_ = SecureRandom.getSeed(32);
//	}
//	String base64Key = CardCryptography.encodeBase64(masterKey_, 0);
//	masterKey.appendChild(doc.createTextNode(base64Key));
//	privateData.appendChild(masterKey);
//	// Add ClaimValueList
//	Element claimValueList = doc.createElementNS(CardContext.IC_NS,
//		"ClaimValueList");
//	java.util.Set setClaimValues = claimValues_.entrySet();
//	Iterator values = setClaimValues.iterator();
//	while (values.hasNext()) {
//	    java.util.Map.Entry meClaimValue = (java.util.Map.Entry) values
//		    .next();
//	    ClaimValue claim = (ClaimValue) meClaimValue.getValue();
//
//	    Element claimValue = doc.createElementNS(CardContext.IC_NS,
//		    "ClaimValue");
//	    claimValue.setAttribute("Uri", claim.getType().getType());
//	    Element value = doc.createElementNS(CardContext.IC_NS, "Value");
//	    value.appendChild(doc.createTextNode(claim.getValue()));
//	    claimValue.appendChild(value);
//	    claimValueList.appendChild(claimValue);
//	}
//
//	privateData.appendChild(claimValueList);
//
//	newRoot.appendChild(metaData);
//	newRoot.appendChild(privateData);
//
//	// Replace the InformationCard with RoamingInformationCard
//	oldRoot.getParentNode().replaceChild(newRoot, oldRoot);
//	return newRoot;
//    }

    /**
     * @param doc
     * @param metaData
     */
    private void addTokenServiceList(Document doc,
	    Element informationCardMetaData) {
	Element tokenServiceList = doc.createElementNS(CardContext.IC_NS,
		"TokenServiceList");
	informationCardMetaData.appendChild(tokenServiceList);
	// ITokenService tokenSrv = (ITokenService) it.next();
	Element tokenService = doc.createElementNS(CardContext.IC_NS,
		"TokenService");
	tokenServiceList.appendChild(tokenService);
	// IEndpointReference endpoint =
	// tokenSrv.getEndpointReference();
	Element endpointReference = doc.createElementNS(CardContext.WSA_NS,
		"EndpointReference");
	tokenService.appendChild(endpointReference);
	// Address
	Element address = doc.createElementNS(CardContext.WSA_NS, "Address");
	XMLUtils.setTextContent(address, ISSUER_SELF);
	endpointReference.appendChild(address);
	// MetaData
	Element metaData = doc.createElementNS(CardContext.WSA_NS, "MetaData");
	XMLUtils.setTextContent(address, ISSUER_SELF);
	endpointReference.appendChild(address);

	// /wsa:EndpointReference/wsa:Metadata is optional, ie:
	// http://www.w3.org/TR/2005/CR-ws-addr-core-20050817/
	// if( endpoint.getMetadata() != null) {
	// Node metadata = doc.importNode(endpoint.getMetadata(), true);
	// endpointReference.appendChild(metadata);
	// }
	// from the old InformationCard.java branch
	// Zurich-Catalyst
	// // if( endpoint.getIdentity() != null) {
	// // Node identity = doc.importNode(endpoint.getIdentity(), true);
	// // endpointReference.appendChild(identity);
	// // }
	// // ICredentialDescriptor uCredential = tokenSrv.getUserCredential();
	// // Node credential = doc.importNode(uCredential.asXML(), true);
	// // tokenService.appendChild(credential);
	// }
    }

	public boolean validatePINCode(String pinCode) 
			throws UnsupportedEncodingException, NoSuchAlgorithmException {		
		byte[] userInputBytes = pinCode.getBytes("UTF-16LE");
		MessageDigest md = MessageDigest.getInstance("SHA1");
		md.update(userInputBytes);
		byte[] inputDigest = md.digest();
		
		return Arrays.equals(inputDigest, pinDigest_);
	}
	
	public void lock(IPinCodeCredential credential) throws Exception{
		if(pinStatus == NO_PIN){
			byte[] pinCode = credential.getPinCode(); //UTF-8 byte array
			encrypt(pinCode);
			
			String pinString = new String(pinCode, "UTF-8");
			byte[] unicodePinString = pinString.getBytes("UTF-16LE");
			MessageDigest md = MessageDigest.getInstance("SHA1");
			md.update(unicodePinString);
			pinDigest_ = md.digest();
			md.reset();
			
			pinStatus = LOCKED;
		}
	}
	
	public void unlock(IPinCodeCredential credential) throws Exception{
		if(pinStatus == LOCKED){
			retrieveClaims(credential);
			pinStatus = UNLOCKED;
		}
	}
    
	private void encrypt(byte[] pinCode) 
			throws NoSuchAlgorithmException, UnsupportedEncodingException, 
			        InvalidKeyException, NoSuchPaddingException, 
			        InvalidAlgorithmParameterException, IllegalBlockSizeException, 
			        BadPaddingException, CardException{		
		byte[] salt = new byte[16];
		SecureRandom.getInstance("SHA1PRNG").nextBytes(salt);
		byte[] iter_count = new byte[] {-24, 3, 0, 0};	//1000	
		byte[] iv = new byte[16];
		SecureRandom.getInstance("SHA1PRNG").nextBytes(iv);
		
		//derieve the key
		MessageDigest sha256 = MessageDigest.getInstance("SHA-256");
		sha256.update(pinCode);
		sha256.update(salt);
		byte[] key = sha256.digest();
		for(int i=1; i<1000; i++){
			sha256.update(key);
			key = sha256.digest();
		}
		sha256.reset();
		
		//encrypt the claim values
		java.util.Set setClaimTypes = dirtyClaimValueMap_.keySet();
		Iterator types = setClaimTypes.iterator();
		while (types.hasNext()) {
			String claimType = (String) types.next();
			String claim = (String) dirtyClaimValueMap_.get(claimType);
			byte[] cleartext_bytes = claim.getBytes("UTF-16LE");
			byte[] encrypted_bytes = cipherField(Cipher.ENCRYPT_MODE, iv, key, 
				cleartext_bytes);
			String encrypted_str = Base64Utility.encode(encrypted_bytes);
			dirtyClaimValueMap_.put(claimType, encrypted_str);
		}
		
		//encrypt the master key
		byte[] enc_master_key = cipherField(Cipher.ENCRYPT_MODE, iv, key, rawMasterKey_);		
		rawMasterKey_ = new byte[1 + enc_master_key.length + salt.length + iv.length + iter_count.length];
		
		rawMasterKey_[0] = 2; //version number
		System.arraycopy(salt, 0, this.rawMasterKey_, 1, 16);
		System.arraycopy(iter_count, 0, this.rawMasterKey_, 17, 4);
		System.arraycopy(iv, 0, this.rawMasterKey_, 21, 16);
		System.arraycopy(enc_master_key, 0, rawMasterKey_, 37, enc_master_key.length);
	}
    
    private void decrypt(byte[] pinCode) 
		throws NoSuchAlgorithmException, CardException, InvalidKeyException, 
			    NoSuchPaddingException, InvalidAlgorithmParameterException, 
			    IllegalBlockSizeException, BadPaddingException, 
			    UnsupportedEncodingException{
		byte[] master_key = getMasterKey();
		byte[] salt = new byte[16];
		System.arraycopy(master_key, 1, salt, 0, 16); 
		byte[] iter_count = new byte[4];
		System.arraycopy(master_key, 17, iter_count, 0, 4);	
		byte[] iv = new byte[16];
		System.arraycopy(master_key, 21, iv, 0, 16);
		
		//Drieve the key from the pin code.
		MessageDigest sha256 = MessageDigest.getInstance("SHA-256");
		sha256.update(pinCode);
		sha256.update(salt);
		byte[] key = sha256.digest();
        for (int i = 1; i < 1000; i++) {
        	sha256.update(key);
            key = sha256.digest();
        }
        sha256.reset();
    			
		//Decrypt the claim values
        Set claimTypeUris = this.dirtyClaimValueMap_.keySet();
		Object[] claimTypeStrs = claimTypeUris.toArray();		
		if(claimTypeStrs!=null){
			for(int i=0; i< claimTypeStrs.length; i++){
				String cv = (String) dirtyClaimValueMap_.get(claimTypeStrs[i]);
				byte[] value = Base64Utility.decode(cv);
				byte[] decrypted_value = cipherField(Cipher.DECRYPT_MODE, iv, key, value);
				String decrypted_str = new String(decrypted_value, "UTF-16LE");
				dirtyClaimValueMap_.put(claimTypeStrs[i], decrypted_str);
			}
		}
		
		//Also decrypt the master key
		byte[] masterKeyValue = new byte[rawMasterKey_.length - 37];
		System.arraycopy(rawMasterKey_, 37, masterKeyValue, 0, rawMasterKey_.length-37);
		byte[] tmp	= cipherField(Cipher.DECRYPT_MODE, iv, key, masterKeyValue);
		rawMasterKey_ = (byte[]) tmp.clone();
	}
    	
	private byte[] cipherField(int mode, byte[] iv, byte[] derivedKey, byte[] field) 
		throws CardException, NoSuchAlgorithmException, NoSuchPaddingException, 
				InvalidKeyException, InvalidAlgorithmParameterException, 
				IllegalBlockSizeException, BadPaddingException{		
        Key encKey = new SecretKeySpec(derivedKey, "AES");
        Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
        IvParameterSpec ips = new IvParameterSpec(iv);
        cipher.init(mode, encKey, ips);
        
        byte[] out = cipher.doFinal(field);
		return out;
	}

	public IInformationCardExtension addExtension(IElement extension) throws CardException {
		//TODO implement this method
		throw new CardException ("Not implemented");
	}

	public IInformationCardExtension[] getExtensions() {
		//TODO implement this method
		return null;
	}

	public boolean hasExtensions() {
		//TODO implement this method
		return false;
	}

	public void removeExtension(IInformationCardExtension extension) throws CardException {
		//TODO implement this method
		throw new CardException ("Not implemented");
	}

	public byte[] getMasterKey() throws CardException {
		return rawMasterKey_;
	}

}
