/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Anthony Bussani - Initial API and implementation
 *     Thomas Gross - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.provider.securestorage;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.CardStoreException;
import org.eclipse.higgins.icard.CardStoreStrategy;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardConstants;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.IInformationCard;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.common.BasicCardProvider;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.common.utils.CardContext;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.registry.IConfiguration;
import org.eclipse.higgins.sts.utilities.XMLHelper;
import org.w3c.dom.Element;

/**
 * ICardProvider that stores ICards in a secure encrypted cardstore file (CRDS).
 * The SecureStorageICardProvider holds the ICards in memory and synchronizes
 * them with the encrypted file on demand.
 * 
 * The ICardProvider leverages instances of ICardStoreStrategy to determine how
 * the ICards are specifically stored and synchronized to disk. The
 * SecureStorageStrategy is the default strategy with an encrypted CRDS file as
 * back-end.
 * 
 * The method for exporting ICards to files is determined by the
 * ICardStoreStrategy: an export under SecureStorageStrategy stores the exported
 * ICards on another encrypted CRDS file.
 * 
 * @see ICardStoreStrategy
 * @see SecureStorageStrategy
 * 
 * @author Thomas Gross
 */
//public class SecureStorageICardProvider implements ICardProvider {
public class SecureStorageICardProvider extends BasicCardProvider {

    public static final String FILENAME = "store.filename";

    public static final String PASSWORD = "store.password";

    protected static String ID = "org.eclipse.higgins.icard.provider.securestorage";

    private static final String CLASS_NAME = SecureStorageICardProvider.class.getName();

    private static final String _name = "Secure Cardspace I-Card Provider";

    private static final String _description = "Stores Cardspace-based ICard in a secure storage. The actual strategy for storing and retrieving cards can be configured.";

    private static final Class DEFAULT_CARD_STORE_STRATEGY = SecureStorageStrategy.class;

    public static final String DS_NS = "http://www.w3.org/2000/09/xmldsig#";

    private static Log log = LogFactory.getLog(SecureStorageICardProvider.class);

    public static final String CS_CARDSTORE_FILE = "cardstore.file";

    public static final String CS_PASSWORD = "cardstore.password";

    protected ProviderConfiguration config = new ProviderConfiguration(ID);

    protected Map _icards;

    private CardStoreStrategy _strategy;

    private boolean _lazy_synch = false;

    /**
         * Creates a new SecureStoraceICardProvider based on the default
         * ICardStoreStrategy. The provider does not synchryonize with the
         * CardStore while constructor invocation.
         * 
         * @author Thomas Gross
         */
    public SecureStorageICardProvider() {
	this(DEFAULT_CARD_STORE_STRATEGY);
    }

    /**
         * Creates a new SecureStoraceICardProvider based a dynamically loaded
         * ICardStoreStrategy class. The class is instanciated with its default
         * constructor. The provider does not synchronize with the CardStore
         * while constructor invocation.
         * 
         * @throws IllegalArgumentException
         *                 if the strategyClass is null
         * 
         * @author Thomas Gross
         */
    public SecureStorageICardProvider(Class strategyClass) {
	log.info("Create a new SecureStorageICardProvider");
	if (strategyClass == null)
	    throw new IllegalArgumentException("Strategy may not be null.");

	Constructor strategyConstructor;
	CardStoreStrategy strategy;

	try {
	    strategyConstructor = strategyClass.getConstructor((Class[]) null);
	} catch (Exception e) {
	    log
		    .error(strategyClass
			    + "::Reflect invocation of strategy constructor failed Strategie class: ["
			    + strategyClass + "] ).");
	    throw new UnsupportedOperationException(
		    strategyClass
			    + "::ICardStoreStrategy cannot be instantiated (Strategie class: ["
			    + strategyClass + "] ).");
	}
	try {
	    strategy = (CardStoreStrategy) strategyConstructor
		    .newInstance((Object[]) null);
	} catch (Exception e) {
	    log
		    .error(strategyClass
			    + "::Reflect invocation of strategy constructor failed Strategie class: ["
			    + strategyClass + "] ).");
	    throw new UnsupportedOperationException(
		    strategyClass
			    + "::ICardStoreStrategy cannot be instantiated (Strategie class: ["
			    + strategyClass + "] ).");
	}

	if (strategy == null)
	    throw new IllegalArgumentException("Strategy may not be null.");

	_icards = new HashMap();
	String filename = (String)config.getProperty(CS_CARDSTORE_FILE);
	String password = (String)config.getProperty(CS_PASSWORD);
	log.info("Cardstore file: ["+filename+"]");
	if ( filename != null && password != null) {
	    log.trace("filename=" + filename);
	    PasswordHandler passwordHandler = new PasswordHandler(password);
	    try {
		strategy.initialize(null, this, filename, passwordHandler,
			config);
	    } catch (CardStoreException e) {
		log.error("Problem when initializing the Strategy", e);
	    }
	}
	_strategy = strategy;
    }

    /**
         * Creates a new SecureStoraceICardProvider based on an instance of
         * ICardStoreStrategy. The provider does not synchryonize with the
         * CardStore while constructor invocation.
         * 
         * @throws IllegalArgumentException
         *                 if the strategyClass is null
         * 
         * @author Thomas Gross
         */
    public SecureStorageICardProvider(CardStoreStrategy strategy) {
	if (strategy == null)
	    throw new IllegalArgumentException("Strategy may not be null.");

	_icards = new HashMap();
	_strategy = strategy;
    }

    /**
         * Validates that a ICard with specific ID can be generated.
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#canCreateCard(javax.security.auth.callback.CallbackHandler,
         *      java.lang.String, java.util.Properties)
         * @throws UnsupportedOperationException
         * 
         * @author Thomas Gross
         */
    public boolean canCreateCard(CallbackHandler authHandler, String id,
	    Properties props) {
	log.trace(CLASS_NAME + "::canCreateCard(): ConnectiveId:" + id);
	// TODO How to check the creation option?
	throw new UnsupportedOperationException(CLASS_NAME
		+ "::canCreateCard() not supported.");
    }

    /**
         * Creates a new ICard with specific ID.
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#createCard(javax.security.auth.callback.CallbackHandler,
         *      java.lang.String, java.util.Properties)
         * @throws UnsupportedOperationException
         * 
         * @author Thomas Gross
         */
    public ICard createCard(CallbackHandler authHandler, String id,
	    Properties props) {
	log.trace("createCards ConnectiveId:" + id);
	// TODO Create new card;
	throw new UnsupportedOperationException(CLASS_NAME
		+ "::createCard() not supported.");
    }

    /**
         * Deletes an ICard from the CardStore. The method assumes that the ID
         * of the ICard is unique. The Method leverages the equals method to
         * assert that indeed the correct card was removed.
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#deleteCard(javax.security.auth.callback.CallbackHandler,
         *      org.eclipse.higgins.icard.ICard)
         * 
         * @author Thomas Gross
         * @throws CardException
         */
    public void deleteCard(CallbackHandler authHandler, ICard card)
	    throws CardException {
	log.trace(CLASS_NAME + "::deleteCard(): card=" + card);
	/*
         * TODO InformationCard.equals() seems still not working for some card
         * and throwing NullException
         * 
         * if (!_icards.containsValue(card)) { throw new CardException("Card not
         * existing:" + card.getID()); }
         */

	if (!_icards.containsKey(card.getID())) {
	    log.error("card [" + card.getName() + "] not found");
	    Iterator allCards = _icards.values().iterator();
	    // precondition: policy != null
	    while (allCards.hasNext()) {
		ICard card1 = (ICard) allCards.next();
		log.error("found card [" + card1.getName() + "]");
	    }
	    throw new CardException("Card not existing:" + card.getID());
	}

	_icards.remove(card.getID());

	if (!_lazy_synch) {
	    _strategy.synchFromMap(authHandler, _icards);
	}
    }

    /**
         * Deletes ALL ICards from the CardStore.
         * 
         * @author Thomas Gross
         * 
         */
    public void clear(CallbackHandler authHandler) {
	log.trace(CLASS_NAME + "::clear()");
	_icards.clear();
    }

    public void addCard(CallbackHandler authHandler, ICard card)
	    throws CardException {
	if (card == null)
	    throw new CardException("ICard to be added is null.");
	_icards.put(card.getID(), card);
	if (!_lazy_synch)
	    _strategy.synchFromMap(authHandler, _icards);

    }

    /**
         * Verifies that the ICardProvider contains an ICard with specified
         * ConnectiveId.
         * 
         * @param ConnectiveId
         * @return true if an ICard with ConnectiveId is present.
         * 
         * @author Thomas Gross
         */
    public boolean containsCard(CallbackHandler authHandler, String id)
	    throws CardException {
	log.trace(CLASS_NAME + "::containsCard(): ConnectiveId=" + id);
	if (!_lazy_synch)
	    _strategy.synchFromStore(authHandler, _icards);

	return _icards.containsKey(id);
    }

    /**
         * Verifies that the ICardProvider contains an ICard.
         * 
         * @param icard
         * @return true if the very ICard is present.
         * 
         * @author Thomas Gross
         */
    public boolean containsCard(CallbackHandler authHandler, ICard icard)
	    throws CardException {
	log.trace(CLASS_NAME + "::containsCard(): by ICard, ConnectiveId="
		+ icard.getID());
	if (!_lazy_synch)
	    _strategy.synchFromStore(authHandler, _icards);

	return _icards.containsValue(icard);
    }

    /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#exportCards(javax.security.auth.callback.CallbackHandler,
         *      java.util.Iterator, java.io.OutputStream)
         */
    public void exportCards(CallbackHandler authHandler, Iterator cards,
	    OutputStream out) throws CardException {
	log.trace(CLASS_NAME + "::exportCards(): cards=" + cards);

	_strategy.exportCards(authHandler, cards, out);
    }

    /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#getDescription()
         */
    public String getDescription() {
	log.trace(CLASS_NAME + "::getDescription()");
	return _description;
    }

    /**
         * Returns an ICard from the CardStore based upon the ID.
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#getICardByID(javax.security.auth.callback.CallbackHandler,
         *      java.lang.String)
         * @throws ICardException
         *                 if the desired card is not found.
         * 
         * @author Thomas Gross
         */
    public ICard getICardByID(CallbackHandler authHandler, String ID)
	    throws CardException {
	log.trace(CLASS_NAME + "::getICardByID(): ID=" + ID);
	// TODO make clear whether that the super interface really uses the
	// ICard ID or CUID. ==> CUID not implemented in InformationCard!

	if (!_lazy_synch)
	    _strategy.synchFromStore(authHandler, _icards);

	if (!_icards.containsKey(ID))
	    throw new CardException(CLASS_NAME
		    + "::getICardByID(): ICard with index not found, ID=" + ID);
	// postcondition: _icards indeed contain key

	return (ICard) _icards.get(ID);
    }

    /**
         * Returns an iterator of all cards that satisfy the specified IPolicy.
         * If the policy is null, the full list of ICards is returned.
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#getICards(javax.security.auth.callback.CallbackHandler,
         *      org.eclipse.higgins.icard.IPolicy)
         * @author Thomas Gross
         */
//    public Iterator getICards(CallbackHandler authHandler, IPolicy policy) throws CardException {
    public Iterator getICards(CallbackHandler authHandler) throws CardException {
//	log.trace(CLASS_NAME + "[" + ID + "]::getICards(): policy=" + policy);

	if (!_lazy_synch)
	    _strategy.synchFromStore(authHandler, _icards);

	Iterator allCards = _icards.values().iterator();
//	if (policy == null)
	    return allCards;
//
//	List satisfiableCards = new ArrayList();
//
//	// precondition: policy != null
//	while (allCards.hasNext()) {
//	    ICard card = (ICard) allCards.next();
//	    if (policy.isSatisfiedBy(card)) {
//		satisfiableCards.add(card);
//	    }
//	}
//
//	return satisfiableCards.iterator();
    }

    /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#getName()
         */
    public String getName() {
	log.trace(CLASS_NAME + "::getName()");
	return _name;
    }

    /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#getType()
         */
    public String getType() {
	String type = ICardConstants.I_CARD_TYPE_CARDSPACE; // + "#BUG";
	log.trace("return -> [" + type + "]");
	return type;
    }

    /**
         * Chages the password of the underlying cardstore. The old password
         * needs to be provided to spawn the password change. Changing the
         * password may require the re-encryption of the whole CardStore.
         * 
         * @param oldPassword
         *                previous password under which the CardStore was
         *                encrypted
         * @param newPassword
         *                new password to be set
         * @throws ICardException
         *                 if the oldPassword paramenter was not correct.
         * @throws ICardException
         *                 if the CardStore file could not be read or written.
         * 
         * @author Thomas Gross
         */
    public void changePassword(char[] oldPassword, char[] newPassword)
	    throws CardException {
	if (!_strategy.isPasswordProtected()) {
	    throw new CardException(
		    "The ICardStoreStrategy does not support passwords.");
	}
	_strategy.changePassword(oldPassword, newPassword);
    }

    /**
         * Chages the password of the underlying cardstore while leveraging a
         * CallbackHandler. The old password needs to be provided to spawn the
         * password change. Changing the password may require the re-encryption
         * of the whole CardStore.
         * 
         * @param authHandler
         * 
         * @throws ICardException
         *                 if the oldPassword paramenter was not correct.
         * @throws ICardException
         *                 if the CardStore file could not be read or written.
         * 
         * @author Thomas Gross
         */
    public void changePassword(CallbackHandler authHandler)
	    throws CardException {
	UIDChangePasswordTriple uid_pass = getChangePasswordCallback(authHandler);

	this.changePassword(uid_pass.getPassword(), uid_pass.getNewPassword());
    }

    /**
         * This is not part of the CardStore interface, maybe the CardManager ?
         * 
         * @param object
         * @param xmlSz
         */
    public void importStore(CallbackHandler authHandler, String asciiStore)
	    throws CardException {
	Map newCards = _strategy.importStore(authHandler, asciiStore);
	_icards = newCards;
	_strategy.synchFromMap(authHandler, _icards);
    }

    /**
         * TODO add it to ICard interface ?
         * 
         * @param object
         * @param idemixCard
         * @throws CardStoreException
         */
    public ICard importCard(CallbackHandler authHandler, ICard icard)
	    throws CardStoreException {
	// side-effect storing the card.
	_icards.put(icard.getID(), icard);

	if (!_lazy_synch)
	    _strategy.synchFromMap(authHandler, _icards);

	// postcondition: icard read and not null.
	return icard;
    }

    /**
         * Imports a new ICard into the ICardProvider. The method assumes that
         * the imported ICard is compatible with
         * org.eclipse.higgins.icard.provider.cardspace.zurich.InformationCard.
         * The ICard is added to the CardStore as side-effect.
         * 
         * The method is specialized on adding a single card from a CRD file.
         * 
         * @see org.eclipse.higgins.icard.ICardProvider#importCard(javax.security.auth.callback.CallbackHandler,
         *      java.lang.String)
         * 
         * @return ICard imported in the CardStore.
         * 
         * @author Thomas Gross
         */
    public ICard importCard(CallbackHandler authHandler, String filename)
	    throws CardException {
	log.trace(CLASS_NAME + "::importCard(): filename: " + filename);
	File file = new File(filename);
	if (file.isDirectory()) {
	    importCardsFromDirectory(authHandler, filename);
	    return null;
	}
	if (file.getName().endsWith(".crds")) {
	    importCardsFromCRDS(authHandler, filename);
	    return null;
	}
	return importCardFromCRD(authHandler, filename);
    }

    protected ICard importCardFromCRD(CallbackHandler authHandler,
	    String filename) throws CardException {
	log.trace(CLASS_NAME + "::importCard(): filename: " + filename);

	ICard icard = null;
	icard = retrieveICardString(filename);
	if (icard == null)
	    throw new CardException(
		    CLASS_NAME
			    + ":: the ICard XML representation could not be parsed. Parsing returned null. Filename="
			    + filename);

	// side-effect storing the card.
	_icards.put(icard.getID(), icard);

	if (!_lazy_synch)
	    _strategy.synchFromMap(authHandler, _icards);

	// postcondition: icard read and not null.
	return icard;
    }

    public Map importCardsFromDirectory(CallbackHandler authHandler,
	    String filename) throws CardException {
	throw new CardException(
		"Import of whole directories not supported. Filename="
			+ filename);
    }

    public Map importCardsFromCRDS(CallbackHandler authHandler, String filename)
	    throws CardException {
	throw new CardException(
		"Import of whole directories not supported. Filename="
			+ filename);
    }

    private ICard retrieveICardString(String filename) throws CardException {
	File file = new File(filename);
	byte[] crdBytes = new byte[(int) file.length()];

	try {
	    FileInputStream fis = new FileInputStream(file);
	    fis.read(crdBytes);
	} catch (FileNotFoundException e) {
	    log.error(CLASS_NAME
		    + "::importCard(): ICard file not found. Filename="
		    + filename);
	    // org.eclipse.higgins.sts.utilities.ExceptionHelper.Log(log,
	    // e);
	    throw new CardException(CLASS_NAME
		    + "::importCard(): ICard file not found. Filename="
		    + filename, e);
	} catch (IOException e) {
	    log
		    .error(CLASS_NAME
			    + "::importCard(): IOException while reading the imported ICard.");
	    // org.eclipse.higgins.sts.utilities.ExceptionHelper.Log(log,
	    // e);
	    throw new CardException(
		    CLASS_NAME
			    + "::importCard(): IOException while reading the imported ICard.",
		    e);
	}

	String strIcard = new String(crdBytes);
	// BUS ICard[] icards = null;
	ICard icard = null;
	try {
	    // BUS icards = InformationCard.parseCrd(strIcard);
	    // a card coming fro outside can only be a Managed Card
	    byte[] bom = { (byte) 0xEF, (byte) 0xBB, (byte) 0xBF };
	    String BOM = new String(bom);

	    if (strIcard.startsWith(BOM)) {
		strIcard = strIcard.substring(BOM.length());
	    }
	    Element element = null;
	    element = XMLHelper.toDOM(strIcard);
	    //	Do not need to look at Signature as we are going directly to childrens
//	    element = XMLUtils.getChildElement(element, DS_NS, "Signature");
	    element = XMLUtils.getChildElement(element, DS_NS, "Object");
	    element = XMLUtils.getChildElement(element, CardContext.IC_NS, "InformationCard");
	    
//	    XPath xPath = XMLUtils.createXPath();
//	    NamespaceContext namespaceContext = (NamespaceContext)xPath.getNamespaceContext();
//	    namespaceContext.addNamespace("ds", DS_NS);
//	    xPath.setNamespaceContext(namespaceContext);
//	    element = (Element) xPath.evaluate(
//		    "/ds:Signature/ds:Object/ic:InformationCard", element, XPathConstants.NODE);
////	TODO: convert for using xalan and JDK 1.4
//	    NamespaceContext ctx = new NamespaceContext();
//	    ctx.addNamespace("ic", CardContext.IC_NS);
//	    ctx.addNamespace("ds", DS_NS);
//	    PrefixResolver prefixResolver = JAXPPrefixResolver( ctx);
//	    XPath xpath = new XPath("/ds:Signature/ds:Object/ic:InformationCard", new NodeLocator(), prefixResolver, SELECT);
//		XPathFactory factory = XPathFactory. newInstance();
//		XPath xpath = factory.newXPath();
//		xpath.setNamespaceContext(ctx);
//
//	    XPath xPath = XMLUtils.createXPath();
//	    NamespaceContext namespaceContext = (NamespaceContext) xPath
//		    .getNamespaceContext();
//	    namespaceContext.addNamespace("ds", DS_NS);
//	    xPath.setNamespaceContext(namespaceContext);
//	    element = (Element) xPath.evaluate(
//		    "/ds:Signature/ds:Object/ic:InformationCard", element,
//		    XPathConstants.NODE);

	    icard = new CardStoreManagedCard(this, element);
	} catch (Exception e) {
	    log.error(CLASS_NAME
		    + "::importCard(): The ICard under filename=\"" + filename
		    + "\" could not be parsed.");
	    throw new CardException(CLASS_NAME
		    + "::importCard(): The ICard under filename=\"" + filename
		    + "\" could not be parsed.", e);
	}

	if (icard == null)
	    throw new CardException(CLASS_NAME
		    + "::importCard(): The ICard file under filename=\""
		    + filename + "\" does not contains any ICards.");

	return icard;
    }

    /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.higgins.registry.IServiceProvider#getConfiguration()
         */
    public IConfiguration getConfiguration() {
	log.trace(CLASS_NAME + "::getConfiguration()");
	return config;
    }

    /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.higgins.registry.IServiceProvider#getID()
         */
    public String getID() {
	log.trace(CLASS_NAME + "::getID()");
	return ID;
    }

    /*
         * (non-Javadoc)
         * 
         * @throws UnsupportedOperationException the ID of the provider is
         * immutable.
         * 
         * @see org.eclipse.higgins.registry.IServiceProvider#setID()
         */
    public void setID(String id) throws Exception {
	log.trace(CLASS_NAME + "::setID()");
	SecureStorageICardProvider.ID = id;
    }

    public CardStoreStrategy getStrategy() {
	return _strategy;
    }

    public void setStrategy(CardStoreStrategy _strategy) {
	this._strategy = _strategy;
    }

    public boolean isLazySynchronizing() {
	return _lazy_synch;
    }

    public void setLazySynchronizing(boolean _lazy_synch) {
	this._lazy_synch = _lazy_synch;
    }

    /**
         * Creates a callback for username and password and submits the callback
         * to the authHandler. The returned username and password pair is
         * encapsulated as UIDPasswordPair.
         * 
         * @param authHandler
         *                authentication handler of Higgins
         * @return pair of username and password.
         * @throws CardException
         *                 if the callback experiences an IOException.
         * @throws CardException
         *                 if the pair of username and password is malformed.
         */
    protected UIDPasswordPair getUIDCallback(CallbackHandler authHandler)
	    throws CardException {
	log.trace(CLASS_NAME + "::getUIDCallback()");
	if (authHandler != null) {
	    NameCallback nc = new NameCallback("User Number: ");
	    PasswordCallback pc = new PasswordCallback("Password: ", false);
	    Callback[] callbacks = new Callback[] { nc, pc };
	    try {
		log.trace(CLASS_NAME
			+ "::getUIDCallback(): send callbacks to authHandler.");
		authHandler.handle(callbacks);

		String username = nc.getName();
		char[] password = pc.getPassword();
		if (username == null || password == null)
		    throw new CardException("Username or password were null.");
		if (username.equals(""))
		    throw new CardException("Username may not be empty.");
		// Password may be empty string, username not.

		UIDPasswordPair uid_pass = new UIDPasswordPair(username,
			password);
		return uid_pass;
	    } catch (IOException e) {
		throw new CardException(e);
	    } catch (UnsupportedCallbackException e) {
		throw new CardException(e);
	    }
	}
	throw new IllegalArgumentException(CLASS_NAME
		+ "::getUIDCallback: authHandler parameter cannot be null");
    }

    /**
         * Creates a callback for username and old as well new password and
         * submits the callback to the authHandler. The returned username and
         * password triple is encapsulated as UIDChangePasswordTriple.
         * 
         * @param authHandler
         *                authentication handler of Higgins
         * @return triple of username and old/new password.
         * @throws CardException
         *                 if the callback experiences an IOException.
         * @throws CardException
         *                 if the pair of username and password is malformed.
         */
    protected UIDChangePasswordTriple getChangePasswordCallback(
	    CallbackHandler authHandler) throws CardException {
	log.trace(CLASS_NAME + "::getChangePasswordCallback()");
	if (authHandler != null) {
	    NameCallback nc = new NameCallback("User Number: ");
	    PasswordCallback pc = new PasswordCallback("Old Password: ", false);
	    PasswordCallback npc = new PasswordCallback("New Password: ", false);
	    Callback[] callbacks = new Callback[] { nc, pc, npc };
	    try {
		log
			.trace(CLASS_NAME
				+ "::getChangePasswordCallback(): send callbacks to authHandler.");
		authHandler.handle(callbacks);

		String username = nc.getName();
		char[] password = pc.getPassword();
		char[] newPassword = npc.getPassword();
		if (username == null || password == null || newPassword == null)
		    throw new CardException("Username or password were null.");
		if (username.equals(""))
		    throw new CardException("Username may not be empty.");
		// Password may be empty string, username not.

		UIDChangePasswordTriple uid_pass = new UIDChangePasswordTriple(
			username, password, newPassword);
		return uid_pass;
	    } catch (IOException e) {
		throw new CardException(e);
	    } catch (UnsupportedCallbackException e) {
		throw new CardException(e);
	    }
	}
	throw new IllegalArgumentException(CLASS_NAME
		+ "::getUIDCallback: authHandler parameter cannot be null");
    }

    // INTERNAL CLASSES

    /**
         * Encapsulated immutable pair of username and password;
         * 
         * @author Thomas Gross
         */
    public class UIDPasswordPair {
	private final String UID;

	private final char[] password;

	UIDPasswordPair(String UID, char[] password) {
	    this.UID = UID;
	    this.password = password;
	}

	String getUID() {
	    return this.UID;
	}

	char[] getPassword() {
	    return this.password;
	}
    }

    /**
         * Encapsulated immutable triple of username and old as well new
         * password;
         * 
         * @author Thomas Gross
         */
    public class UIDChangePasswordTriple {
	private final String UID;

	private final char[] password;

	private final char[] newPassword;

	UIDChangePasswordTriple(String UID, char[] password, char[] newPassword) {
	    this.UID = UID;
	    this.password = password;
	    this.newPassword = newPassword;
	}

	String getUID() {
	    return this.UID;
	}

	char[] getPassword() {
	    return this.password;
	}

	char[] getNewPassword() {
	    return this.newPassword;
	}
    }

    public void synchToStorage(CallbackHandler authHandler)
	    throws CardStoreException {
	_strategy.synchFromMap(authHandler, _icards);
    }

    public void synchFromStorage(CallbackHandler authHandler)
	    throws CardStoreException {
	_strategy.synchFromStore(authHandler, _icards);
    }

    public boolean canImportICard(CallbackHandler authHandler, IElement card) {
	// TODO Auto-generated method stub
	return false;
    }

    public ICard createCard(CallbackHandler authHandler, ICardTemplate template)
	    throws CardException {
	// TODO Auto-generated method stub
	return null;
    }

    public ICardTemplate[] getCardCreationTemplates(CallbackHandler authHandler) {
	// TODO Auto-generated method stub
	return null;
    }

    public ICardTemplate getCardCreationTemplatesByID(CallbackHandler authHandler, String ID) {
		// TODO Auto-generated method stub
		return null;
	}

	public ICard getICardByCUID(CallbackHandler authHandler, String CUID)
	    throws CardException {
	return getICardByCUID(authHandler, new CUID(CUID));    }

    public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid)
	    throws CardException {
	log.info("getICardByCUID(" + cuid + ")");
	Set set = _icards.keySet();
	Iterator iter = set.iterator();
	if (iter.hasNext()) {
	    log.info("first key of the map:" + iter.next());

	}
	if (!this.getID().equals(cuid.getProviderID())) {
	    log.info("Mysmatch: Current Provider[" + this.getID()
		    + "]  Asked Provider [" + cuid.getProviderID() + "]");
	    return null;
	}
	ICard icard = (ICard)_icards.get(cuid.getCardID());
	log.info("retrieving card("+cuid.getCardID()+") -> "+icard);
	return icard;
    }

    public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid,
	    ICredential userCredential) throws AuthenticationException,
	    CardException {
	return getICardByCUID(authHandler, cuid);    }

    public Class[] getSupportedTypes() {
	return new Class[] {IInformationCard.class};
    }

    public ICard importICard(CallbackHandler authHandler, IElement card)
	    throws CardException {
	// TODO Auto-generated method stub
	return null;
    }

    public String getFilename() {
	return (String)config.getProperty(CS_CARDSTORE_FILE);	
    }


    private static IConfiguration initConfig(ICardProvider provider) {
    	try {
    		return ICardRegistry.getInstance().getConfiguration(provider);
    	}
    	catch(Exception e) {
    		log.error(e, e);
    		return null;
    	}
    }

	public ICard parseCardElement(CallbackHandler authHandler, IElement card) {
		// TODO Auto-generated method stub
		return null;
	}

}
