/*******************************************************************************
 * Copyright (c) 2006-2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim
 *******************************************************************************/

package org.eclipse.higgins.idas.api;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.model.IContextModel;

/**
 * As defined at <a href="http://spwiki.editme.com/Context">http://spwiki.editme.com/Context</a> and <a href="http://spwiki.editme.com/ContextProvider">http://spwiki.editme.com/ContextProvider</a>, This interface holds a set of {@link INode} objects.<p>
 * 
 */
public interface IContext
{
	/*
	 * Context-wide methods
	 */

	/**
	 * Opens this Context using the provided authentication object.<p>
	 * Upon success (return with no exception), this Context is in an "open" state 
	 * and is available to have other methods called (such as 
	 * {@link #addRelationship(URI)}, 
	 * {@link #addNode(URI, String)}, 
	 * {@link #exportData(String, String)}, 
	 * {@link #getRelationships()}, 
	 * {@link #getNode(String)}, 
	 * {@link #getNode(String, Iterator)}, 
	 * {@link #getNodes(IFilter)},
	 * {@link #getNodes(IFilter, Iterator)},
	 * {@link #importData(String, String)},
	 * {@link #verifyNodeAttributes(String, Iterator)}}
	 * ). 
	 * Methods requiring the context to be open throw {@link ContextNotOpenException} 
	 * when in an un-open state.<p>
	 * An open state specifies that the specified authentication materials have been used to successfully authenticate to this Context.<p>
	 * If this IContext is already in an open state, this throws {@link ContextOpenException}.<p>
	 * The antithesis of this method is close().
	 *
	 * @param authentication AuthN object used to open the context.  This may be an instance of {@link IAuthNAttributesMaterials} or some other object used to authenticate. 
	 * @return Node ID of the Node representing the identity used to open the context.  May be null.
	 * @throws IdASException
	 * @throws ContextOpenException, AuthenticationException
	 * @see #close() 
	 */
	public String open(Object authentication) throws IdASException, ContextOpenException;
	
	/**
	 * Closes this Context and returns it to an un-open state.<p>
	 * Methods requiring the context to be open throw ContextNotOpenException 
	 * when in an un-open state.<p>
	 * 
	 * @throws IdASException
	 */
	public void close() throws IdASException;

	/**
	 * Tests whether this context is open and authenticated using the specified identity
	 * @param identity 	May be {@code null}. 
	 * 					When provided, asserts that this context was opened with the specified identity. 
	 * 					When (@code null), no identity check is performed. 
	 * @return {@code true} if this context is open as specified, {@code false} otherwise
	 * @throws IdASException
	 */
	public boolean isOpen(Object identity) throws IdASException;
	
	/**
	 * Returns the schema governing this context.<p>
	 * TODO (Doc): Add pointers to OWL.<p>
	 * TODO (Doc): Add (or point to) documentation which includes sample schema.<p>
	 * TODO (Arch)(Impl): Add method(s) which return the schema as a stream, DOM document, or other iterable object.
	 * 
	 * @return schema is returned as a {@link String} in OWL-DL format.
	 * @throws IdASException
	 */
	public String getSchema() throws IdASException;

	/**
	 * Sets the schema governing this context.<p>
	 * TODO (Doc): Add pointers to OWL.<p>
	 * TODO (Doc): Add (or point to) documentation which includes sample schema.<p>
	 * TODO (Arch)(Impl): Add method(s) which take the schema as a stream, DOM document, or other iterable object.
	 * 
	 * @param schema a {@link String} in OWL-DL format.
	 * @throws IdASException
	 */
	public void setSchema(String schema) throws IdASException;

	/**
	 * Returns a model of the context object as it defined in the context's
	 * schema. Returned model could be used to analyze context's schema and
	 * discover what types of Nodes are defined.
	 * 
	 * @return the model of this Context.
	 * @throws IdASException
	 */
	public IContextModel getContextModel() throws IdASException;

	/**
	 * Returns a reference to this Context. This value is used in Context Relationships and used to open this Context.
	 * 
	 * @return the reference to this Context.
	 * @throws IdASException
	 */
	public URI getContextID() throws IdASException;

	/**
	 * Exports this Context.<p>
	 * TODO (Arch): filter format TBD<p>
	 * TODO (Arch): Does this export only Nodes, or the entire Context (Metadata, Schema, Nodes, Relationships)?
	 * @param filter May be {@code null}. Used to select a subset of this Context to be exported.
	 * @param representationFormat Indicates the format of the string to be returned. TODO (Arch)(Doc): Specify format type. Give example
	 * @return String in the form of representationFormat containing the exported Context.
	 * @throws IdASException
	 */
	public String exportData(String filter, String representationFormat) throws IdASException;

	/**
	 * Imports this Context.<p>
	 * TODO (Arch): filter format TBD<p>
	 * TODO (Arch): Does this import only Nodes, or the entire Context (Metadata, Schema, Nodes, Relationships)?
	 * TODO (Arch)(Doc): Discuss collisions.
	 * @param filter May be {@code null}. Used to select a subset of the external data set to be imported.
	 * @param representationFormat Indicates the format of the external data string being specified in external. TODO (Arch)(Doc): Specify format type. Give example
	 * @throws IdASException
	 */
	public void	importData(String filter, String representationFormat) throws IdASException;

	/*
	 * Factory-like methods for creating various IdAS elements 
	 */
	
	/**
	 * Builds an {@link IAuthNAttributesMaterials} that may be used when calling {@link #open(Object)}
	 */
	public IAuthNAttributesMaterials buildAuthNAttributesMaterials() throws IdASException;
	
	/**
	 * Builds an IAttribute (unassociated with any node).
	 * This is used to build up {@link IAuthNAttributesMaterials}
	 * @param attrID
	 * @return An {@link IAttribute}
	 * @throws IdASException 
	 */
	public IAttribute buildAttribute(URI attrID) throws IdASException;
	
	/**
	 * Builds an IFilter instance that may be used when calling {@link #getNodes(IFilter)}
	 */
	public IFilter buildFilter() throws IdASException;
	
	/**
	 * Builds an assertion for Node Attributes 
	 * (see {@link INode#getAttributes()} and {@link INode#getAttribute(URI)}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 */
	public IFilterAttributeAssertion buildAttributeAssertion() throws IdASException;

	/**
	 * Builds an assertion for Node IDs 
	 * (see {@link INode#getNodeID()}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 * Note that this is simply a convenience method equivalent to creating an
	 * IFilterAttributeAssertion for the 
	 * http://www.eclipse.org/higgins/ontologies/2006/higgins#uniqueIdentifier attribute
	 */
	public IFilterNodeIDAssertion buildNodeIDAssertion() throws IdASException;

	/**
	 * Builds an assertion for Node Metadata 
	 * (see {@link INode#getMetadataSet()}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 */
	public IFilterMetadataAssertion buildMetadataAssertion() throws IdASException;

	/**
	 * Builds an assertion for Node Types 
	 * (see {@link INode#getNodeType()}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 */
	public IFilterNodeTypeAssertion buildNodeTypeAssertion() throws IdASException;

	/**
	 * Builds a simple attribute value to be passed to {@link IFilterAttributeAssertion#setAssertionValue(IAttributeValue)}.<p>
	 * Example:<p>
	 * <code>
	 * IAttributeValue value = buildSimpleValue("http://www.w3.org/2001/XMLSchema/normalizedString", new String("Some Value"));
	 * </code>
	 * @param dataType The type for the simple value
	 * @param value The value for the simple type. 
	 * @return the built attribute value
	 * @throws IdASException
	 * @throws InvalidTypeException
	 */
	public ISimpleAttrValue buildSimpleAttrValue(URI dataType, Object value) throws IdASException, InvalidTypeException;
	
	/**
	 * Builds a complex attribute value to be passed to {@link IFilterAttributeAssertion#setAssertionValue(IAttributeValue)}.<p>
	 * Example:<p>
	 * <code>
	 * TODO: do an example of building this and then adding some attributes
	 * </code>
	 * @param dataType The type for the simple value
	 * @return the built attribute value
	 * @throws IdASException
	 * @throws InvalidTypeException
	 */
	public IComplexAttrValue buildComplexAttrValue(URI dataType) throws IdASException, InvalidTypeException;
	
	/**
	 * Builds a relationship to be passed to {@link #addRelationship(URI)}.<p>
	 * @param relationshipType The type of relationship. TODO (Doc): give example.
	 * @return The resulting Relationship.
	 * @throws IdASException
	 */
	public IContextRelation buildRelationship(URI relationshipType) throws IdASException;


	/*
	 * Methods relating to Nodes
	 */

	/**
	 * Creates a new Node for this Context.<br>
	 * This call is typically followed by one or more calls to 
	 * {@link INode#addAttribute}, {@link INode#addMetadata}
	 * on the returned INode
	 * Note: This operation is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()}
	 * @param nodeType The URI specifying the type of Node being created
	 * @param nodeID May be null.  When specified, uniquely names the node within
	 * this Context.  When null, indicates that the caller plans to later add an 
	 * attribute or attributes which will be used to construct the unique node ID
	 * @return The resulting INode.  This is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()}
	 * @throws InvalidTypeException when the nodeType is invalid
	 * @throws InvalidNodeIDException when the nodeID is invalid
	 * @throws NodeExistsException when the nodeID is known to already exist
	 * @throws IdASException
	 */
	INode addNode(URI nodeType, String nodeID) throws IdASException, InvalidTypeException, InvalidNodeIDException, NodeExistsException;

	/**
	 * Creates a new Node for this Context by copying the data from the passed node<br>
	 * Note: This operation is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()} is called.
	 * @param copyFrom an {@link INode} from which data is copied to construct a new 
	 * Node
	 * @return The resulting INode.  This is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()} is called.
	 * @throws NodeExistsException when copyFrom is known to already exist
	 * @throws IdASException
	 */
	INode addNode(INode copyFrom) throws IdASException, NodeExistsException;

	/**
	 * This is the same as passing null as the attrSelectionList in {@link #getNode(String, Iterator)}
	 *	@see #getNode(String, Iterator)
	 *	 
	 */
	public INode getNode(String nodeID) throws IdASException;
	/**
	 * Returns the Node matching the specified nodeID.
	 * This is effectively the same as calling {@link #getNodes} with
	 * the filter set to {@literal <appropriate filter for nodeID=name>}<p>
	 * 
	 * @param nodeID Contextually unique ID of Node to return.
	 * @param attrSelectionList Iterator holding the {@link URI}s of IAttribute types 
	 * 							that the caller expects to access.  
	 * 							A {@code null}indicates that the caller expects to read all attributes.  
	 * 							An empty set indicates that the caller expects to read no attributes.
	 *                          Note that this is only an indicator of what the caller expects to read. 
	 *                          The caller may eventually end up reading more or less.  The purpose of this 
	 *                          parameter is so that a context provider which gathers data across a network 
	 *                          connection can optimize based on the callers intended usage. 
	 * @return INode.
	 * @throws IdASException (typically a subclass of. See below).
	 * @throws NoSuchNodeException when node specified by nodeID is not found.
	 *
	 */
	public INode getNode(String nodeID, Iterator attrSelectionList) throws IdASException;
	
	/**
	 * @see #getNodes(IFilter, Iterator)
	 * This is the same as passing null as the attrSelectionList in {@link #getNodes(IFilter, Iterator)}
	 * @return an Iterator of {@link INode}s that match the filter 
	 */
	public Iterator getNodes(IFilter filter) throws IdASException;
	
	/**
	 * From this context, return {@link INode}s that match the specified filter <p>
	 * @param filter 	Filter used to constrain the set of {@link INode}s returned.
	 * 					A {@code null} or empty filter causes all {@link INode}s to be returned.
	 * @param attrSelectionList Iterator of {@link URI}s of IAttribute types that the caller expects to access.  A {@code null} indicates that all attributes will be read.  An empty set indicates that no attributes will be read.
	 * @return an Iterator of {@link INode}s that match the filter 
	 * @throws IdASException
	 *
	 */
	public Iterator getNodes(IFilter filter, Iterator attrSelectionList) throws IdASException;
	
	/**
	 * Simple method for comparing a set of attribute assertions against a Node.<p>
	 * TODO (Arch): This doesn't leave a way to verify that a Node representing a group contains a specific member in its member value.
	 * 		We may want to consider allowing a filter to be passed here, or at least a partial attribute.
	 * 
	 * @param nodeID Contextually unique ID of Node being verified.
	 * @param attributes Iterator of {@link IAttribute} to be compared to the Node specified by nodeID. 
	 * @return {@code true} if all attribute assertions in attributes are found to be equal to those attributes in the Node specified by nodeID. {@code false} otherwise. 
	 * @throws IdASException (typically a subclass of. See below).
	 * @throws NoSuchNodeException when node specified by nodeID is not found.
	 */
	public boolean verifyNodeAttributes(String nodeID, Iterator attributes) throws IdASException;

	/**
	 * Returns the relationships this Object has to others.
	 * @return Iterator of IContextRelation objects. The set of Relationships this Object has.
	 * @throws IdASException
	 */
	public Iterator getRelationships() throws IdASException;

	/**
	 * Adds a context relationship to this context. 
	 * This call is typically followed by a call to {@link IContextRelation#addRelatedContext(URI)}
	 * Note: This operation is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()} is called.
	 * @param relationshipType The type of relationship to add.
	 * @throws IdASException
	 */
	public IContextRelation addRelationship(URI relationshipType) throws IdASException;

	/**
	 * Applies all update operations which have been called on this Context 
	 * and called on any element held by this Context 
	 * (including nodess, metadata, attributes, attribute metadata and it's value(s), and attribute values) 
	 * to a backing data store.  These updates are applied atomically when possible, but may not be 
	 * applied atomically by Context Providers which do not support this.
	 * Update operations which will be applied are any of the following:<br>
	 * {@link IContext#addRelationship(URI)}<br>
	 * {@link IContext#addNode(URI, String)}<br>
	 * {@link IContext#addNode(INode)}<br>
	 * {@link INode#remove()}<br>
	 * {@link IHasAttributes#addAttribute(IAttribute)}<br>
	 * {@link IHasAttributes#addAttribute(URI)}<br>
	 * {@link IHasAttributes#removeAttribute(URI)}<br>
	 * {@link IHasAttributes#removeAttributeValue(IAttribute)}<br>
	 * {@link IHasAttributes#rremoveAttributeValue(URI, Object)}<br>
	 * {@link IHasMetadata#addMetadata(IMetadata)}<br>
	 * {@link IHasMetadata#addMetadata(URI)}<br>
	 * {@link IAttribute#addComplexValue(URI)}<br>
	 * {@link IAttribute#addSimpleValue(URI, Object)}<br>
	 * {@link IAttribute#addValue(IAttributeValue)}<br>
	 * {@link IAttribute#addValue(URI)}<br>
	 * {@link IAttribute#remove()}<br>
	 * {@link IAttributeValue#remove()}<br>
	 * {@link ITypedValue#setData(Object)}<br>
	 * On a successful return, all updates which have been performed on this Context 
	 * or on any contained elements are said to be fully applied to the Context.  
	 * Subsequent update operations must be followed by another call to applyUpdates.
	 * On error (when an exception is thrown), the updates are said to not yet fully applied
	 * to the Context.  In this case, the caller may attempt to remedy the issue which 
	 * caused the call to applyUpdates by performing further update operations followed
	 * by another call to {@link #applyUpdates()}.  Alternately, the caller may call 
	 * {@link #cancelUpdates()} in order to cancel all updates which have not yet been applied.
	 * @throws IdASException
	 */
	void applyUpdates() throws IdASException;	

	/**
	 * Cancels all update operations which have been called on this Context 
	 * and called on any element held by this Context 
	 * (including nodes, metadata, attributes, 
	 * attribute metadata and it's value(s), and attribute values).
	 * Any updates which have already been applied (by calling {@link #applyUpdates()}) are not canceled.   
	 * @throws IdASException
	 * @see #applyUpdates()
	 */
	void cancelUpdates() throws IdASException;	

	
}
