/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.client;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.common.AuthNAnonymousMaterials;
import org.eclipse.higgins.idas.cp.xdi.XDIContextFactory;
import org.eclipse.higgins.idas.udi.IAttributeUDIMetadata;
import org.eclipse.higgins.idas.udi.IContextUDIMetadata;
import org.eclipse.higgins.idas.udi.IEntityUDIMetadata;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Literal;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Reference;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.constants.TypeConstants;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageEnvelope;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.client.XDIClient;
import org.eclipse.higgins.xdi4j.messaging.client.http.XDIHttpClient;
import org.eclipse.higgins.xdi4j.messaging.error.ErrorMessageResult;
import org.eclipse.higgins.xdi4j.signatures.Signatures;
import org.eclipse.higgins.xdi4j.xri3.XRISubSegment;
import org.eclipse.higgins.xdi4j.xri3.XRIXRef;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class IdASClient {

	private static IdASClient impl = null;

	public static final XRI3Segment XRI_AUTHNMATERIALSTYPES = new XRI3Segment("$authn$type");
	public static final XRI3Segment XRI_AUTHNMATERIALSTYPE = new XRI3Segment("$authn$type");
	public static final XRI3Segment XRI_AUTHNMATERIALS = new XRI3Segment("$authn");
	public static final XRI3Segment XRI_CONTEXTTYPES = new XRI3Segment("$is$a");
	public static final XRI3Segment XRI_CONTEXTURI = new XRI3Segment("$uri$http");
	public static final XRI3Segment XRI_ENTITYID = new XRI3Segment("=");
	public static final XRI3Segment XRI_ATTRIBUTEID = new XRI3Segment("+");

	public static final String DEFAULT_IDASPROXYENDPOINT = null;
	public static final String DEFAULT_IDASSENDER = "=sender";

	private String idasProxyEndpoint;
	private String idasSender;
	private PrivateKey privateKey;
	private XDIClient xdiClient;

	private MessageEnvelope lastMessageEnvelope;
	private MessageResult lastMessageResult;
	
	public static synchronized IdASClient getInstance() {

		if (impl == null) impl = new IdASClient();

		return(impl);
	}

	public IdASClient(String idasProxyEndpoint, String idasSender) {

		this.idasProxyEndpoint = idasProxyEndpoint;
		this.idasSender = idasSender;
		this.xdiClient = new XDIHttpClient(this.idasProxyEndpoint);
	}

	public IdASClient(String idasProxyEndpoint) {

		this(idasProxyEndpoint, DEFAULT_IDASSENDER);
	}

	public IdASClient() {

		this(DEFAULT_IDASPROXYENDPOINT, DEFAULT_IDASSENDER);
	}

	public String getIdasProxyEndpoint() {

		return(this.idasProxyEndpoint);
	}

	public void setIdasProxyEndpoint(String idasProxyEndpoint) {

		this.idasProxyEndpoint = idasProxyEndpoint;
		this.xdiClient = new XDIHttpClient(this.idasProxyEndpoint);
	}

	public String getIdasSender() {

		return(this.idasSender);
	}

	public void setIdasSender(String idasSender) {

		this.idasSender = idasSender;
	}

	public IContextUDIMetadata resolveContextMetadata(String contextUDI) throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(new XRI3Segment(this.idasSender));
		Operation operation = message.createGetOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		operationGraph.createStatement(new XRI3Segment("$"), XRI_CONTEXTTYPES);
		operationGraph.createStatement(new XRI3Segment("$"), XRI_AUTHNMATERIALSTYPES);
		operationGraph.createStatement(new XRI3Segment("$"), XRI_AUTHNMATERIALS);
		messageEnvelope.getGraph().createStatement(new XRI3Segment("$"), TypeConstants.XRI_VALUE, contextUDI);

		// send the message

		MessageResult messageResult;
		final Subject subject;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.lastMessageEnvelope = messageEnvelope;
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.lastMessageResult = messageResult;
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());

			subject = messageResult.getGraph().getSubject(new XRI3Segment("$"));
		} catch (IdASException ex) {
			
			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while talking to IdAS proxy: " + ex.getMessage(), ex);
		}

		if (subject == null) throw new IdASException("No Context UDI Metadata result received.");

		// return the Context UDI Metadata

		return(new IContextUDIMetadata() {

			public String[] getTypes() throws IdASException {

				return(extractContextTypes(subject));
			}

			public String[] getAuthNMaterialsTypes() throws IdASException {

				return(extractAuthNMaterialsTypes(subject));
			}

			public String getAuthNMaterials() throws IdASException {

				return(extractAuthNMaterials(subject));
			}

			public Map getConfiguration() throws IdASException {

				return(null);
			}
		});
	}

	public IEntityUDIMetadata resolveEntityMetadata(String entityUDI) throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(new XRI3Segment(this.idasSender));
		Operation operation = message.createGetOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		operationGraph.createStatement(new XRI3Segment("="), XRI_CONTEXTTYPES);
		operationGraph.createStatement(new XRI3Segment("="), XRI_AUTHNMATERIALSTYPES);
		operationGraph.createStatement(new XRI3Segment("="), XRI_AUTHNMATERIALS);
		operationGraph.createStatement(new XRI3Segment("="), XRI_ENTITYID);
		messageEnvelope.getGraph().createStatement(new XRI3Segment("="), TypeConstants.XRI_VALUE, entityUDI);

		// send the message

		MessageResult messageResult;
		final Subject subject;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.lastMessageEnvelope = messageEnvelope;
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.lastMessageResult = messageResult;
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());

			subject = messageResult.getGraph().getSubject(new XRI3Segment("="));
		} catch (IdASException ex) {
			
			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while talking to IdAS proxy: " + ex.getMessage(), ex);
		}

		if (subject == null) throw new IdASException("No Entity UDI Metadata result received.");

		// return the Entity UDI Metadata

		return(new IEntityUDIMetadata() {

			public IContextUDIMetadata getContextUDIMetadata() throws IdASException {

				return(new IContextUDIMetadata() {

					public String[] getTypes() throws IdASException {

						return(extractContextTypes(subject));
					}

					public String[] getAuthNMaterialsTypes() throws IdASException {

						return(extractAuthNMaterialsTypes(subject));
					}

					public String getAuthNMaterials() throws IdASException {

						return(extractAuthNMaterials(subject));
					}

					public Map getConfiguration() throws IdASException {

						return(null);
					}
				});
			}

			public String getEntityID() throws IdASException {

				return(extractEntityID(subject));
			}
		});
	}

	public IAttributeUDIMetadata resolveAttributeMetadata(String attributeUDI) throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(new XRI3Segment(this.idasSender));
		Operation operation = message.createGetOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		operationGraph.createStatement(new XRI3Segment("+"), XRI_CONTEXTTYPES);
		operationGraph.createStatement(new XRI3Segment("+"), XRI_AUTHNMATERIALSTYPES);
		operationGraph.createStatement(new XRI3Segment("+"), XRI_AUTHNMATERIALS);
		operationGraph.createStatement(new XRI3Segment("+"), XRI_ENTITYID);
		operationGraph.createStatement(new XRI3Segment("+"), XRI_ATTRIBUTEID);
		messageEnvelope.getGraph().createStatement(new XRI3Segment("+"), TypeConstants.XRI_VALUE, attributeUDI);

		// send the message

		MessageResult messageResult;
		final Subject subject;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.lastMessageEnvelope = messageEnvelope;
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.lastMessageResult = messageResult;
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());

			subject = messageResult.getGraph().getSubject(new XRI3Segment("+"));
		} catch (IdASException ex) {
			
			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while talking to IdAS proxy: " + ex.getMessage(), ex);
		}

		if (subject == null) throw new IdASException("No Attribute UDI Metadata result received.");

		// return the Attribute UDI Metadata

		return(new IAttributeUDIMetadata() {

			public IEntityUDIMetadata getEntityUDIMetadata() throws IdASException {

				return(new IEntityUDIMetadata() {

					public IContextUDIMetadata getContextUDIMetadata() throws IdASException {

						return(new IContextUDIMetadata() {

							public String[] getTypes() throws IdASException {

								return(extractContextTypes(subject));
							}

							public String[] getAuthNMaterialsTypes() throws IdASException {

								return(extractAuthNMaterialsTypes(subject));
							}

							public String getAuthNMaterials() throws IdASException {

								return(extractAuthNMaterials(subject));
							}

							public Map getConfiguration() throws IdASException {

								return(null);
							}
						});
					}

					public String getEntityID() throws IdASException {

						return(extractEntityID(subject));
					}
				});
			}

			public URI getAttributeID() throws IdASException {

				return(extractAttributeID(subject));
			}
		});
	}

	public IContext resolveContext(String contextUDI, String authNMaterialsType, String authNMaterials) throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(new XRI3Segment(this.idasSender));
		Operation operation = message.createGetOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		operationGraph.createStatement(new XRI3Segment("$"), XRI_CONTEXTURI);
		messageEnvelope.getGraph().createStatement(new XRI3Segment("$"), TypeConstants.XRI_VALUE, contextUDI);
		if (authNMaterialsType != null) messageEnvelope.getGraph().createStatement(new XRI3Segment("$"), XRI_AUTHNMATERIALSTYPE, authNMaterialsType);
		if (authNMaterials != null) messageEnvelope.getGraph().createStatement(new XRI3Segment("$"), XRI_AUTHNMATERIALS, authNMaterials);

		// send the message

		MessageResult messageResult;
		final Subject subject;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.lastMessageEnvelope = messageEnvelope;
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.lastMessageResult = messageResult;
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());

			subject = messageResult.getGraph().getSubject(new XRI3Segment("$"));
		} catch (IdASException ex) {
			
			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while talking to IdAS proxy: " + ex.getMessage(), ex);
		}

		if (subject == null) throw new IdASException("No Context result received.");

		// return the XDI Context

		XDIContextFactory contextFactory = new XDIContextFactory();
		IContext context = contextFactory.createContext(extractContextUri(subject));
		context.open(new AuthNAnonymousMaterials());

		return(context);
	}

	public IEntity resolveEntity(String entityUDI, String authNMaterialsType, String authNMaterials) throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(new XRI3Segment(this.idasSender));
		Operation operation = message.createGetOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		operationGraph.createStatement(new XRI3Segment("="), XRI_CONTEXTURI);
		operationGraph.createStatement(new XRI3Segment("="), XRI_ENTITYID);
		messageEnvelope.getGraph().createStatement(new XRI3Segment("="), TypeConstants.XRI_VALUE, entityUDI);
		if (authNMaterialsType != null) messageEnvelope.getGraph().createStatement(new XRI3Segment("="), XRI_AUTHNMATERIALSTYPE, authNMaterialsType);
		if (authNMaterials != null) messageEnvelope.getGraph().createStatement(new XRI3Segment("="), XRI_AUTHNMATERIALS, authNMaterials);

		// send the message

		MessageResult messageResult;
		final Subject subject;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.lastMessageEnvelope = messageEnvelope;
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.lastMessageResult = messageResult;
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());

			subject = messageResult.getGraph().getSubject(new XRI3Segment("="));
		} catch (IdASException ex) {
			
			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while talking to IdAS proxy: " + ex.getMessage(), ex);
		}

		if (subject == null) throw new IdASException("No Entity result received.");

		// return the XDI Entity

		XDIContextFactory contextFactory = new XDIContextFactory();
		IContext context = contextFactory.createContext(extractContextUri(subject));
		context.open(new AuthNAnonymousMaterials());
		IEntity entity = context.getEntity(extractEntityID(subject));

		return(entity);
	}

	public IAttribute resolveAttribute(String attributeUDI, String authNMaterialsType, String authNMaterials) throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(new XRI3Segment(this.idasSender));
		Operation operation = message.createGetOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		operationGraph.createStatement(new XRI3Segment("+"), XRI_CONTEXTURI);
		operationGraph.createStatement(new XRI3Segment("+"), XRI_ENTITYID);
		operationGraph.createStatement(new XRI3Segment("+"), XRI_ATTRIBUTEID);
		messageEnvelope.getGraph().createStatement(new XRI3Segment("+"), TypeConstants.XRI_VALUE, attributeUDI);
		if (authNMaterialsType != null) messageEnvelope.getGraph().createStatement(new XRI3Segment("+"), XRI_AUTHNMATERIALSTYPE, authNMaterialsType);
		if (authNMaterials != null) messageEnvelope.getGraph().createStatement(new XRI3Segment("+"), XRI_AUTHNMATERIALS, authNMaterials);

		// send the message

		MessageResult messageResult;
		final Subject subject;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.lastMessageEnvelope = messageEnvelope;
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.lastMessageResult = messageResult;
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());

			subject = messageResult.getGraph().getSubject(new XRI3Segment("+"));
		} catch (IdASException ex) {
			
			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while talking to IdAS proxy: " + ex.getMessage(), ex);
		}

		if (subject == null) throw new IdASException("No Attribute result received.");

		// return the XDI Attribute

		XDIContextFactory contextFactory = new XDIContextFactory();
		IContext context = contextFactory.createContext(extractContextUri(subject));
		context.open(new AuthNAnonymousMaterials());
		IEntity entity = context.getEntity(extractEntityID(subject));
		IAttribute attribute = entity.getAttribute(extractAttributeID(subject));

		return(attribute);
	}

	/*
	 * For debugging purposes we always remember the last XDI message envelope and
	 * XDI message result
	 */
	
	public MessageEnvelope getLastMessageEnvelope() {
		
		return(this.lastMessageEnvelope);
	}
	
	public MessageResult getLastMessageResult() {
		
		return(this.lastMessageResult);
	}

	/*
	 * Helper methods for extracting data from an XDI message result 
	 */

	private static String[] extractContextTypes(Subject subject) {

		Predicate predicate = subject.getPredicate(XRI_CONTEXTTYPES);
		if (predicate == null) return(null);

		Iterator references = predicate.getReferences();

		List contextTypesList = new ArrayList();
		while (references.hasNext()) {

			Reference reference = (Reference) references.next();

			String contextType = reference.getReferenceXri().toString();
			contextTypesList.add(contextType);
		}

		String[] contextTypes = new String[contextTypesList.size()];
		for (int i=0; i<contextTypesList.size(); i++) contextTypes[i] = (String) contextTypesList.get(i);
		return(contextTypes);
	}

	private static String[] extractAuthNMaterialsTypes(Subject subject) {

		Predicate predicate = subject.getPredicate(XRI_AUTHNMATERIALSTYPES);
		if (predicate == null) return(null);

		Iterator references = predicate.getReferences();

		List authNMaterialsTypesList = new ArrayList();
		while (references.hasNext()) {

			Reference reference = (Reference) references.next();

			XRISubSegment subSegment = reference.getReferenceXri().getFirstSubSegment();
			if (subSegment == null) continue;
			XRIXRef xref = subSegment.getXRef();
			if (xref == null) continue;

			if (xref.getXRIReference() != null) {

				String authNMaterialsType = xref.getXRIReference().toString();
				authNMaterialsTypesList.add(authNMaterialsType);
			} else if (xref.getIRI() != null) {

				String authNMaterialsType = xref.getIRI();
				authNMaterialsTypesList.add(authNMaterialsType);
			} else {

				continue;
			}
		}

		String[] authNMaterialsTypes = new String[authNMaterialsTypesList.size()];
		for (int i=0; i<authNMaterialsTypesList.size(); i++) authNMaterialsTypes[i] = (String) authNMaterialsTypesList.get(i);
		return(authNMaterialsTypes);
	}

	private static String extractAuthNMaterials(Subject subject) {

		Predicate predicate = subject.getPredicate(XRI_AUTHNMATERIALS);
		if (predicate == null) return(null);

		Literal literal = predicate.getLiteral();
		if (literal == null) return(null);

		return(literal.getData());
	}

	private static String extractEntityID(Subject subject) {

		Predicate predicate = subject.getPredicate(XRI_ENTITYID);
		if (predicate == null) return(null);

		Reference reference = predicate.getReference();
		if (reference == null) return(null);

		return(IdASUtil.xriToEntityID(reference.getReferenceXri()));
	}

	private static URI extractAttributeID(Subject subject) {

		Predicate predicate = subject.getPredicate(XRI_ATTRIBUTEID);
		if (predicate == null) return(null);

		Reference reference = predicate.getReference();
		if (reference == null) return(null);

		return(IdASUtil.xriToAttrID(reference.getReferenceXri()));
	}

	private static URL extractContextUri(Subject subject) {

		Predicate predicate = subject.getPredicate(XRI_CONTEXTURI);
		if (predicate == null) return(null);

		Literal literal = predicate.getLiteral();
		if (literal == null) return(null);

		try {

			return(new URL(literal.getData()));
		} catch (MalformedURLException ex) {

			throw new RuntimeException();
		}
	}
}
