/*******************************************************************************
 * Copyright (c) 2009 Azigo, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.common;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;

import org.eclipse.higgins.idas.api.IAuthNMaterials;
import org.eclipse.higgins.idas.api.IdASException;

/**
 * This class can serialize and deserialize IdAS Authentication Materials
 * to/from strings.
 * 
 * For deserialization the Authentication Material Type must be known
 * @see org.eclipse.higgins.idas.api.IAuthNMaterials
 */
public class AuthNMaterialsSerializer {

	public static String serialize(IAuthNMaterials authNMaterials) throws IdASException {

		StringBuffer buffer = new StringBuffer();

		if (authNMaterials instanceof AuthNNamePasswordMaterials) {

			try {

				buffer.append(URLEncoder.encode(((AuthNNamePasswordMaterials) authNMaterials).getUsername(), "UTF-8"));
				buffer.append(" ");
				buffer.append(URLEncoder.encode(((AuthNNamePasswordMaterials) authNMaterials).getPassword(), "UTF-8"));
			} catch (UnsupportedEncodingException ex) {

				throw new IdASException(ex);
			}

			return buffer.toString();
		} else if (authNMaterials instanceof AuthNAnonymousMaterials) {

			return "";
		} else if (authNMaterials instanceof AuthNLeastPrivilegedUserMaterials) {

			return "";
		}

		throw new IdASException("Unknown Authentication Materials Type");
	}

	public static IAuthNMaterials deserialize(String authNMaterialsType, String authNMaterialsStr) throws IdASException {

		if (authNMaterialsType == null) {

			return new AuthNAnonymousMaterials();
		} else if (authNMaterialsType.equals(IAuthNMaterials.AUTHNMATERIALS_TYPE_USERNAMEPASSWORD)) {

			String[] fields = authNMaterialsStr.split(" ");

			if (fields.length == 2) {

				String username;
				String password;

				try {

					username = URLDecoder.decode(fields[0], "UTF-8");
					password = URLDecoder.decode(fields[1], "UTF-8");
				} catch (UnsupportedEncodingException ex) {

					throw new IdASException(ex);
				}

				return new AuthNNamePasswordMaterials(username, password);
			} else {

				throw new IdASException("Invalid encoding of Authentication Materials Type: " + authNMaterialsType);
			}
		} else if (authNMaterialsType.equals(IAuthNMaterials.AUTHNMATERIALS_TYPE_ANONYMOUS)) {

			return new AuthNAnonymousMaterials();
		} else if (authNMaterialsType.equals(IAuthNMaterials.AUTHNMATERIALS_TYPE_LEASTPRIVILEGED)) {

			return new AuthNLeastPrivilegedUserMaterials();
		}

		throw new IdASException("Unknown Authentication Materials Type: " + authNMaterialsType);
	}
}
