/*******************************************************************************
 * Copyright (c) 2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut
 *    Markus Sabadello - added JavaBean mode without IContext
 *******************************************************************************/
package org.eclipse.higgins.idas.common;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAuthNAttributesMaterials;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.MultipleValuesExistException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;

/**
 * Simple username and password authorization materials.
 * 
 * The username and password can be held by this class in a JavaBean-style way
 * (in this case, use the constructor WITHOUT IContext)
 * 
 * OR
 * 
 * The username and password can be held inside an IContext as attributes.
 * (in this case, use the constructor WITH IContext)
 */
public class AuthNNamePasswordMaterials implements IAuthNAttributesMaterials {

	private static final long serialVersionUID = 5824707419631574507L;

	private String _username;
	private String _password;

	private IAuthNAttributesMaterials _impl;
	private IAttribute _managedAttr;

	public static final URI ATTR_TYPE_USERNAME = URI.create(IValue.BASE_OWL_URI + "userName");
	public static final URI ATTR_MANAGED_URI = URI.create(IValue.BASE_OWL_URI + "password");

	/**
	 * Constructs an AuthNNamePasswordMaterials by taking the username and password as String objects.
	 * @param context needed in order to build attributes from username and password
	 * @param username name of identity
	 * @param password password of identity
	 * @throws IdASException
	 */
	public AuthNNamePasswordMaterials(
			IContext context, 
			String username, 
			String password) throws IdASException
			{
		_impl = context.buildAuthNAttributesMaterials();
		IAttribute attr; 

		attr = addAttribute(ATTR_TYPE_USERNAME);
		attr.addSimpleValue(ITypedValue.STRING_TYPE_URI, username);

		if (password != null)
		{
			_managedAttr = addAttribute(ATTR_MANAGED_URI);
			_managedAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, password);
		}
			}

	/**
	 * Constructs an AuthNNamePasswordMaterials by taking the username and password as String objects.
	 * @param username name of identity
	 * @param password password of identity
	 * @throws IdASException
	 */
	public AuthNNamePasswordMaterials(
			String username, 
			String password)
	{
		_username = username;
		_password = password;
	}

	/**
	 * Returns the username.  IdAS Context Providers may also wish to access the username in IAttribute form as follows:
	 * <pre> IAttribute usernameAttr = thisMaterials.getAttribute(new URI(thisMaterials.ATTR_TYPE_USERNAME));</pre>
	 * @return The username is String form
	 * @throws IdASException
	 */
	public String getUsername() {

		if (_impl != null) {

			try {

				IAttribute attr = getAttribute(ATTR_TYPE_USERNAME);
				Iterator iter = attr.getValues();
				String ret = null;
				if (iter.hasNext()) {
					IValue val = (IValue) iter.next();
					if (val.isSimple()) {
						ret =((ISimpleValue)val).getLexical();
					} else {
						throw new IdASException("Simple value expected, got complex value: " + val.toString());
					}
					if (iter.hasNext()) {
						throw new MultipleValuesExistException("Single value expected, got multiple");
					}
				}
				if (ret == null) {
					throw new IdASException("No " + ATTR_TYPE_USERNAME + " Attribute present");
				}
				return ret;
			} catch (IdASException ex) {

				throw new RuntimeException(ex);
			}
		} else {

			return _username;
		}
	}

	/**
	 * Returns the password.  IdAS Context Providers may also wish to access the password in IAttribute form as follows:
	 * <pre> IAttribute passwordAttr = thisMaterials.getAttribute(new URI(thisMaterials.ATTR_TYPE_PASSWORD));</pre>
	 * @return The password is String form
	 * @throws IdASException
	 */
	public String getPassword() {

		if (_impl != null) {

			try {

				IAttribute attr = getAttribute(ATTR_MANAGED_URI);
				Iterator iter = attr.getValues();
				String ret = null;
				if (iter.hasNext()) {
					IValue val = (IValue) iter.next();
					if (val.isSimple()) {
						ret =((ISimpleValue)val).getLexical();
					} else {
						throw new IdASException("Simple value expected, got complex value: " + val.toString());
					}
					if (iter.hasNext()) {
						throw new MultipleValuesExistException("Single value expected, got multiple");
					}
				}
				if (ret == null) {
					throw new IdASException("No " + ATTR_MANAGED_URI + " attribute present");
				}
				return ret;
			} catch (IdASException ex) {

				throw new RuntimeException(ex);
			}
		} else {

			return _password;
		}
	}

	public void setUsername(String username) {
		
		if (_impl != null) {
			
		} else {
			
			_username = username;
		}
	}

	public String toString() {

		return "Username: " + this.getUsername() + ", Password: XXXXXX";
	}

	public int hashCode() {

		int hashCode = 1;
		if (this.getUsername() != null) hashCode += this.getUsername().hashCode() * 31;
		if (this.getPassword() != null) hashCode += this.getPassword().hashCode() * 31;

		return hashCode;
	}

	public boolean equals(Object object) {

		if (object == this) return true;
		if (! (object instanceof AuthNNamePasswordMaterials)) return false;

		AuthNNamePasswordMaterials other = (AuthNNamePasswordMaterials) object;

		if (this.getUsername() == null && other.getUsername() != null) return false;
		if (this.getUsername() != null && other.getUsername() == null) return false;

		if (this.getPassword() == null && other.getPassword() != null) return false;
		if (this.getPassword() != null && other.getPassword() == null) return false;

		return this.getUsername().equals(other.getUsername()) && this.getPassword().equals(other.getPassword());
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		return _impl.addAttribute(attrID);
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		return _impl.addAttribute(copyFrom);
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		return _impl.getAttribute(attrID);
	}

	public Iterator getAttributes() throws IdASException {
		return _impl.getAttributes();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {
		return _impl.getSingleValuedAttribute(attrID);
	}

	public void removeAttribute(URI attrID)  throws IdASException{
		_impl.removeAttribute(attrID);
	}

	public void removeAttributeValue(URI attrID, Object value)  throws IdASException{
		_impl.removeAttributeValue(attrID, value);
	}

	public void removeAttributeValue(IAttribute attr)  throws IdASException{
		_impl.removeAttributeValue(attr);
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		return _impl.equals(attributes);
	}

	public IAttribute getManagedAttr() {
		return _managedAttr;
	}

	public URI getManagedAttrID() {
		return ATTR_MANAGED_URI;
	}

	public IEntity addAttributeValue(URI attrType, String entityId) throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}
}
