/*******************************************************************************
 * Copyright (c) 2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim
 *******************************************************************************/
package org.eclipse.higgins.idas.common;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IAuthNAttributesMaterials;
import org.eclipse.higgins.idas.api.IAuthNMaterials;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;


/**
 * Authorization materials comprised of privatepersonalidentifier, publicKey modulus, and publicKey exponent which may be used in 
 * {@link IContext#open(Object)}.
 * 
 * This class is intended to be used when the IdAS consumer is servicing a CardSpace request for secure token which uses a self-issued card as credentials. See Section 5.6 of <a href=http://download.microsoft.com/download/6/c/3/6c3c2ba2-e5f0-4fe3-be7f-c5dcb86af6de/infocard-guide-beta2-published.pdf>A Guide to Integrating with InfoCard v1.0</a>
 *
 */
public class AuthNSelfIssuedMaterials extends AuthNDigestMaterials implements IAuthNMaterials {

	private static final long serialVersionUID = 318420137164994318L;

	private IAuthNAttributesMaterials _impl;
	private ISimpleValue _ppidAttr;
	private ISimpleValue _modAttr;
	private ISimpleValue _expAttr;
	private IAttribute _managedAttr;
	/**
	 * Attribute URI for the attribute used to manage the secrets for this type of materials
	 */
	public static final URI MANAGED_ATTR_URI = URI.create(IValue.BASE_OWL_URI + "cardKeyHash");
	public static final URI MANAGED_ATTR_TYPE_URI = URI.create(IValue.BASE_OWL_URI + "cardKeyHashType");
	/**
	 */
	public static final URI ATTR_TYPE_PPID = URI.create("http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier");
	/**
	 */
	public static final URI ATTR_TYPE_PUBLIC_KEY_MODULUS = URI.create(IValue.BASE_OWL_URI + "publicKeyModulus");
	/**
	 */
	public static final URI ATTR_TYPE_PUBLIC_KEY_EXPONENT = URI.create(IValue.BASE_OWL_URI + "publicKeyExponent");

	/**
	 * Constructs an AuthNSelfIssuedMaterials by taking the ppid and publicKey as <code>byte []</code> objects.
	 * @param ppid private personal identifier of the identity
	 * @param publicKeyMod public key modulus for the identity
	 * @param publicKeyExp public key exponent for the identity
	 * @throws IdASException
	 */
	public AuthNSelfIssuedMaterials(
		IContext context,
		byte[] ppid, 
		byte[] publicKeyMod, 
		byte[] publicKeyExp) throws IdASException
	{
		_impl = context.buildAuthNAttributesMaterials();
		IAttribute attrTemp; 

		_managedAttr = addAttribute(MANAGED_ATTR_URI);
		IEntity val = _managedAttr.addComplexValue(MANAGED_ATTR_TYPE_URI);

		attrTemp = val.addAttribute(ATTR_TYPE_PPID);
		_ppidAttr = attrTemp.addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI, ppid);

		attrTemp = val.addAttribute(ATTR_TYPE_PUBLIC_KEY_MODULUS);
		_modAttr = attrTemp.addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI, publicKeyMod);

		attrTemp = val.addAttribute(ATTR_TYPE_PUBLIC_KEY_EXPONENT);
		_expAttr = attrTemp.addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI, publicKeyExp);
	}
	
	/**
	 * Constructs an AuthNSelfIssuedMaterials by taking the ppid and publicKey as <code>byte []</code> objects.
	 * @param b64PPID Base64-encoded private personal identifier of the identity
	 * @param b64PublicKeyMod Base64-encoded public key modulus for the identity
	 * @param b64PublicKeyExp Base64-encoded public key exponent for the identity
	 * @throws IdASException
	 */
	public AuthNSelfIssuedMaterials(
		IContext context,
		String b64PPID, 
		String b64PublicKeyMod, 
		String b64PublicKeyExp) throws IdASException 
	{
		_impl = context.buildAuthNAttributesMaterials();
		IAttribute attrTemp; 

		_managedAttr = addAttribute(MANAGED_ATTR_URI);
		IEntity val = _managedAttr.addComplexValue(MANAGED_ATTR_TYPE_URI);

		attrTemp = val.addAttribute(ATTR_TYPE_PPID);
		_ppidAttr = attrTemp.addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI, b64PPID);

		attrTemp = val.addAttribute(ATTR_TYPE_PUBLIC_KEY_MODULUS);
		_modAttr = attrTemp.addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI, b64PublicKeyMod);

		attrTemp = val.addAttribute(ATTR_TYPE_PUBLIC_KEY_EXPONENT);
		_expAttr = attrTemp.addSimpleValue(ITypedValue.BASE64BINARY_TYPE_URI, b64PublicKeyExp);
	}
	
	/**
	 * Returns the private personal identifier (ppid).
	 * @return The ppid in <code>byte []</code> form
	 * @throws IdASException
	 */
	public byte[] getPPIDBytes() throws IdASException {
		return ((ByteBuffer)_ppidAttr.getData()).array();
	}
	
	/**
	 * Returns the public key modulus.  IdAS Context Providers may also wish to access the public key in IAttribute form as follows:
	 * <pre> IAttribute publicKeyAttr = thisMaterials.getAttribute(new URI(thisMaterials.ATTR_TYPE_PUBLIC_KEY_MODULUS));</pre>
	 * @return The public key in <code>byte []</code> form
	 * @throws IdASException
	 */
	public byte[] getPublicKeyModBytes()  throws IdASException {
		return ((ByteBuffer)_modAttr.getData()).array();
	}

	/**
	 * Returns the public key exponent.  IdAS Context Providers may also wish to access the public key in IAttribute form as follows:
	 * <pre> IAttribute publicKeyAttr = thisMaterials.getAttribute(new URI(thisMaterials.ATTR_TYPE_PUBLIC_KEY_EXPONENT));</pre>
	 * @return The public key in <code>byte []</code> form
	 * @throws IdASException
	 */
	public byte[] getPublicKeyExpBytes()  throws IdASException {
		return ((ByteBuffer)_expAttr.getData()).array();
	}

	/**
	 * Returns the private personal identifier (ppid).  IdAS Context Providers may also wish to access the ppid in IAttribute form as follows:
	 * <pre> IAttribute ppidAttr = thisMaterials.getAttribute(new URI(thisMaterials.ATTR_TYPE_PPID));</pre>
	 * @return The ppid in {@link String} form
	 * @throws IdASException
	 */
	public String getPPIDString() throws IdASException {
		return _ppidAttr.getLexical();
	}
	
	/**
	 * Returns the public key modulus.  IdAS Context Providers may also wish to access the public key in IAttribute form as follows:
	 * <pre> IAttribute publicKeyAttr = thisMaterials.getAttribute(new URI(thisMaterials.ATTR_TYPE_PUBLIC_KEY_MODULUS));</pre>
	 * @return The public key in {@link String} form
	 * @throws IdASException
	 */
	public String getPublicKeyModString()  throws IdASException {
		return _modAttr.getLexical();
	}

	/**
	 * Returns the public key exponent.  IdAS Context Providers may also wish to access the public key in IAttribute form as follows:
	 * <pre> IAttribute publicKeyAttr = thisMaterials.getAttribute(new URI(thisMaterials.ATTR_TYPE_PUBLIC_KEY_EXPONENT));</pre>
	 * @return The public key in {@link String} form
	 * @throws IdASException
	 */
	public String getPublicKeyExpString()  throws IdASException {
		return _expAttr.getLexical();
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		return _impl.addAttribute(attrID);
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		return _impl.addAttribute(copyFrom);
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		return _impl.getAttribute(attrID);
	}

	public Iterator getAttributes() throws IdASException {
		return _impl.getAttributes();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {
		return _impl.getSingleValuedAttribute(attrID);
	}

	public void removeAttribute(URI attrID) throws IdASException{
		_impl.removeAttribute(attrID);
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException{
		_impl.removeAttributeValue(attrID, value);
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		_impl.removeAttributeValue(attr);
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		return _impl.equals(attributes);
	}

	public IAttribute getManagedAttr() {
		return _managedAttr;
	}

	public URI getManagedAttrID() {
		return MANAGED_ATTR_URI;
	}
	
	public final byte[] getSHA1Digest() throws IdASException {
		MessageDigest md;
		ByteArrayOutputStream cardKey = new ByteArrayOutputStream();

		try {
			cardKey.write(getPPIDBytes());
			cardKey.write(getPublicKeyModBytes());
			cardKey.write(getPublicKeyExpBytes());
			md = MessageDigest.getInstance("SHA-1");

		} catch (IOException e) {
			throw new IdASException(e);
		} catch (NoSuchAlgorithmException e) {
			throw new IdASException(e);
		}

		return md.digest(cardKey.toByteArray());
	}

	public IEntity addAttributeValue(URI attrType, String entityId) throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

}
