/*******************************************************************************
 * Copyright (c) 2009 Azigo, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.common;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;

/**
 * This class can serialize and deserialize IdAS data types
 * 
 * For deserialization the data type must be known
 * @see org.eclipse.higgins.idas.api.ITypedValue
 */
public class SimpleValueSerializer {

	public static String serialize(Object data) throws IdASException {

		// try to serialize the data

		try {

			if (data instanceof String) {

				return(new String((String) data));
			} else if (data instanceof Boolean) {

				return(((Boolean) data).toString());
			} else if (data instanceof BigInteger) {

				return(((BigInteger) data).toString());
			} else if (data instanceof BigDecimal) {

				return(((BigDecimal) data).toString());
			} else if (data instanceof Float) {

				return(((Float) data).toString());
			} else if (data instanceof Double) {

				return(((Double) data).toString());
			} else if (data instanceof Long) {

				return(((Long) data).toString());
			} else if (data instanceof Integer) {

				return(((Integer) data).toString());
			} else if (data instanceof Short) {

				return(((Short) data).toString());
			} else if (data instanceof Byte) {

				return(((Byte) data).toString());
			} else if (data instanceof ByteBuffer) {

				StringBuffer stringBuffer = new StringBuffer();
				byte[] array = ((ByteBuffer) data).array();

				for (int i=0; i<array.length; i++) {

					if (i > 0) stringBuffer.append(" ");
					stringBuffer.append(Byte.toString(array[i]));
				}
				return(stringBuffer.toString());
			} else if (data instanceof Date) {

				return(Long.toString(((Date) data).getTime()));
			} else if (data instanceof URI) {

				return(((URI) data).toString());
			}

		} catch (Exception ex) {

			throw new IdASException("Problem while serializing the data: " + ex.getMessage(), ex);
		}

		throw new IdASException("Cannot serialize the Java object of class " + data.getClass().getName());
	}

	public static Object deserialize(String dataStr, String typeUriStr) throws IdASException {

		// find out what Java class to use for this IdAS data type

		String className = null;

		for (int i=0; i<ITypedValue.typeMap.length; i++) {

			if (ITypedValue.typeMap[i][0].equals(typeUriStr)) {

				className = ITypedValue.typeMap[i][1];
				break;
			}
		}

		if (className == null) throw new IdASException(typeUriStr + " is not a valid IdAS data type.");

		// try to deserialize the data

		try {

			if (className.equals(String.class.getName())) {

				return(new String(dataStr));
			} else if (className.equals(Boolean.class.getName())) {

				return(new Boolean(dataStr));
			} else if (className.equals(BigDecimal.class.getName())) {

				return(new BigDecimal(dataStr));
			} else if (className.equals(BigInteger.class.getName())) {

				return(new BigInteger(dataStr));
			} else if (className.equals(Float.class.getName())) {

				return(new Float(dataStr));
			} else if (className.equals(Double.class.getName())) {

				return(new Double(dataStr));
			} else if (className.equals(Long.class.getName())) {

				return(new Long(dataStr));
			} else if (className.equals(Integer.class.getName())) {

				return(new Integer(dataStr));
			} else if (className.equals(Short.class.getName())) {

				return(new Short(dataStr));
			} else if (className.equals(Byte.class.getName())) {

				return(new Byte(dataStr));
			} else if (className.equals(ByteBuffer.class.getName())) {

				StringTokenizer stringTokenizer = new StringTokenizer(dataStr, " ");
				List byteList = new ArrayList();
				while (stringTokenizer.hasMoreElements()) byteList.add(new Byte(stringTokenizer.nextToken()));
				byte[] array = new byte[byteList.size()];
				for (int i=0; i<byteList.size(); i++) array[i] = ((Byte) byteList.get(i)).byteValue();
				return(ByteBuffer.wrap(array));
			} else if (className.equals(Date.class.getName())) {

				return(new Date(Long.parseLong(dataStr)));
			} else if (className.equals(URI.class.getName())) {

				return(new URI(dataStr));
			}
		} catch (Exception ex) {

			throw new IdASException("Problem while deserializing data: " + ex.getMessage(), ex);
		}

		throw new IdASException("Cannot deserialize the IdAS data type " + typeUriStr + " to a Java object of class " + className);
	}
}
