/**
 * Copyright (c) 2006, 2008 Oracle and/or its affiliates. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Mohamad Raja - Initial implementation
 */
package org.eclipse.higgins.idas.cp.google;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.ContextNotOpenException;
import org.eclipse.higgins.idas.api.ContextOpenException;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IAuthNMaterials;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.idas.spi.AttributeNotification;
import org.eclipse.higgins.idas.spi.ValueNotification;
import org.eclipse.higgins.idas.spi.BasicContext;
import org.eclipse.higgins.idas.spi.BasicFilter;
import org.eclipse.higgins.idas.spi.BasicFilterAssertion;
import org.eclipse.higgins.idas.spi.BasicFilterAttributeAssertion;
import org.eclipse.higgins.idas.spi.EntityNotification;

import com.google.gdata.client.Query;
import com.google.gdata.client.contacts.ContactsService;
import com.google.gdata.data.DateTime;
import com.google.gdata.data.PlainTextConstruct;
import com.google.gdata.data.TextConstruct;
import com.google.gdata.data.contacts.ContactEntry;
import com.google.gdata.data.contacts.ContactFeed;
import com.google.gdata.data.extensions.Email;
import com.google.gdata.data.extensions.OrgName;
import com.google.gdata.data.extensions.OrgTitle;
import com.google.gdata.data.extensions.Organization;

public class GDataContext extends BasicContext
{
    private static final Log log = LogFactory.getLog(GDataContext.class.getName());

    public static URI PROP_CONTENT = URI.create("content"); 
    public static URI PROP_TITLE = URI.create("title");
    public static URI PROP_EMAIL = URI.create("email");
    public static URI PROP_IM = URI.create("im");
    public static URI PROP_PHONE = URI.create("phoneNumber");
    public static URI PROP_POSTAL_ADDRESS = URI.create("postalAddress");

    public static URI PROP_ORGANIZATION = URI.create("organization");
    public static URI PROP_ORG_NAME = URI.create("orgName");
    public static URI PROP_ORG_TITLE = URI.create("orgTitle");
    public static URI PROP_TEXT = URI.create("text");

    public static URI PROP_ADDRESS = URI.create("address");
    public static URI PROP_REL = URI.create("rel");
    public static URI PROP_LABEL = URI.create("label");
    public static URI PROP_PRIMARY = URI.create("primary");

    public static URI PARAM_MAX_RESULTS = URI.create("maxResults");
    public static URI PARAM_START_INDEX = URI.create("startIndex");
    public static URI PARAM_UPDATED_MIN = URI.create("updatedMin");
    
    private final IContextId m_contextId;

    /**
     * Service used to communicate with contacts feed.
     */
    private final ContactsService m_service;

    /**
     * Base URL for the feed
     */
    private URL m_feedURL;

    /**
     * Projection used for the feed
     */
    private final String m_projection;
    private final Map<?,?> m_connectionSettings;

    public GDataContext(IContextId contextId) throws IdASException
    {
        super();
        m_contextId = contextId;
        m_service = new ContactsService("Higgins Google Contacts CP");

        Map<?,?> contextSettings = contextId.getConfiguration();
        m_connectionSettings = (Map<?,?>)contextSettings.get("Connection");
        if (m_connectionSettings == null)
            throw new IdASException("No connection specified for Google Contacts");

        m_projection = (String)m_connectionSettings.get("Projection");
    }

    @Override
    public String open(IAuthNMaterials identity) throws IdASException
    {
        if (isOpen(null))
            throw new ContextOpenException();

        if (identity instanceof AuthNNamePasswordMaterials)
        {
            String userName = ((AuthNNamePasswordMaterials)identity).getUsername();
            String password = ((AuthNNamePasswordMaterials)identity).getPassword();
            try
            {
                m_feedURL = new URL(m_connectionSettings.get("FeedURL") 
                    + "contacts/" + userName + "/" + m_projection);
                m_service.setUserCredentials(userName, password);
                setOpen(true);
            }
            catch (Exception e)
            {
                throw new IdASException(e);
            }
            return userName;
        }
        throw new IdASException("Unsupported authentication material");
    }

    @Override
    public IEntity addEntity(URI entityType, String entityID) throws IdASException
    {
        if (!isOpen(null))
            throw new ContextNotOpenException();

        GDataEntity entity = new GDataEntity(this, entityType, entityID);
        updateNotification(new EntityNotification(entity, 
            EntityNotification.UPDATE_ADD, null));
        return entity;
    }

    @Override
    public Iterator<IEntity> getEntities(IFilter filter, Iterator attrSelectionList)
        throws IdASException
    {
        if (!isOpen(null))
            throw new ContextNotOpenException();

        ContactFeed resultFeed = null;
        List<IEntity> contactList = new LinkedList<IEntity>();
        try
        {
            if (filter != null)
            {
                Query query = new Query(m_feedURL);
                buildQuery((BasicFilter)filter, query);
                resultFeed = m_service.query(query, ContactFeed.class);
            }
            else
                resultFeed = m_service.getFeed(m_feedURL, ContactFeed.class);
        }
        catch (Exception ex)
        {
            throw new IdASException(ex);
        }

        for (ContactEntry contact : resultFeed.getEntries())
            contactList.add(new GDataEntity(this, null, contact.getId(), contact));
        return contactList.iterator();
    }
    
    public IEntity getEntity(String entityID) throws IdASException {
    	
    	Iterator<IEntity> entities = this.getEntities((IFilter)null);
    	while (entities.hasNext()) {
    		
    		IEntity entity = entities.next();
    		if (entity.getEntityID().equals(entityID)) return entity;
    	}
    	throw new NoSuchEntityException();
    }

    private static void buildQuery(BasicFilter basicFilter, Query query)
        throws IdASException
    {
        BasicFilterAssertion assertion = 
            (BasicFilterAssertion)basicFilter.getAssertion();

        if (assertion == null)
        {
            String operator = basicFilter.getOperator();
            if (!operator.equals(IFilter.OP_AND))
                throw new NotImplementedException("Unsupported operator:" + operator);

            Iterator<IFilter> filters = basicFilter.getFilters();
            while (filters.hasNext())
            {
                IFilter filter = filters.next();
                buildQuery((BasicFilter)filter, query);
            }
        }
        else
        {
            URI attrURI = assertion.getID();
            String comparator = assertion.getComparator();
            if (!comparator.equals(IFilterAttributeAssertion.COMP_ATTR_EQ))
                throw new NotImplementedException("Unsupported comparator:" + comparator);
    
            if (assertion instanceof BasicFilterAttributeAssertion)
            {
                BasicFilterAttributeAssertion attributeAssertion = 
                    (BasicFilterAttributeAssertion)assertion;

                IValue assertionValue = attributeAssertion.getAssertionValue();
                if (!assertionValue.isSimple())
                    throw new NotImplementedException("Complex value not supported");
    
                String value = ((ISimpleValue)assertionValue).getCanonical();
                if (attrURI.equals(PARAM_MAX_RESULTS))
                    query.setMaxResults(Integer.parseInt(value));
                else if (attrURI.equals(PARAM_START_INDEX))
                    query.setStartIndex(Integer.parseInt(value));
                else if (attrURI.equals(PARAM_UPDATED_MIN))
                    query.setUpdatedMin(DateTime.parseDateTime(value));
                else
                    query.setStringCustomParameter(attrURI.toString(), value);
            }
        }
    }

    @Override
    public void applyUpdates() throws IdASException
    {
        if (!isOpen(null))
            throw new ContextNotOpenException();

        Map<String, List<EntityNotification>> updateList = super.getUpdateList();
        try
        {
            for (Map.Entry<String, List<EntityNotification>> entrySet : updateList.entrySet())
            {
                log.debug("Updating entry [" + entrySet.getKey() + "]");
                ContactEntry contact = null;
                for (EntityNotification notification : entrySet.getValue())
                {
                    String action = notification.getAction();
                    if (action.equals(EntityNotification.UPDATE_ADD))
                    {
                        GDataEntity entity = (GDataEntity)notification.getEntity();
                        entity.setContactEntry(addContact(entity));
                        break;
                    }
                    else if (action.equals(EntityNotification.UPDATE_REMOVE))
                    {
                        removeContact(notification.getEntity());
                    }
                    else if (action.equals(EntityNotification.UPDATE_ATTR_NOTIFY))
                    {
                        if (contact == null)
                            contact = getContact(notification.getEntity());
                        updateContactWithNotification(notification, contact);
                    }
                    else
                    {
                        throw new IdASException("Invalid Entity Notification Action: "
                            + action);
                    }
                }

                if (contact != null)
                {
                    GDataEntity entity = 
                        (GDataEntity)entrySet.getValue().get(0).getEntity();
                    entity.setContactEntry(updateContact(contact));
                    contact = null;
                }
            }
        }
        finally
        {
            updateList.clear();
        }
    }

    @Override
    public URI getContextID() throws IdASException
    {
        try
        {
            return new URI(m_contextId.toString());
        }
        catch (URISyntaxException e)
        {
            throw new IdASException(e);
        }
    }

    @Override
    public void close() throws IdASException
    {
        if (!isOpen(null))
            throw new ContextNotOpenException();
        setOpen(false);
    }

    private ContactEntry addContact(IEntity entity) throws IdASException
    {
        try
        {
            ContactEntry contact = new ContactEntry();
            Iterator<IAttribute> attrIter = entity.getAttributes();

            while (attrIter.hasNext())
                updateContact(contact, attrIter.next(), false);
            return m_service.insert(m_feedURL, contact);
        }
        catch (Exception ex)
        {
            throw new IdASException("Could not add contact", ex);
        }
    }

    private ContactEntry updateContact(ContactEntry contact) throws IdASException
    {
        try
        {
            URL entityURL = new URL(contact.getEditLink().getHref());
            contact = m_service.update(entityURL, contact);
        }
        catch (Exception ex)
        {
            new IdASException("Could not update entity", ex);
        }
        return contact;
    }

    private void updateContactWithNotification(EntityNotification entityNotif,
        ContactEntry contact) throws IdASException
    {
        AttributeNotification attrNotif = entityNotif.getAttributeNotification();
        String attrAction = attrNotif.getAction(); 
        if (attrAction.equals(AttributeNotification.UPDATE_VALUE_NOTIFY))
        {
            ValueNotification attrValueNotif = 
                attrNotif.getAttributeValueNotification();

            String valueAction = attrValueNotif.getAction();
            if (valueAction.equals(ValueNotification.UPDATE_ADD))
            {
                updateContact(contact, attrNotif.getAttr(), true);
            }
            else if (valueAction.equals(ValueNotification.UPDATE_REMOVE))
            {
                //TODO Implement replace & remove of attribute values.
            }
        }
        else if (attrAction.equals(AttributeNotification.UPDATE_REMOVE))
        {
            URI attrID = attrNotif.getAttr().getType();
            if (attrID.equals(PROP_TITLE))
                contact.setTitle(null);
            else if (attrID.equals(PROP_CONTENT))
                contact.setContent((TextConstruct)null);
            else if (attrID.equals(PROP_EMAIL))
                contact.getEmailAddresses().clear();
            else if (attrID.equals(PROP_IM))
                contact.getImAddresses().clear();
            else if (attrID.equals(PROP_PHONE))
                contact.getPhoneNumbers().clear();
            else if (attrID.equals(PROP_POSTAL_ADDRESS))
                contact.getPostalAddresses().clear();
            else if (attrID.equals(PROP_ORGANIZATION))
                contact.getOrganizations().clear();
        }
        else if (!attrAction.equals(AttributeNotification.UPDATE_ADD))
        {
            throw new NotImplementedException(attrNotif.getAction());
        }
    }

    private void removeContact(IEntity entity) throws IdASException
    {
        try
        {
            ContactEntry contactEntry = getContact(entity);
            URL entityURL = new URL(contactEntry.getEditLink().getHref());
            m_service.delete(entityURL);
        }
        catch (Exception ex)
        {
            new IdASException("Could not delete entity", ex);
        }
    }

    private ContactEntry getContact(IEntity entity) throws IdASException
    {
        ContactEntry contactEntry = null;
        try
        {
            String entityID = entity.getEntityID();
            URL entityURL = 
                new URL(entityID.replace("/base/", "/" + m_projection + "/"));

            contactEntry = m_service.getEntry(entityURL, ContactEntry.class);
            if (contactEntry == null)
                throw new IdASException("No contact found with ID: " + entityID);
        }
        catch (Exception ex)
        {
            new IdASException("Could not delete entity", ex);
        }
        return contactEntry;
    }

    private static void updateContact(ContactEntry contact, IAttribute attr,
        boolean replace) throws IdASException
    {
        URI attrID = attr.getType(); 
        if (attrID.equals(PROP_TITLE))
        {
            ISimpleValue title = (ISimpleValue)attr.getValues().next();
            contact.setTitle(new PlainTextConstruct(title.getCanonical()));
        }
        else if (attrID.equals(PROP_CONTENT))
        {
            ISimpleValue content = (ISimpleValue)attr.getValues().next();
            contact.setContent(new PlainTextConstruct(content.getCanonical()));
        }
        else if (attrID.equals(PROP_EMAIL))
        {
            if (replace)
                contact.getEmailAddresses().clear();

            Iterator<IEntity> emailIter = attr.getValues();
            while (emailIter.hasNext())
            {
                IEntity attrValue = emailIter.next();
                IAttribute emailAttr = attrValue.getAttribute(PROP_ADDRESS);
                Iterator<ISimpleValue> iter = emailAttr.getValues();

                Email email = new Email();
                email.setAddress(iter.next().getCanonical());
                contact.addEmailAddress(email);

                emailAttr = attrValue.getAttribute(PROP_REL);
                if (emailAttr != null)
                {
                    iter = emailAttr.getValues();
                    email.setRel(iter.next().getCanonical());
                }
            }
        }
        else if (attrID.equals(PROP_ORGANIZATION))
        {
            if (replace)
                contact.getOrganizations().clear();

            Iterator<IEntity> orgIter = attr.getValues();
            while (orgIter.hasNext())
            {
                IEntity attrValue = orgIter.next();
                IAttribute orgAttr = attrValue.getAttribute(PROP_ORG_NAME);
                Iterator<ISimpleValue> iter = orgAttr.getValues();

                Organization organization = new Organization();
                organization.setOrgName(new OrgName(iter.next().getCanonical()));
                contact.addOrganization(organization);

                orgAttr = attrValue.getAttribute(PROP_ORG_TITLE);
                if (orgAttr != null)
                {
                    iter = orgAttr.getValues();
                    organization.setOrgTitle(new OrgTitle(iter.next().getCanonical()));
                }

                orgAttr = attrValue.getAttribute(PROP_REL);
                if (orgAttr != null)
                {
                    iter = orgAttr.getValues();
                    organization.setRel(iter.next().getCanonical());
                }
            }
        }
    }
}
