/**
 * Copyright (c) 2006, 2008 Oracle and/or its affiliates. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Mohamad Raja - Initial implementation
 */
package org.eclipse.higgins.idas.cp.google;

import java.net.URI;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.spi.BasicAttribute;
import org.eclipse.higgins.idas.spi.BasicAttributeSet;
import org.eclipse.higgins.idas.spi.BasicContext;
import org.eclipse.higgins.idas.spi.BasicEntity;

import com.google.gdata.data.contacts.ContactEntry;
import com.google.gdata.data.extensions.Email;
import com.google.gdata.data.extensions.Im;
import com.google.gdata.data.extensions.Organization;
import com.google.gdata.data.extensions.PhoneNumber;
import com.google.gdata.data.extensions.PostalAddress;

public class GDataEntity extends BasicEntity
{
    private static final Log log = LogFactory.getLog(GDataEntity.class.getName());

    private ContactEntry m_contact;
    private boolean m_loadAttrs;

    public GDataEntity(IContext context, URI uriType, String entityID)
        throws IdASException
    {
        this(context, uriType, entityID, null);
    }

    public GDataEntity(IContext context, URI uriType, String entityID, 
        ContactEntry contact) throws IdASException
    {
        super(context, uriType, entityID);
        setContainer((BasicContext)context);
        setContactEntry(contact);
    }

    @Override
    public IAttribute addAttribute(URI type) throws IdASException
    {
        BasicAttributeSet attrSet = getBasicAttributeSet();
        IAttribute attr = attrSet.getAttribute(type);
        if (attr != null)
            return attr;
        return super.addAttribute(type);
    }

    @Override
    public IAttribute getAttribute(URI attrID) throws IdASException
    {
        return null;
    }

    @Override
    public Iterator<IAttribute> getAttributes() throws IdASException
    {
        BasicAttributeSet attributeSet = getBasicAttributeSet();
        if (m_contact != null && !m_loadAttrs)
        {
            loadAttributes(m_contact, attributeSet);
            m_loadAttrs = true;
        }
        return attributeSet.getAttributes();
    }

    protected void setContactEntry(ContactEntry contact)
    {
        m_contact = contact;
        if (contact != null)
            setEntityID(contact.getEditLink().getHref());
    }

    private static void loadAttributes(ContactEntry contact, 
        BasicAttributeSet attributeSet) throws IdASException
    {
        BasicAttribute attr = null;
        if (contact.getTitle() != null)
        {
            attr = attributeSet.addUnnotifiedAttribute(
                GDataContext.PROP_TITLE);
            String contactName = contact.getTitle().getPlainText();
            attr.addSimpleValue(ITypedValue.STRING_TYPE_URI, contactName);
        }

        if (contact.getContent() != null)
        {
            attr = attributeSet.addUnnotifiedAttribute(
                GDataContext.PROP_CONTENT);
            String contactNotes = contact.getTextContent()
                .getContent().getPlainText();
            attr.addSimpleValue(ITypedValue.STRING_TYPE_URI, contactNotes);
        }

        List<Email> emailAddresses = contact.getEmailAddresses();
        if (emailAddresses.size() > 0)
        {
            attr = attributeSet.addUnnotifiedAttribute(
                GDataContext.PROP_EMAIL);

            for (Email emailAddress : emailAddresses)
            {
                IEntity attrValue = 
                    attr.addComplexValue(ITypedValue.ANYURI_TYPE_URI);

                IAttribute compAttr = attrValue.addAttribute(GDataContext.PROP_ADDRESS);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    emailAddress.getAddress());

                compAttr = attrValue.addAttribute(GDataContext.PROP_REL);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    emailAddress.getRel());

                compAttr = attrValue.addAttribute(GDataContext.PROP_PRIMARY);
                compAttr.addSimpleValue(ITypedValue.BOOLEAN_TYPE_URI, 
                    emailAddress.getPrimary());
            }
        }

        List<PhoneNumber> phoneNumbers = contact.getPhoneNumbers();
        if (phoneNumbers.size() > 0)
        {
            attr = attributeSet.addUnnotifiedAttribute(
                GDataContext.PROP_PHONE);

            for (PhoneNumber phoneNumber : phoneNumbers)
            {
                IEntity attrValue = 
                    attr.addComplexValue(ITypedValue.ANYURI_TYPE_URI);

                IAttribute compAttr = attrValue.addAttribute(GDataContext.PROP_TEXT);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    phoneNumber.getPhoneNumber());

                compAttr = attrValue.addAttribute(GDataContext.PROP_REL);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    phoneNumber.getRel());

                compAttr = attrValue.addAttribute(GDataContext.PROP_PRIMARY);
                compAttr.addSimpleValue(ITypedValue.BOOLEAN_TYPE_URI, 
                    phoneNumber.getPrimary());
            }
        }

        List<Organization> organizations = contact.getOrganizations();
        if (organizations.size() > 0)
        {
            attr = attributeSet.addUnnotifiedAttribute(
                GDataContext.PROP_ORGANIZATION);

            for (Organization organization : organizations)
            {
                IEntity attrValue = 
                    attr.addComplexValue(ITypedValue.ANYURI_TYPE_URI);

                IAttribute compAttr = 
                    attrValue.addAttribute(GDataContext.PROP_ORG_NAME);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    organization.getOrgName().getValue());

                compAttr = attrValue.addAttribute(GDataContext.PROP_ORG_TITLE);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    organization.getOrgTitle().getValue());

                compAttr = attrValue.addAttribute(GDataContext.PROP_REL);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    organization.getRel());

                compAttr = attrValue.addAttribute(GDataContext.PROP_PRIMARY);
                compAttr.addSimpleValue(ITypedValue.BOOLEAN_TYPE_URI, 
                    organization.getPrimary());
            }
        }

        List<Im> imAddresses = contact.getImAddresses();
        if (imAddresses.size() > 0)
        {
            attr = attributeSet.addUnnotifiedAttribute(
                GDataContext.PROP_IM);

            for (Im imAddress : imAddresses)
            {
                IEntity attrValue = 
                    attr.addComplexValue(ITypedValue.ANYURI_TYPE_URI);

                IAttribute compAttr = attrValue.addAttribute(GDataContext.PROP_ADDRESS);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    imAddress.getAddress());

                compAttr = attrValue.addAttribute(GDataContext.PROP_REL);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    imAddress.getRel());

                compAttr = attrValue.addAttribute(GDataContext.PROP_PRIMARY);
                compAttr.addSimpleValue(ITypedValue.BOOLEAN_TYPE_URI, 
                    imAddress.getPrimary());
            }
        }

        List<PostalAddress> postalAddresses = contact.getPostalAddresses();
        if (postalAddresses.size() > 0)
        {
            attr = attributeSet.addUnnotifiedAttribute(
                GDataContext.PROP_POSTAL_ADDRESS);

            for (PostalAddress postalAddress : postalAddresses)
            {
                IEntity attrValue = 
                    attr.addComplexValue(ITypedValue.ANYURI_TYPE_URI);

                IAttribute compAttr = attrValue.addAttribute(GDataContext.PROP_TEXT);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    postalAddress.getValue());

                compAttr = attrValue.addAttribute(GDataContext.PROP_REL);
                compAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, 
                    postalAddress.getRel());

                compAttr = attrValue.addAttribute(GDataContext.PROP_PRIMARY);
                compAttr.addSimpleValue(ITypedValue.BOOLEAN_TYPE_URI, 
                    postalAddress.getPrimary());
            }
        }
    }

    @Override
    public String toString()
    {
        StringBuilder buff = new StringBuilder();
        try
        {
            buff.append("\nEntity ID:").append(getEntityID());
            buff.append(toString(getAttributes()));
        }
        catch (IdASException ex)
        {
            log.error(ex.getMessage(), ex);
        }
        return buff.toString();
    }

    private static String toString(Iterator<IAttribute> attrIterator)
        throws IdASException
    {
        StringBuilder buff = new StringBuilder();
        while (attrIterator.hasNext())
        {
            IAttribute attribute = attrIterator.next();
            buff.append("\n Attr ID:").append(attribute.getType());
            buff.append("\n    Value:");

            Iterator<IValue> attrValues = attribute.getValues();
            while (attrValues.hasNext())
            {
                IValue attrValue = attrValues.next();
                if (!attrValue.isSimple())
                {
                    IEntity value = (IEntity)attrValue;
                    buff.append(toString(value.getAttributes()));
                }
                else
                    buff.append(((ISimpleValue)attrValue).getCanonical());
            }
        }
        return buff.toString();
    }
}
