/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.hb.authentication;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.apache.axiom.om.util.UUIDGenerator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.impl.dv.util.Base64;
import org.eclipse.higgins.idas.api.AuthenticationException;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;
import org.eclipse.higgins.idas.common.AuthNAnonymousMaterials;
import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.idas.common.AuthNSelfIssuedMaterials;
import org.eclipse.higgins.idas.cp.hb.IAuthenticationModule;
import org.eclipse.higgins.idas.cp.hb.IHibernateEntity;
import org.eclipse.higgins.idas.cp.hb.IUserAccount;
import org.eclipse.higgins.idas.cp.hb.impl.Context;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;

/**
 * Used to perform an authentication in <code>open</code> method of
 * <code>Context</code>
 * 
 */
public class AuthenticationModule implements IAuthenticationModule {
	private Log log = LogFactory.getLog(AuthenticationModule.class);

	protected Context context_ = null;

	/**
	 * @param context
	 * @throws IdASException
	 */
	public AuthenticationModule(Context context) throws IdASException {
		if (context == null)
			throw new IdASException("The parameter \"context\" is null");
		context_ = context;
	}

	/**
	 * @param credentials
	 * @return
	 * @throws IdASException
	 */
	public IUserAccount authenticateNamePasswordMaterials(AuthNNamePasswordMaterials credentials) throws IdASException {
		String name = credentials.getUsername();
		if (name == null)
			throw new IdASException("AuthNNamePasswordMaterials contains null username.");
		name = name.trim();
		if (name.length() == 0)
			throw new IdASException("AuthNNamePasswordMaterials contains empty string username.");
		String password = credentials.getPassword();
		ArrayList list = getAccountIDByName(name);
		IHibernateEntity userAccount = null;
		switch (list.size()) {
		case 0: {
			try {
				userAccount = createNewAccount(name, password);
			} catch (Exception e) {
				log.error(e, e);
				throw new IdASException(e);
			}
			break;
		}
		case 1: {
			try {
				userAccount = (IHibernateEntity)list.get(0);
			} catch (Exception e) {
				log.error(e, e);
				throw new IdASException(e);
			}
			String passwordHash = getPasswordHash(userAccount);
			checkPassword(passwordHash, password);
			break;
		}
		default:
			throw new IdASException("There are more then one user with name = " + name);
		}
		Serializable userID = getUserID(userAccount);
		return new PasswordBasedUserAccount(context_, userID, userAccount, credentials);
	}

	/**
	 * Find Entities with simple attribute
	 * <code>AuthConstants.USER_PPID_PROPERTY</code>
	 * 
	 * @param ppid
	 * @return
	 * @throws IdASException
	 */
	private IHibernateEntity getUserIDByPPIDAttribute(String ppid) throws IdASException {
		IContextModel model = context_.getContextModel();
		URI ppidOwnerEntityType = context_.getTypeResolver().getPpidOwnerEntityType();
		if (ppidOwnerEntityType == null)
			throw new IdASModelException("This context does not support PPID authentication .There is no entity with attribute = " + AuthConstants.PPID_PROPERTY_URI.toString());
		IEntityModel ppidOwner = context_.getContextModel().getEntityModel(ppidOwnerEntityType);
		int maxCrd = ppidOwner.getMaxCardinality(AuthNSelfIssuedMaterials.ATTR_TYPE_PPID);
		String hbEntityName = context_.getTypeResolver().modelTypeToHibernateType(ppidOwner);
		String queryStr;
		if (maxCrd == 1) {
			queryStr = "SELECT obj FROM " + hbEntityName + "as obj WHERE " + AuthConstants.PPID_HIBERNATE_NAME + ".value = '" + ppid.trim().replaceAll("\\'", "''") + "'";
		}
		else {
			queryStr = "SELECT obj FROM " + hbEntityName + " as obj inner join obj.PPID obj2 WHERE obj2.value = " +			
			"'" + ppid.trim().replaceAll("\\'", "''") + "'";
		}
		Session session = null;
		try {
			session = context_.getSessionFactory().openSession().getSession(context_.getHibernateEntityFactory().getEntityMode());
			Query query = session.createQuery(queryStr);
			List lst = query.list();
			if (lst.size() == 0)
				return null;
			if (lst.size() > 1)
				throw new AuthenticationException("There are more than one Entity with the same value of " + AuthConstants.PPID_PROPERTY + " attribute");
			Object obj = lst.get(0);
			IHibernateEntity entity = context_.getHibernateEntityFactory().create(obj, null);
			List accounts = entity.getCollection(AuthConstants.USER_PROPERTY_NAME);
			if (accounts.size() == 0) {
				throw new IdASException("The entity, assosiated with PPID value, does not contain any user account object ("
						+ AuthConstants.USER_ACCOUNT + ")");
			}
			else if (accounts.size() > 1) {
				throw new IdASException("The entity, assosiated with PPID value, contains more than one user account object ("
						+ AuthConstants.USER_ACCOUNT + ")");
			}
			else {
				return (IHibernateEntity)accounts.get(0);
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new IdASException(e);
		} finally {
			try {
				if (session != null)
					session.close();
			} catch (Exception e) {
				log.error(e, e);
				throw new IdASException(e);
			}
		}
	}

//	/**
//	 * @param entity
//	 * @return
//	 * @throws IdASException
//	 */
//	private IHibernateEntity getUserTokenByEntity(Element entity) throws IdASException {
//		if (entity == null)
//			throw new IdASException("Parameter \"node\" is null.");
//		Element userAttr = entity.element(AuthConstants.USER_PROPERTY_NAME);
//		if (userAttr == null)
//			return null;
//		return userAttr.element(AuthConstants.USER_ACCOUNT);
//	}

	/**
	 * @param credentials
	 * @return
	 * @throws IdASException
	 */
	public IUserAccount authenticateSelfIssuedMaterials(AuthNSelfIssuedMaterials credentials) throws IdASException {
		if (credentials == null)
			return null;
		String ppid = null;
		ByteArrayOutputStream cardKey = new ByteArrayOutputStream();
		try {
			cardKey.write(credentials.getPPIDBytes());
			cardKey.write(credentials.getPublicKeyModBytes());
			cardKey.write(credentials.getPublicKeyExpBytes());
			MessageDigest md = MessageDigest.getInstance("SHA-1");
			byte[] hash = md.digest(cardKey.toByteArray());
			ppid = Base64.encode(hash).trim();
		} catch (IOException e) {
			log.error(e, e);
			throw new IdASException(e);
		} catch (NoSuchAlgorithmException e) {
			log.error(e, e);
			throw new IdASException(e);
		}
		Serializable userAccountID = null;
		IHibernateEntity userAccount = getUserIDByPPIDAttribute(ppid);
		if (userAccount == null)
			return null;
		else {
			userAccountID = getUserID(userAccount);
			return new PPIDBasedUserAccount(context_, userAccountID, userAccount, credentials);
		}
	}

	public IUserAccount authenticate(Object credentials) throws IdASException {
		if (credentials instanceof AuthNNamePasswordMaterials) {
			AuthNNamePasswordMaterials am = (AuthNNamePasswordMaterials) credentials;
			return authenticateNamePasswordMaterials(am);
		} else if (credentials instanceof AuthNSelfIssuedMaterials) {
			AuthNSelfIssuedMaterials am = (AuthNSelfIssuedMaterials) credentials;
			return authenticateSelfIssuedMaterials(am);
		} else if (credentials instanceof AuthNAnonymousMaterials) {
			if (context_.isAuthenticationRequired()) {
				throw new IdASException("Context requires authentication.");
			}
			return new AnonymousUserAccount(context_, credentials);
		}
		throw new IdASException("Unsupported credentials type.");
	}

	/**
	 * @param name
	 * @param password
	 * @return
	 * @throws IdASException
	 */
	private IHibernateEntity createNewAccount(String name, String password) throws IdASException {
		Session session = null;
		Transaction trans = null;
		IHibernateEntity account = context_.getHibernateEntityFactory().create(null, AuthConstants.USER_ACCOUNT);
		try {
			session = context_.getSessionFactory().openSession().getSession(context_.getHibernateEntityFactory().getEntityMode());
			trans = session.beginTransaction();
			trans.begin();
			account.setObject(AuthConstants.USER_LOGIN, name);
			account.setObject(AuthConstants.USER_TOKEN, UUIDGenerator.getUUID());
			setNewPassword(account, password);
			session.save(account.getEntityName(), account.getEntityObject());
			trans.commit();
		} catch (Exception e) {
			if (trans != null) {
				try {
					trans.rollback();
				} catch (Exception ex) {
					log.error(ex, ex);
					throw new IdASException(ex);
				}
			}
			log.error(e, e);
			throw new IdASException(e);
		} finally {
			try {
				if (session != null)
					session.close();
			} catch (Exception e) {
				log.error(e, e);
				throw new IdASException(e);
			}
		}
		return account;
	}

	/**
	 * @param user
	 * @param password
	 * @throws IdASException
	 */
	private void setNewPassword(IHibernateEntity userAccount, String password) throws IdASException {
		String newHash = null;
		byte[] bytes = getHash(password);
		if (bytes != null)
			newHash = Base64.encode(bytes);
		userAccount.setObject(AuthConstants.USER_PASSWORD_HASH, newHash);
	}

	/**
	 * @param passHash
	 * @param password
	 * @throws IdASException
	 */
	private void checkPassword(String passHash, String password) throws IdASException {
		if (passHash == null || passHash.length() == 0) {
			if (password != null && password.length() > 0)
				throw new IdASException("Wrong password.");
		} else {
			if (password == null || password.length() == 0)
				throw new IdASException("Wrong password.");
			byte[] realPassHash = null;
			byte[] testedHash = null;
			testedHash = getHash(password);
			try {
				realPassHash = Base64.decode(passHash);
			} catch (Exception e) {
				log.error(e, e);
				throw new IdASException(e.getMessage());
			}
			if (Arrays.equals(realPassHash, testedHash) == false)
				throw new IdASException("Wrong password.");
		}
	}

	private String getPasswordHash(IHibernateEntity userAccount) throws IdASException {
		return (String)userAccount.getObject(AuthConstants.USER_PASSWORD_HASH);
	}

	private Serializable getUserID(IHibernateEntity userAccount) throws IdASException {
		return (Serializable)userAccount.getObject("id_");
	}

	/**
	 * @param str
	 * @return
	 * @throws IdASException
	 */
	private byte[] getHash(String str) throws IdASException {
		if (str != null && str.length() > 0) {
			try {
				MessageDigest md = MessageDigest.getInstance("SHA-256");
				return md.digest(str.getBytes("UTF-8"));
			} catch (Exception e) {
				log.error(e, e);
				throw new IdASException(e.getMessage());
			}
		} else
			return null;
	}

	/**
	 * @param userName
	 * @return
	 * @throws IdASException
	 */
	public ArrayList getAccountIDByName(String userName) throws IdASException {
		Session session = null;
		try {
			session = context_.getSessionFactory().openSession().getSession(context_.getHibernateEntityFactory().getEntityMode());
			String queryStr = "FROM " + AuthConstants.USER_ACCOUNT + " WHERE " + AuthConstants.USER_LOGIN + " = '"
					+ userName.trim().replaceAll("\\'", "''") + "'";
			Query query = session.createQuery(queryStr);
			ArrayList accs = new ArrayList();
			Iterator itr = query.iterate();
			while (itr.hasNext()) {
				Object accountObj = itr.next();
				IHibernateEntity account = context_.getHibernateEntityFactory().create(accountObj, null);
//				account.element("password_hash");
//				account.element("id_");
				accs.add(account);
			}
			return accs;
		} catch (Exception e) {
			log.error(e, e);
			throw new IdASException(e);
		} finally {
			try {
				if (session != null)
					session.close();
			} catch (Exception e) {
				log.error(e, e);
				throw new IdASException(e);
			}
		}
	}

}
