/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.hb.filter.impl;

import java.net.URI;
import java.util.Iterator;
import org.eclipse.higgins.idas.api.BadFilterException;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.ISimpleValueModel;
import org.eclipse.higgins.idas.cp.hb.impl.Context;
import org.eclipse.higgins.idas.cp.hb.util.HBVocabulary;
import org.eclipse.higgins.idas.cp.model.util.HigginsVocabulary;
import org.hibernate.Criteria;
import org.hibernate.criterion.Conjunction;
import org.hibernate.criterion.CriteriaSpecification;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

public class FilterAttributeAssertion extends FilterAssertion implements IFilterAttributeAssertion {
//	private Log log = LogFactory.getLog(FilterAttributeAssertion.class);

	private IValue value_ = null;

	private Filter attributeFilter_ = null;

	private IContextModel contextModel_;

	private Context context_;

	private IAttributeModel attModel_ = null;

	/*
	 * check if the value model conform the model of attribute (ID) ie simple or complex value in according
	 * attribute type determine if the value is list (complex or simple with cardinality >1) or one value
	 * (simple)
	 */
	private void checkAssertion() throws IdASException {
		if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_PRESENT) && value_ != null)
			throw new BadFilterException("Assertion Value does not need to be set when the comparator is " 
					+ IFilterAttributeAssertion.COMP_ATTR_PRESENT);

		if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_INFERRED_REL_EQ))
		{
			//TODO check if the ID is the attribute of type entity relation
			//and the value is the reference to another entity (EntityId)
			//TODO it need to check the sub-typing of attributes, perhaps will be implemented with new HOWL
			if (type_ != HigginsVocabulary.entityRelationURI &&
					type_ != HigginsVocabulary.entityCorrelationURI) {
				throw new BadFilterException("Attribute type should be Entity relation or Entity correlation, "
						+ " but occured " + type_);
			}
			if (!value_.isSimple()) {
				//implement with new HOWL - checking value type
				throw new BadFilterException("Assertion value must be simpl and contain Entity ID");
			}
		}
		
		if (attModel_.isSimple() && !value_.isSimple())
			throw new BadFilterException("Attribute " + type_ + " defined as simple but assertion value is complex.");
		if (!attModel_.isSimple() && value_.isSimple())
			throw new BadFilterException("Attribute " + type_ + " defined as complex but assertion value is simple.");
	}

	/**
	 * @param context
	 * @throws IdASException 
	 */
	public FilterAttributeAssertion(Context context) throws IdASException {
		if (context == null)
			throw new BadFilterException("Context cannot be null");
		context_ = context;
		contextModel_ = context.getContextModel();
	}

	public void setID(URI id) throws IdASException {
		if (id == null)
			throw new IdASException("Attribute type id can not be null!");
		for (Iterator itEntity = contextModel_.getEntityModels().iterator(); itEntity.hasNext();) {
			IEntityModel entityModel = (IEntityModel) itEntity.next();
			for (Iterator itAtt = entityModel.getAttributeModels().iterator(); itAtt.hasNext();) {
				IAttributeModel att = (IAttributeModel) itAtt.next();
				if (att.getType().equals(id)) {
					super.setID(id);
					attModel_ = att;
					return;
				}
			}
		}
		
		attModel_ = null;
		throw new BadFilterException("This ID is not valid ID for Entity attribute " + id);
	}
	
	public void setComparator(String comparator) throws IdASException {
		if (comparator == null)
			throw new BadFilterException("Comparator string cannot be null");
		if (!comparator.equals(IFilterAttributeAssertion.COMP_ATTR_PRESENT)
				&& !comparator.equals(IFilterAttributeAssertion.COMP_ATTR_GE)
				&& !comparator.equals(IFilterAttributeAssertion.COMP_ATTR_LE)
				&& !comparator.equals(IFilterAttributeAssertion.COMP_ATTR_EQ)
				&& !comparator.equals(IFilterAttributeAssertion.COMP_ATTR_SUBSTR)
				&& !comparator.equals(IFilterAttributeAssertion.COMP_ATTR_INFERRED_REL_EQ)
			)
			throw new IdASException("Unsupported compatator: " + comparator + " for " + this.getClass().getName());
		comparator_ = comparator;
	}
	
	public void setAssertionValue(IValue assertionValue) throws IdASException {
		if (assertionValue == null)
			throw new BadFilterException("Assertion value cannot be null");
		if (attModel_ == null)
			throw new BadFilterException("Set Type ID first");
		
		if (attModel_.isSimple() && !assertionValue.isSimple())
			throw new BadFilterException("Assertion value must be simple for attribute " + attModel_.getType());
		if (!attModel_.isSimple() && assertionValue.isSimple())
			throw new BadFilterException("Assertion value must be complex for attribute " + attModel_.getType());
			
		value_ = assertionValue;			
	}
	
	public void setAttributeFilter(IFilter attributeFilter) throws IdASException {
		if (attributeFilter instanceof Filter)
			attributeFilter_ = (Filter) attributeFilter;
		else
			throw new BadFilterException("Filter created by another context provider implementation cannot be added");
	}
	
	public Criterion getRestriction(Criteria criteria) throws IdASException {
		checkAssertion();
		Criterion res = null;
		String hibernateId = context_.getTypeResolver().modelTypeToHibernateType(attModel_);
		if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_PRESENT)) {
			if (attModel_.isSimple())
				res = Property.forName(hibernateId).isNotNull();
				//criteria.add(Property.forName(attModel_.getId()).isNotNull());
			else
				res = Property.forName(hibernateId).isNotEmpty();
				//criteria.add(Property.forName(attModel_.getId()).isNotEmpty());

		} else if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_INFERRED_REL_EQ)) {
			ISimpleValue val = (ISimpleValue) value_;
			String entityId = (String) val.getData();
			res = Property.forName(hibernateId).eq(entityId);
			//criteria.add(Property.forName(attModel_.getId()).eq(entityId));
		} else {
			if (attModel_.isSimple()) 
				res = processSimple((ISimpleValueModel)attModel_.getValueModel(), (ISimpleValue) value_, criteria, null);
				//processSimple((SimpleValueModel)attModel_.getValueModel(), (ISimpleAttrValue) value_, criteria);
			else 
				res = processComplex((IEntityModel)attModel_.getValueModel(), (IEntity) value_, criteria);
				//processComplex((ComplexValueModel)attModel_.getValueModel(), (IComplexAttrValue) value_, criteria);
		}
		
		if (attributeFilter_ != null)
			res = Restrictions.and(res, FilterContext.getFilterResult(attributeFilter_, criteria));
			//criteria.add(FilterContext.getFilterResult(attributeFilter_, criteria));
		return res;
	}

	private Criterion processSimple(ISimpleValueModel valModel, ISimpleValue attValue, Criteria criteria, String alias) throws IdASException {
		Object value = attValue.getData();
		String hibernateId = context_.getTypeResolver().modelTypeToHibernateType(attModel_);
		if ((attModel_.getAttributeMode() == IAttributeModel.SINGLE_VALUED_MODE)) {
			Property prop = Property.forName((alias != null ? alias + "." : "") + hibernateId);
			if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_EQ)) 
				return prop.eq(value);
			else if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_GE)) 
				return prop.ge(value);
			else if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_LE)) 
				return prop.le(value);
			else if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_SUBSTR)) 
				return prop.like(value.toString(), MatchMode.ANYWHERE);
		}
		
		   // !!!  MappingHelper.getEntityNameForMultivaluedAttribute(valModel) - should be used to get the name of SV entitiy list !!!!
		else {
			//in case of list simple value model must return the name of entity (table) where values are stored
			//DetachedCriteria det = DetachedCriteria.forEntityName(valModel.getId());
			//DetachedCriteria det = DetachedCriteria.forEntityName(valModel.getHbMappingOfListObject());
			//DetachedCriteria det = DetachedCriteria.forEntityName("jut_SimplePersonNOMV_jut_email");
			alias = (alias != null ? alias + "." : "") + hibernateId + "_";
			criteria.createAlias(hibernateId, alias, CriteriaSpecification.LEFT_JOIN);
			
			Property prop = Property.forName(alias + "." + HBVocabulary.SIMPLE_VALUE_FIELD_NAME);
			
			if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_EQ))
				//det.add(Restrictions.sqlRestriction("{alias}."+HBVocabulary.SIMPLE_VALUE_FIELD_NAME+"='"+value+"'"));
				//res = prop.eq(value);
				return prop.eq(value);
			else if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_GE))
				//res = prop.ge(value);
				return prop.ge(value);
			else if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_LE))
				//res = prop.le(value);
				return prop.le(value);
			else if (comparator_.equals(IFilterAttributeAssertion.COMP_ATTR_SUBSTR))
				//res = prop.like(value.toString(), MatchMode.ANYWHERE);
				return prop.like(value.toString(), MatchMode.ANYWHERE);
			
			//return Restrictions.and(Restrictions.isNotEmpty(valModel.getId()), Subqueries.exists(det));
			//res);
		}
		
		return null;
	}
	
	private Criterion processComplex(IEntityModel valModel, IEntity value, Criteria criteria) throws IdASException {
		Conjunction conj = Restrictions.conjunction();
		String hibernateId = context_.getTypeResolver().modelTypeToHibernateType(attModel_);
		String alias = hibernateId + "_";
		criteria.createAlias(hibernateId, alias, CriteriaSpecification.LEFT_JOIN);
		
		for (Iterator iterator = value.getAttributes(); iterator.hasNext();) {
			IAttribute att = (IAttribute) iterator.next();
			//attr model is taken by ID from parent model, not value!
			//because value may be BasicComplexValue and not implement getModel()
			IAttributeModel attModel = (IAttributeModel)valModel.getAttributeModel(att.getType());
			if (attModel == null)
				throw new BadFilterException("Unknown attribute " + att.getType() + " for complex value " + valModel.getType());
			IValue val = (IValue) att.getValues().next(); //works only one value
			if (attModel.isSimple()) {
				conj.add(processSimple((ISimpleValueModel)attModel.getValueModel(), (ISimpleValue) val, criteria, alias));
			}
			else
				conj.add(processComplex((IEntityModel)attModel.getValueModel(), (IEntity) val, criteria));
		} 
		return conj;
	}



	public void setMetadataFilter(IFilter arg0) throws IdASException {
		// TODO Auto-generated method stub
		
	}

}
