/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.hb.filter.impl;

import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.BadFilterException;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.cp.hb.authentication.AuthConstants;
import org.eclipse.higgins.idas.cp.hb.impl.Context;
import org.eclipse.higgins.idas.cp.hb.util.MappingBuilder;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Conjunction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Restrictions;


public class FilterContext {
	private Log log = LogFactory.getLog(FilterContext.class);

	//private Session session_ = null;
	//private ContextModel model_;
	private Context ctx;


	/**
	 * @param idasContext
	 * @throws IdASException
	 */
	public FilterContext(Context context) {
		ctx = context;
	}

	static Criterion getFilterResult(Filter filter, Criteria criteria) throws IdASException {
		Criterion crit = null;
		FilterAssertion assertion = (FilterAssertion) filter.getAssertion();

		if (assertion == null) { //mulpiple conditions, process underlying filters
			List undFilters = filter.getFilters();
			if (undFilters == null || undFilters.size() == 0 || filter.getOperator() == null) {
				throw new BadFilterException("Both filters and operator should be set.");
			} 
			
			String op = filter.getOperator();
			if (op == IFilter.OP_NOT) {
				if (undFilters.size() > 1)
					throw new IdASException("Only one filter should be present for operator " + filter.getOperator());				
				Filter flt = (Filter) undFilters.get(0);

				crit = Restrictions.not(getFilterResult(flt, criteria));
			} else {
				if (undFilters.size() < 2)
					throw new IdASException("At least 2 filters should be present for operator " + filter.getOperator());
				
				if (op == IFilter.OP_AND) {
					Conjunction conj = Restrictions.conjunction();
					//iterate all inner filters, get their results and process them with AND
					for (Iterator iterator = undFilters.iterator(); iterator.hasNext();) {
						Filter flt = (Filter) iterator.next();
						conj.add(getFilterResult(flt, criteria));
					}
					crit = conj;
				}
				if (op == IFilter.OP_OR) {
					Disjunction disj = Restrictions.disjunction();
					//iterate all inner filters, get their results and process them with OR
					for (Iterator iterator = undFilters.iterator(); iterator.hasNext();) {
						Filter flt = (Filter) iterator.next();
						disj.add(getFilterResult(flt, criteria));
					}
					crit = disj;
				}
			}			
		} else //filter is only simple assertion
			crit = assertion.getRestriction(criteria);

		return crit;
	}

	/**
	 * @param filter
	 * @return
	 * @throws IdASException
	 */
	public List getEntityElements(Filter filter) throws IdASException {
		log.trace(getClass().getName()+"::getEntities");
		
		try {
			IContextModel model = (IContextModel) ctx.getContextModel();
			Session session = ctx.getSession();
			
			Criteria crit = session.createCriteria(MappingBuilder.higgins_Entity);
			if (filter != null)
				crit.add(getFilterResult(filter, crit));
			if (ctx.isAuthenticationRequired()) {
				Object userAccount = ctx.getUserAccount().getUserAccount();
				crit.add( Restrictions.eq(AuthConstants.USER_ACCOUNT, userAccount) );
			}
			return crit.list();
		} catch (Exception e) {
			log.error(e, e);
			throw new IdASException(e);
		}
	}


//	public void addUserTokenToQuery() throws IdASException {
//		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::addUserTokenToQuery");
//		IUserAccount account = idasContext_.getUserAccount();
//		if (account != null) {
//			String userToken = account.getUserToken();
//			if (userToken != null && "".equals(userToken.trim()) == false) {
//				GraphPattern pattern = registerPattern(new GraphPattern(getSubjectEntity(), new PropertyEntity(AuthConstants.USER_TOKEN_PROPERTY),
//						new LiteralEntity(), null));
//				CompoundCondition cc = new CompoundCondition(Filter.OP_AND);
//				cc.addCondition(queryCondition_);
//				cc.addCondition(new UntypedValueCondition((LiteralEntity) pattern.getObject(), IFilterAttributeAssertion.COMP_ATTR_EQ, userToken));
//				queryCondition_ = cc;
//			}
//		}
//	}

}
