/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.hb.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.ISimpleValueModel;
import org.eclipse.higgins.idas.api.model.IValueModel;
import org.eclipse.higgins.idas.cp.hb.IHibernateEntity;
import org.eclipse.higgins.idas.cp.hb.IStatefulObject;
import org.eclipse.higgins.idas.cp.hb.util.TypeResolver;

public class Attribute implements IAttribute {
	private static Log log = LogFactory.getLog(Attribute.class);

	protected Context context_ = null;

	protected IAttributeModel attModel_ = null;

	// the entity which owns this attribute (directly or through any level of complex value)
	protected Entity entity_ = null;

	protected StatefulObject directOwner_ = null;

	protected ArrayList values_ = null;
	
	protected boolean isPolymorphismSupported_ = true;

	protected Attribute(Context context, Entity entity, IAttributeModel attModel) throws IdASException {
		if (context == null)
			throw new IdASException("Parameter \"context\" is null.");
		if (entity == null)
			throw new IdASException("Parameter \"entity\" is null.");
		if (attModel == null)
			throw new IdASException("Parameter \"attModel\" is null.");
		context_ = context;
		attModel_ = attModel;
		entity_ = entity;
		directOwner_ = entity;
		initValues();
	}

	protected IValueModel getValueModel(URI type) throws IdASException {
		if (type == null) {
			return attModel_.getValueModel();
		}
		Iterator values = attModel_.getValueModels().iterator();
		while (values.hasNext()) {
			IValueModel value = (IValueModel)values.next();
			if (type.equals(value.getType()))
				return value;
			if (!attModel_.isSimple() && isPolymorphismSupported_) {
				IEntityModel subValue = getSubEntityModel((IEntityModel)value, type);
				if (subValue != null)
					return subValue;
			}
		}
		return null;
	}

	private IEntityModel getSubEntityModel(IEntityModel superModel, URI type) throws IdASException {
		Iterator values = superModel.getSubEntityModels().iterator();
		while (values.hasNext()) {
			IEntityModel value = (IEntityModel)values.next();
			if (type.equals(value.getType()))
				return value;
			else {
				IEntityModel subValue = getSubEntityModel(value, type);
				if (subValue != null)
					return subValue;
			}
		}
		return null;
	}

	/**
	 * @return <code>Element</code> of <code>Entity</code> or <code>ComplexValue</code> which owns this attribute
	 */
	public IHibernateEntity getOwnerEntity() {
		return directOwner_.getHibernateEntity();
	}

	public StatefulObject getOwner() {
		return directOwner_;
	}

	/**
	 * @return Entity which owns this attribute (directly or through any level of complex value)
	 */
	public Entity getEntityOwner() {
		return entity_;
	}

	public IEntity addComplexValue(URI type) throws IdASException, InvalidTypeException {
		if (attModel_.isSimple() == false) {
			context_.registerChangedSubject(entity_);
			IEntityModel vm = (IEntityModel)getValueModel(type);
			String attrId = context_.getTypeResolver().modelTypeToHibernateType(attModel_);
			String valueId = context_.getTypeResolver().modelTypeToHibernateType(vm);
			boolean isSingleValued = isSingleValued();
			IHibernateEntity valueEntity = getOwnerEntity().buildCollectionEntity(attrId, valueId, isSingleValued);
			Entity newCV = new Entity(context_, this, vm, valueEntity, false);
			values_.add(newCV);
			return newCV; 
		}
		else
			throw new IdASException("Can not create complex value for simple attribute with type " + getType().toString());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#addSimpleValue(java.net.URI, java.lang.Object)
	 */
	public ISimpleValue addSimpleValue(URI type, Object data) throws IdASException, InvalidTypeException {
		if (attModel_.isSimple() == true) {
			SimpleValue sv = null;
			context_.registerChangedSubject(entity_);
			ISimpleValueModel vm = (ISimpleValueModel)getValueModel(type);
			if (attModel_.getAttributeMode() == IAttributeModel.SINGLE_VALUED_MODE) {
				sv = new SimpleValue(context_, this, null, vm, false);
			}
			else {
				String attrId = context_.getTypeResolver().modelTypeToHibernateType(attModel_);
				String valueId = attrId + TypeResolver.SIMPLE_VALUE_ENTITY_SUFIX;
				boolean isSingleValued = isSingleValued();
				IHibernateEntity valueEntity = getOwnerEntity().buildCollectionEntity(attrId, valueId, isSingleValued);
				sv = new SimpleValue(context_, this, valueEntity, vm, false);
			}
			sv.setData(data);
			values_.add(sv);
			return sv;
		} else
			throw new IdASException("Can not create simple value for complex attribute with type " + getType().toString());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#addValue(java.net.URI)
	 */
	public IValue addValue(URI type) throws IdASException, InvalidTypeException {
		if (attModel_.isSimple())
			return addSimpleValue(type, null);
		else
			return addComplexValue(type);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#addValue(org.eclipse.higgins.idas.api.IValue)
	 */
	public IValue addValue(IValue copyFrom) throws IdASException {
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		URI newType = copyFrom.getType();
		if (newType == null)
			throw new IdASException("Type of passed value \"copyFrom\" is null.");
		if (newType.equals(this.getModel().getValueModel().getType()) == false) {
			//TODO make fix!!!!!!!!!!!!!!!!!!!!
			log.warn("Does not match value with type " + newType.toString() 
					+ " for attribute " + getType().toString() 
					+ ", expecting " + this.getModel().getValueModel().getType());
			/*throw new IdASException("Can not create value with type " + newType.toString() 
					+ " for attribute " + getAttrID().toString() 
					+ ", expecting " + this.getModel().getValueModel().getType());*/
		}
		if (attModel_.isSimple()) {
			if (copyFrom instanceof ISimpleValue) {
				ISimpleValue from = (ISimpleValue) copyFrom;
				ISimpleValue newVal = addSimpleValue(newType, from.getData());
				return newVal;
			} else
				throw new IdASException("Passed value \"copyFrom\" with type " + newType.toString() + " is not an instance of ISimpleValue");
		} else {
			if (copyFrom instanceof IEntity) {
				IEntity from = (IEntity) copyFrom;
				IEntity newVal = addComplexValue(newType);
				Iterator itrAttr = from.getAttributes();
				if (itrAttr != null) {
					while (itrAttr.hasNext()) {
						IAttribute attr = (IAttribute) itrAttr.next();
						newVal.addAttribute(attr);
					}
				}
				return newVal;
			} else
				throw new IdASException("Passed value \"copyFrom\" with type " + newType.toString() + " is not an instance of IEntity");
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#getType()
	 */
	public URI getType() throws IdASException {
		return attModel_.getType();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#getModel()
	 */
	public IAttributeModel getModel() throws IdASException {
		return attModel_;
	}

	private void initSimpleValues() throws IdASException {
		values_ = new ArrayList();
		ISimpleValueModel vm = (ISimpleValueModel)attModel_.getValueModel();
		if (attModel_.getAttributeMode() == IAttributeModel.SINGLE_VALUED_MODE) {
			SimpleValue sv = new SimpleValue(context_, this, null, vm, true);
			values_.add(sv);
		}
		else {
			String valId = context_.getTypeResolver().modelTypeToHibernateType(attModel_);
			List values = getOwnerEntity().getCollection(valId);
			Iterator itr = values.iterator();
			while (itr.hasNext()) {
				IHibernateEntity val = (IHibernateEntity)itr.next();
				SimpleValue sv = new SimpleValue(context_, this, val, vm, true);
				values_.add(sv);
			}
		}
	}

	private void initComplexValues() throws IdASException {
		values_ = new ArrayList();
		List values = getOwnerEntity().getCollection(context_.getTypeResolver().modelTypeToHibernateType(attModel_));
		Iterator itr = values.iterator();
		while (itr.hasNext()) {
			IHibernateEntity val = (IHibernateEntity)itr.next();
			URI entityType = context_.getTypeResolver().hibernateTypeToModelType(val.getEntityName());
			IEntityModel vm = (IEntityModel)getValueModel(entityType);
			if (vm != null) {
				Entity cv = new Entity(context_, this, vm, val, true);
				values_.add(cv);
			}
			else {
				log.error("Unexpected entity type " +  val.getEntityName() + " for attribute " + getType().toString() + ". Value will be skipped.");
			}
		}
	}

	public synchronized void initValues() throws IdASException {
		if (values_ == null) {
			if (attModel_.isSimple())
				initSimpleValues();
			else
				initComplexValues();
		}
	}
	
	/**
	 * @return For internal use. Return all values i.e. removed from this attribute 
	 */
	public ArrayList getInternalValues() {
		return values_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#getValues()
	 */
	public Iterator getValues() throws IdASException {
		ArrayList vals = new ArrayList();
		Iterator itr = values_.iterator();
		while (itr.hasNext()) {
			IStatefulObject so = (IStatefulObject)itr.next();
			String state = so.getState();
			if (! IStatefulObject.PRE_DELETED_STORED_OBJ.equals(state ) && ! IStatefulObject.PRE_DELETED_NEW_OBJ.equals(state)) {
				vals.add(so);
			}
		}
		return Collections.unmodifiableList(vals).iterator();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#isSingleValued()
	 */
	public boolean isSingleValued() throws IdASException {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#remove()
	 */
	public void remove() throws IdASException {
		Iterator itr = getValues();
		while (itr.hasNext()) {
			IValue val = (IValue)itr.next();
			val.remove();
		}
	}

	public boolean equals(IAttribute attr) throws IdASException {
		return false;
	}

	public Iterator getValues(IExtension[] extensions) throws IdASException {
		throw new NotImplementedException();
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		throw new NotImplementedException();
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		throw new NotImplementedException();
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		throw new NotImplementedException();
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		throw new NotImplementedException();
	}

	public Iterator getAttributes() throws IdASException {
		throw new NotImplementedException();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {
		throw new NotImplementedException();
	}

	public void removeAttribute(URI attrID) throws IdASException {
		throw new NotImplementedException();
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {
		throw new NotImplementedException();
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		throw new NotImplementedException();
	}

	/**
	 * @param val
	 */
	protected void removeValue(IValue val) {
		if (values_.contains(val)) {
			values_.remove(val);
		}
	}
//
//	protected String getHibernatePropertyName() {
//		return attModel_.getId();
//	}

	public IEntity addComplexValue(String entityID) throws IdASException, InvalidTypeException {
		throw new NotImplementedException();
	}

	public IEntity addAttributeValue(URI attrType, String entityId) throws IdASException {
		throw new NotImplementedException();
	}

}