/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.hb.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IValueModel;
import org.eclipse.higgins.idas.cp.hb.IHibernateEntity;
import org.eclipse.higgins.idas.cp.hb.IStatefulObject;
import org.eclipse.higgins.idas.cp.hb.util.MappingBuilder;

public class Entity extends StatefulObject implements IEntity {
	protected Context context_ = null;
	protected IEntityModel model_ = null;
	protected IHibernateEntity entity_ = null;
	protected Attribute attributeOwner_;
	private HashMap attributes_ = new HashMap();
	private String entityId_;

	public Entity(Context context, Attribute attributeOwner, IEntityModel entityModel, IHibernateEntity entity, boolean isStoredEntity)
			throws IdASException {
		if (context == null)
			throw new IdASException("Couldn't create Entity with null context");
		if (entity == null)
			throw new IdASException("Couldn't create Entity for null entity");
		if (entityModel == null)
			throw new IdASException("Couldn't create Entity for null entity entityModel");
		context_ = context;
		entity_ = entity;
		model_ = entityModel;
		entityId_ = (String)entity.getObject(MappingBuilder.higgins_entityId);

		attributeOwner_ = attributeOwner;
		if (isStoredEntity) {
			setState(STORED_OBJ);
		}
	}

	public IContext getContext() throws IdASException {
		return context_;
	}

	// public Context getHBContext() {
	// return context_;
	// }

	public IHibernateEntity getHibernateEntity() {
		return entity_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IBlankEntity#getModel()
	 */
	public IEntityModel getModel() {
		return model_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IEntity#getEntityID()
	 */
	public String getEntityID() throws IdASException {
		return entityId_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IValue#getType()
	 */
	public URI getType() throws IdASException {
		return model_.getType();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IDigitalSubject#remove()
	 */
	public void remove() throws IdASException {
		context_.registerChangedSubject(this);
		Iterator attrs = getAttributes();
		while (attrs.hasNext()) {
			Attribute attr = (Attribute) attrs.next();
			attr.remove();
		}
		markDeleted();
		if (attributeOwner_ != null && IStatefulObject.PRE_DELETED_NEW_OBJ.equals(state)) {
			String attrId = context_.getTypeResolver().modelTypeToHibernateType(attributeOwner_.getModel());
			attributeOwner_.getOwnerEntity().removeFromCollection(attrId, entity_);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(java.net.URI)
	 */
	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		return getAttribute(attrID);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(org.eclipse.
	 * higgins.idas.api.IAttribute)
	 */
	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		IAttribute newAttr = getAttribute(copyFrom.getType());
		context_.registerChangedSubject(this);
		Iterator itrVal = copyFrom.getValues();
		while (itrVal.hasNext()) {
			IValue value = (IValue) itrVal.next();
			newAttr.addValue(value);
		}
		return newAttr;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.api.IHasAttributes#getAttribute(java.net.URI)
	 */
	public IAttribute getAttribute(URI attrID) throws IdASException {
		if (attrID == null)
			throw new IdASException("Parameter \"attrID\" is null.");
		if (attributes_.containsKey(attrID))
			return (IAttribute) attributes_.get(attrID);
		else {
			return initAttribute(attrID);
		}
	}

	/**
	 * @param attrID
	 * @return
	 * @throws IdASException
	 */
	private synchronized IAttribute initAttribute(URI attrType) throws IdASException {
		if (attributes_.containsKey(attrType)) {
			return (IAttribute)attributes_.get(attrType);
		}
		else {
			IAttributeModel attrModel = (IAttributeModel) model_.getAttributeModel(attrType);
			String attrIDStr = attrType.toString();
			if (attrModel == null)
				throw new IdASException("Entity with type " + getType().toString()
						+ " can not have an attribute with type " + attrIDStr);
			int maxCardinality = model_.getMaxCardinality(attrType);
			Attribute attr = null;
			if (maxCardinality == 1)
				attr = new SingleValuedAttribute(context_, this, attrModel);
			else
				attr = new Attribute(context_, this, attrModel);
			attributes_.put(attrType, attr);
			return attr;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttributes()
	 */
	public Iterator getAttributes() throws IdASException {
		List attrs = new ArrayList();
		Iterator itr = model_.getAttributeModels().iterator();
		while (itr.hasNext()) {
			IAttributeModel attrModel = (IAttributeModel) itr.next();
			URI attrType = attrModel.getType();
			IAttribute attr = getAttribute(attrType);
			attrs.add(attr);
		}
		return attrs.iterator();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.api.IHasAttributes#getSingleValuedAttribute(
	 * java.net.URI)
	 */
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException {
		String idURI = attrID.toString();
		IAttribute attr = getAttribute(attrID);
		if (attr == null)
			return null;
		if (attr instanceof ISingleValuedAttribute == false)
			throw new IdASException("Attribute " + idURI + " is not an instance of ISingleValuedAttribute");
		return (ISingleValuedAttribute) attr;
	}

//	/*
//	 * (non-Javadoc)
//	 * 
//	 * @see java.lang.Object#equals(java.lang.Object)
//	 */
//	public boolean equals(Object obj) {
//		if (obj == null)
//			return false;
//		if (obj == this)
//			return true;
//		if (obj instanceof Entity == false)
//			return false;
//		Entity nd = (Entity) obj;
//		return entity_.equals(nd.getHibernateEntity());
//	}
//
//	/*
//	 * (non-Javadoc)
//	 * 
//	 * @see
//	 * org.eclipse.higgins.idas.api.IHasAttributes#equals(org.eclipse.higgins
//	 * .idas.api.IHasAttributes)
//	 */
//	public boolean equals(IHasAttributes attributes) throws IdASException {
//		return equals((Object) attributes);
//	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.api.IHasAttributes#removeAttribute(java.net.URI)
	 */
	public void removeAttribute(URI attrID) throws IdASException {
		IAttribute atr = getAttribute(attrID);
		if (atr != null)
			atr.remove();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.api.IHasAttributes#removeAttributeValue(java
	 * .net.URI, java.lang.Object)
	 */
	public void removeAttributeValue(URI attrID, Object value) throws IdASException {
		throw new NotImplementedException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.api.IHasAttributes#removeAttributeValue(org.
	 * eclipse.higgins.idas.api.IAttribute)
	 */
	public void removeAttributeValue(IAttribute attr) throws IdASException {
		throw new NotImplementedException();
	}

	public boolean isProxy() throws IdASException {
		return false;
	}

	public IValueModel getValueModel() {
		return model_;
	}

	public boolean isSimple() throws IdASException {
		return false;
	}

	public IEntity addAttributeValue(URI attrType, String entityId) throws IdASException {
		throw new  IdASException("Not implemented.");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.hb.IStatefulObject#getParentObject()
	 */
	public IStatefulObject getParentObject() {
		return (attributeOwner_ != null) ? attributeOwner_.getOwner() : null;
	}

	public void commitState() {
		if (attributeOwner_ != null) {
			if (IStatefulObject.PRE_DELETED_NEW_OBJ.equals(state) || IStatefulObject.PRE_DELETED_STORED_OBJ.equals(state)) {
				attributeOwner_.removeValue(this);
			}
		}
		super.commitState();
	}

	public String toString() {
		return "" + entity_.getEntityName() + ", ID : " + entityId_; 
	}

}
