/**
 * Copyright (c) 2009 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */
package org.eclipse.higgins.idas.cp.hb.impl.entity;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.hb.IHibernateEntity;

/**
 * Hibernate entity wrapper for MAP entity mode
 * 
 */
public class MapHibernateEntity implements IHibernateEntity {
	private static Log log = LogFactory.getLog(MapHibernateEntity.class);

	public final static String TYPE_PROPERTY = "$type$";
	protected String name;
	protected Map persistentMap;
	protected Map childEntities;

	public MapHibernateEntity(Map map, String name) throws IdASException {
		if (name == null) {
			if (map == null || !map.containsKey(TYPE_PROPERTY))
				throw new IdASException("Could not get the type of entity.");
			this.name = (String) map.get(TYPE_PROPERTY);
		} else {
			this.name = name;
		}
		if (map == null) {
			persistentMap = new HashMap();
			if (name != null) {
				persistentMap.put(MapHibernateEntity.TYPE_PROPERTY, name);
			}
		} else {
			persistentMap = map;
		}
		childEntities = new HashMap();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.hb.IHibernateEntity#setObject(java.lang.String ,
	 *      java.lang.Object)
	 */
	public void setObject(String objectName, Object value) {
		persistentMap.put(objectName, value);
	}

	// public void addToCollection(String collectionName, IHibernateEntity
	// hibernateEntity) throws IdASException {
	// if (hibernateEntity instanceof MapHibernateEntity == false) {
	// throw new
	// IdASException("Parameter \"hibernateEntity\" should be an instance of "
	// + this.getClass().getName());
	// }
	// List list = getList(collectionName);
	// list.add(hibernateEntity);
	// }

	public synchronized IHibernateEntity buildCollectionEntity(String collectionName, String entityName, boolean singleValued)
			throws IdASException {
		List persistentList = getPersistentList(collectionName);
		List childEntities = getChildEntitiesList(collectionName);
		if (singleValued && persistentList.size() > 0)
			throw new IdASException("Collection already contains single value.");
		MapHibernateEntity me = new MapHibernateEntity(new HashMap(), entityName);
		childEntities.add(me);
		persistentList.add(me.getEntityObject());
		return me;
	}

	/**
	 * Create and return persistent list from persistent map
	 * 
	 * @param listName
	 * @return
	 * @throws IdASException
	 */
	private synchronized List getPersistentList(String attributeName) throws IdASException {
		List list = null;
		if (persistentMap.containsKey(attributeName)) {
			list = (List) persistentMap.get(attributeName);
		} else {
			list = new ArrayList();
			persistentMap.put(attributeName, list);
		}
		return list;
	}

	private void sendErrorDuplicateMemberOfCollection(Map collectionOwner, String collectionName) {
		try {
			String id = "UNKNOWN";
			String type = "UNKNOWN";
			if (collectionOwner.containsKey("id_")) {
				id = collectionOwner.get("id_").toString();
			}
			if (collectionOwner.containsKey(TYPE_PROPERTY)) {
				type = (String)collectionOwner.get(TYPE_PROPERTY);
			}
			log.error("!!!! ENTITY with TYPE = " + type + " and ID + " + id + " CONTAINS DUPLICATES IN LIST with name = " + collectionName);			
		}
		catch(Exception e) {
			log.error(e, e);
		}
	}

	/**
	 * @param persistObjects
	 * @param entities
	 * @throws IdASException
	 */
	private synchronized void populateChildEntitiesFromPersistObjects(List persistObjects, List entities, Map parentPersist,
			String collectionName) throws IdASException {
		Iterator itr = persistObjects.iterator();
		while (itr.hasNext()) {
			Map map = (Map) itr.next();
			if (map == null) {
				// the member of persist list can be empty if there is a
				// duplicates with the same LIST_ID value in database. Such
				// duplicates were created be some unknown reason using DOM4J mode
				sendErrorDuplicateMemberOfCollection(parentPersist, collectionName);

			} else {
				MapHibernateEntity me = new MapHibernateEntity(map, null);
				entities.add(me);
			}
		}
	}

	/**
	 * Create and return persistent list from persistent map
	 * 
	 * @param listName
	 * @return
	 * @throws IdASException
	 */
	private synchronized List getChildEntitiesList(String attributeName) throws IdASException {
		List list = null;
		if (childEntities.containsKey(attributeName)) {
			list = (List) childEntities.get(attributeName);
		} else {
			list = new ArrayList();
			List persist = getPersistentList(attributeName);
			populateChildEntitiesFromPersistObjects(persist, list, persistentMap, attributeName);
			childEntities.put(attributeName, list);
		}
		return list;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.hb.IHibernateEntity#getCollection(java.lang
	 *      .String)
	 */
	public List getCollection(String collectionName) throws IdASException {
		return getChildEntitiesList(collectionName);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.hb.IHibernateEntity#getEntityObject()
	 */
	public Object getEntityObject() {
		return persistentMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.hb.IHibernateEntity#getObject(java.lang.String )
	 */
	public Object getObject(String collectionName) {
		if (persistentMap.containsKey(collectionName)) {
			return persistentMap.get(collectionName);
		} else
			return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.hb.IHibernateEntity#removeFromCollection(
	 *      java.lang.String, org.eclipse.higgins.idas.cp.hb.IHibernateEntity)
	 */
	public void removeFromCollection(String collectionName, IHibernateEntity hibernateEntity) throws IdASException {
		List persistentList = getPersistentList(collectionName);
		List childEntities = getChildEntitiesList(collectionName);
		persistentList.remove(hibernateEntity.getEntityObject());
		childEntities.remove(hibernateEntity);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.hb.IHibernateEntity#getEntityName()
	 */
	public String getEntityName() {
		return name;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((persistentMap == null) ? 0 : persistentMap.hashCode());
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		MapHibernateEntity other = (MapHibernateEntity) obj;
		if (persistentMap == null) {
			if (other.persistentMap != null)
				return false;
		} else if (!persistentMap.equals(other.persistentMap))
			return false;
		return true;
	}


	public String toString() {
		return name;
	}

}
