/**
 * Copyright (c) 2009 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */
package org.eclipse.higgins.idas.cp.hb.impl.entity;

import java.net.URI;
import java.nio.ByteBuffer;
import java.sql.Blob;
import java.util.Date;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.impl.dv.util.Base64;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.hb.IHibernateEntity;
import org.eclipse.higgins.idas.cp.hb.IHibernateEntityFactory;
import org.eclipse.higgins.idas.cp.hb.util.DatatypeMapper;
import org.hibernate.EntityMode;
import org.hibernate.lob.BlobImpl;

public class MapHibernateEntityFactory implements IHibernateEntityFactory {
	private static Log log = LogFactory.getLog(MapHibernateEntityFactory.class);

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.cp.hb.IHibernateEntityFactory#create(java.lang
	 * .Object, java.lang.String)
	 */
	public IHibernateEntity create(Object hibernateObject, String entityName) throws IdASException {
		try {
			return new MapHibernateEntity((Map) hibernateObject, entityName);
		}
		catch(ClassCastException e) {
			throw new IdASException("Persistent object expected to be Map", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.cp.hb.IHibernateEntityFactory#hibernateToIdas
	 * (java.lang.Object, java.net.URI)
	 */
	public Object hibernateToIdas(Object value, URI valueType) throws IdASException {
		if (value == null)
			return null;
		else {
			try {
				String hbType = DatatypeMapper.getHibernateType(valueType.toString());
				if (DatatypeMapper.STRING_TYPE.equals(hbType) || DatatypeMapper.TEXT_TYPE.equals(hbType)) {
					String val = (String) value;
					if (ITypedValue.ANYURI_TYPE_URI.equals(valueType)) {
						return new URI(val);
					} else if (ITypedValue.BASE64BINARY_TYPE_URI.equals(valueType)) {
						if (val == null) {
							return null;
						} else {
							byte[] data = Base64.decode(val);
							ByteBuffer bb = ByteBuffer.allocate(data.length);
							bb.put(data);
							return bb;
						}
					} else
						return val;
				}
				if (DatatypeMapper.DATE_TYPE.equals(hbType)) {
					return (Date) value;
				}
				if (DatatypeMapper.TIMESTAMP_TYPE.equals(hbType)) {
					return (Date) value;
				}
				if (DatatypeMapper.TIME_TYPE.equals(hbType)) {
					return (Date) value;
				}
				if (DatatypeMapper.BINARY_TYPE.equals(hbType) || DatatypeMapper.BLOB_TYPE.equals(hbType)) {
					Blob val = (Blob) value;
					byte[] data = (val != null) ? val.getBytes(1, (int)val.length()) : null;
					ByteBuffer bb = ByteBuffer.allocate(data.length);
					bb.put(data);
					return bb;
				}
				if (DatatypeMapper.INTEGER_TYPE.equals(hbType)) {
					return (Integer) value;
				}
				if (DatatypeMapper.BOOLEAN_TYPE.equals(hbType)) {
					return (Boolean) value;
				}
				if (DatatypeMapper.DOUBLE_TYPE.equals(hbType)) {
					return (Double) value;
				}
				if (DatatypeMapper.FLOAT_TYPE.equals(hbType)) {
					return (Float) value;
				}
				if (DatatypeMapper.LONG_TYPE.equals(hbType)) {
					return (Long) value;
				}
				if (DatatypeMapper.SHORT_TYPE.equals(hbType)) {
					return (Short) value;
				}
				if (DatatypeMapper.CHARACTER_TYPE.equals(hbType)) {
					return (String) value;
				}
			} catch (Exception e) {
				log.error(e, e);
				if (e instanceof IdASException)
					throw (IdASException) e;
				else
					throw new IdASException(e);
			}
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.cp.hb.IHibernateEntityFactory#idasToHibernate
	 * (java.lang.Object, java.net.URI)
	 */
	public Object idasToHibernate(Object obj, URI valueType) throws IdASException {
		if (obj instanceof ByteBuffer) {
			ByteBuffer bb = (ByteBuffer) obj;
			
			return new BlobImpl(bb.array());
		}
		if (obj instanceof URI) {
			return obj.toString();
		}
		return obj;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.idas.cp.hb.IHibernateEntityFactory#getEntityMode()
	 */
	public EntityMode getEntityMode() {
		return EntityMode.MAP;
	}
}
