/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */
package org.eclipse.higgins.idas.cp.hb.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentFactory;
import org.dom4j.Element;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.common.AuthNSelfIssuedMaterials;
import org.eclipse.higgins.idas.cp.hb.authentication.AuthConstants;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.ISimpleValueModel;
import org.hibernate.cfg.Configuration;

public class MappingBuilder {

	private static final Log log = LogFactory.getLog(MappingBuilder.class);

	public static String higgins_Entity = "higgins_Entity";

	public static String higgins_entityId = "higgins_entityId";

	private static String idGeneratorType_ = "native";

	private static String classIdName_ = "id_";

	private IContextModel model;

	private Properties props;

	private Configuration config;

	private TypeResolver typeResolver;
	
	public MappingBuilder(IContextModel model, Properties props) throws IdASException {
		this.model = model;
		this.props = props;
	}

	public void build() throws IdASException {
		typeResolver = new TypeResolver();
		config = new Configuration();
		config.setProperties(props);
		Document doc = buildHBMapping(model);
		doc.addDocType("hibernate-mapping", "-//Hibernate/Hibernate Mapping DTD 3.0//EN",
				"http://hibernate.sourceforge.net/hibernate-mapping-3.0.dtd");
		String s = doc.asXML();
		System.out.println(s);
		log.debug(s);
		config.addXML(s);
		config.buildMappings();
		config.buildSettings();
	}

	private Document buildHBMapping(IContextModel model) throws IdASException {
		DocumentFactory df = DocumentFactory.getInstance();
		Document doc = df.createDocument();
		Element mapping = doc.addElement(HBVocabulary.hibernate_mapping);
		addUserAccount(mapping);
		addEntities(mapping, model);
		return doc;
	}

	private ArrayList getBaseEntities(IContextModel contextModel) throws IdASException {
		ArrayList base = new ArrayList();
		List entities = contextModel.getEntityModels();
		for (int i = 0, s = entities.size(); i < s; i++) {
			IEntityModel em = (IEntityModel) entities.get(i);
			if (em.getSuperEntityModel() == null) {
				base.add(em);
			}
		}
		return base;
	}

	private void addEntities(Element root, IContextModel contextModel) throws IdASException {
		addRootEntity(root);
		List baseEntities = getBaseEntities(contextModel);
		for (int i = 0, s = baseEntities.size(); i < s; i++) {
			IEntityModel em = (IEntityModel) baseEntities.get(i);
			addEntity(root, em);
		}
	}

	private void addUserAccount(Element root) {
		Element userProfile = root.addElement(HBVocabulary.class_attr);
		userProfile.addAttribute(HBVocabulary.entity_name_attr, AuthConstants.USER_ACCOUNT);
		userProfile.addAttribute(HBVocabulary.node_attr, AuthConstants.USER_PROPERTY_NAME);
		Element id = userProfile.addElement(HBVocabulary.id);
		id.addAttribute(HBVocabulary.name_attr, classIdName_);
		id.addAttribute(HBVocabulary.type_attr, "long");
		Element generator = id.addElement(HBVocabulary.generator);
		generator.addAttribute(HBVocabulary.class_attr, idGeneratorType_);
		Element login = userProfile.addElement(HBVocabulary.property);
		login.addAttribute(HBVocabulary.name_attr, AuthConstants.USER_LOGIN);
		login.addAttribute(HBVocabulary.type_attr, "string");
		Element password = userProfile.addElement(HBVocabulary.property);
		password.addAttribute(HBVocabulary.name_attr, AuthConstants.USER_PASSWORD_HASH);
		password.addAttribute(HBVocabulary.type_attr, "string");
		Element userToken = userProfile.addElement(HBVocabulary.property);
		userToken.addAttribute(HBVocabulary.name_attr, AuthConstants.USER_TOKEN);
		userToken.addAttribute(HBVocabulary.type_attr, "string");
	}

	private void addRootEntity(Element root) throws IdASException {
		Element rootEntity = root.addElement(HBVocabulary.root_class);
		// TODO add to root element
		rootEntity.addAttribute(HBVocabulary.entity_name_attr, higgins_Entity);
		rootEntity.addAttribute(HBVocabulary.node_attr, higgins_Entity);
		Element id = rootEntity.addElement(HBVocabulary.id);
		id.addAttribute(HBVocabulary.name_attr, classIdName_);
		id.addAttribute(HBVocabulary.type_attr, "long");
		Element generator = id.addElement(HBVocabulary.generator);
		generator.addAttribute(HBVocabulary.class_attr, idGeneratorType_);
		//
		Element entityId = rootEntity.addElement(HBVocabulary.property);
		entityId.addAttribute(HBVocabulary.name_attr, higgins_entityId);
		entityId.addAttribute(HBVocabulary.type_attr, "string");
		entityId.addAttribute(HBVocabulary.unique_attr, "true");
		//
		Element user = rootEntity.addElement(HBVocabulary.many_to_one);
		user.addAttribute(HBVocabulary.name_attr, AuthConstants.USER_PROPERTY_NAME);
		user.addAttribute(HBVocabulary.entity_name_attr, AuthConstants.USER_ACCOUNT);
	}

	private void addEntity(Element root, IEntityModel entity) throws IdASException {
		Element entityElm = root.addElement(HBVocabulary.joined_subclass);
		entityElm.addAttribute(HBVocabulary.entity_name_attr, typeResolver.getHibernateType(entity));
		IEntityModel parentEntity = entity.getSuperEntityModel();
		String parentId = (parentEntity != null) ? typeResolver.getHibernateType(parentEntity) : higgins_Entity;
		entityElm.addAttribute(HBVocabulary.extends_attr, parentId);
		Element key = entityElm.addElement(HBVocabulary.key);
		key.addAttribute(HBVocabulary.column_attr, typeResolver.getHibernateType(entity) + "_id_");
		addAttributes(entityElm, entity, root);
		List subEntities = entity.getSubEntityModels();
		for (int i = 0, s = subEntities.size(); i < s; i++) {
			IEntityModel em = (IEntityModel) subEntities.get(i);
			addEntity(root, em);
		}
	}

	private void addAttributes(Element ownerElm, IEntityModel entityOwner, Element root) throws IdASException {
		Iterator attributes = entityOwner.getOwnAttributeModels().iterator();
		while (attributes.hasNext()) {
			IAttributeModel attr = (IAttributeModel) attributes.next();
			if (AuthNSelfIssuedMaterials.MANAGED_ATTR_URI.equals(attr.getType())) {
				typeResolver.setPpidOwnerEntityType(entityOwner.getType());
			}
			if (attr.isSimple()) {
				addSimpleAttribute(attr, ownerElm, entityOwner, root);
			} else {
				IEntityModel entityValue = (IEntityModel) attr.getValueModel();
				if (entityValue != null) {
					addComplexAttribute(attr, ownerElm, entityOwner, entityValue);
				} else {
					log.error("Attribute " + attr.getType().toString() + " has no value model.");
				}
			}
		}
	}

	private void addSimpleAttribute(IAttributeModel attr, Element entityElm, IEntityModel entity, Element root) throws IdASException {
		ISimpleValueModel svm = (ISimpleValueModel) attr.getValueModel();
		String attrHbType = typeResolver.getHibernateType(attr);
		if (attr.getAttributeMode() == IAttributeModel.SINGLE_VALUED_MODE) {
			Element prop = entityElm.addElement(HBVocabulary.property);
			prop.addAttribute(HBVocabulary.name_attr, attrHbType);
			prop.addAttribute(HBVocabulary.type_attr, DatatypeMapper.getHibernateType(svm.getType().toString()));
			if (svm.isUnique()) {
				prop.addAttribute(HBVocabulary.unique_attr, "true");
			}
			int len = svm.getLength();
			if (len > 0) {
				prop.addAttribute(HBVocabulary.length_attr, String.valueOf(len));
			}
		} else {
			Element list = entityElm.addElement(HBVocabulary.list);
			list.addAttribute(HBVocabulary.name_attr, attrHbType);
			Element key = list.addElement(HBVocabulary.key);
			key.addAttribute(HBVocabulary.column_attr, typeResolver.getHibernateType(entity) + "_id");
			Element listIndex = list.addElement(HBVocabulary.list_index);
			listIndex.addAttribute(HBVocabulary.column_attr, "LIST_INDEX");
			Element oneToMany = list.addElement(HBVocabulary.one_to_many);
			String simpleMultivaluedEntityName = attrHbType + TypeResolver.SIMPLE_VALUE_ENTITY_SUFIX;
			oneToMany.addAttribute(HBVocabulary.class_attr, simpleMultivaluedEntityName);
			addSimpleValueClass(svm, root, simpleMultivaluedEntityName);
		}
	}

	private Element getEntityByName(Element root, String entityName) throws IdASException {
		Iterator elements = root.elementIterator();
		while (elements.hasNext()) {
			Element elm = (Element) elements.next();
			String name = elm.attributeValue(HBVocabulary.entity_name_attr);
			if (entityName.equals(name))
				return elm;
		}
		return null;
	}

	private void addSimpleValueClass(ISimpleValueModel svm, Element root, String simpleMultivaluedEntityName) throws IdASException {
		Element elm = getEntityByName(root, simpleMultivaluedEntityName);
		if (elm == null) {
			Element valueList = root.addElement(HBVocabulary.root_class);
			valueList.addAttribute(HBVocabulary.entity_name_attr, simpleMultivaluedEntityName);
			valueList.addAttribute(HBVocabulary.node_attr, simpleMultivaluedEntityName);
			Element id = valueList.addElement(HBVocabulary.id);
			id.addAttribute(HBVocabulary.name_attr, classIdName_);
			id.addAttribute(HBVocabulary.type_attr, "long");
			Element generator = id.addElement(HBVocabulary.generator);
			generator.addAttribute(HBVocabulary.class_attr, idGeneratorType_);
			Element valueProp = valueList.addElement(HBVocabulary.property);
			valueProp.addAttribute(HBVocabulary.name_attr, "value");
			valueProp.addAttribute(HBVocabulary.type_attr, DatatypeMapper.getHibernateType(svm.getType().toString()));
			if (svm.isUnique()) {
				valueProp.addAttribute(HBVocabulary.unique_attr, "true");
			}
			int len = svm.getLength();
			if (len > 0) {
				valueProp.addAttribute(HBVocabulary.length_attr, String.valueOf(len));
			}
		} else {
			int newLen = svm.getLength();
			if (newLen > 0) {
				Element valueProp = elm.element(HBVocabulary.property);
				String oldLength = valueProp.attributeValue(HBVocabulary.length_attr);
				int oldLen = (oldLength != null) ? Integer.parseInt(oldLength) : 0;
				if (oldLen < newLen) {
					valueProp.addAttribute(HBVocabulary.length_attr, String.valueOf(newLen));
				}
			}
		}
	}

	// addComplexAttribute(attr, ownerElm, entityOwner, entityValue, root);

	private void addComplexAttribute(IAttributeModel atrr, Element entityOwnerElm, IEntityModel entityOwner, IEntityModel entityValue)
			throws IdASException {
		Element list = entityOwnerElm.addElement(HBVocabulary.list);
		list.addAttribute(HBVocabulary.name_attr, typeResolver.getHibernateType(atrr));
		Element key = list.addElement(HBVocabulary.key);
		key.addAttribute(HBVocabulary.column_attr, typeResolver.getHibernateType(entityOwner) + "_id");
		if (entityOwner.getMaxCardinality(atrr.getType()) == 1) {
			key.addAttribute(HBVocabulary.unique_attr, "true");
		}
		Element listIndex = list.addElement(HBVocabulary.list_index);
		listIndex.addAttribute(HBVocabulary.column_attr, "LIST_INDEX");
		Element oneToMany = list.addElement(HBVocabulary.one_to_many);
		oneToMany.addAttribute(HBVocabulary.class_attr, typeResolver.getHibernateType(entityValue));
	}

	public Configuration getConfiguration() {
		return config;
	}

	public TypeResolver getTypeResolver() {
		return typeResolver;
	}

}
