/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */
package org.eclipse.higgins.idas.cp.hb.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;

import com.hp.hpl.jena.ontology.CardinalityRestriction;
import com.hp.hpl.jena.ontology.MaxCardinalityRestriction;
import com.hp.hpl.jena.ontology.MinCardinalityRestriction;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.ontology.Restriction;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class ModelUtils {
	private static Log log = LogFactory.getLog(ModelUtils.class);

	public static boolean isClassRelative(OntClass ancestor, OntClass descendant) throws IdASException {
		if (ancestor.equals(descendant))
			return true;
		try {
			Iterator itr = descendant.listSuperClasses();
			while (itr.hasNext()) {
				OntClass cls = (OntClass) itr.next();
				if (isClassRelative(ancestor, cls))
					return true;
			}
		} catch (Exception e) {
			log.error(e,e);
			throw new IdASException(e.getMessage());
		}
		return false;
	}

	public static boolean isPropertyRelative(OntProperty ancestor, OntProperty descendant) throws IdASException {
		OntProperty tmpDescendant = descendant;
		try {
			while (tmpDescendant != null) {
				if (tmpDescendant.hasSuperProperty(ancestor, false))
					return true;
				tmpDescendant = tmpDescendant.getSuperProperty();
			}
		} catch (Exception e) {
			log.error(e,e);
			throw new IdASException(e.getMessage());
		}
		return false;
	}

	private static void fillListSubClasses(List list, OntClass cls) {
		ExtendedIterator ei = cls.listSubClasses();
		while (ei.hasNext()) {
			OntClass child = (OntClass)ei.next();
			if (child != null && !child.isRestriction()) {
				list.add(cls);
				fillListSubClasses(list, cls);
			}
		}
	}

	public static OntClass getSuperClass(OntClass clazz) throws IdASException {
		Iterator itr = clazz.listSuperClasses();
		while(itr.hasNext()) {
			OntClass cls = (OntClass) itr.next();
			if (cls.isRestriction() == false)
				return cls;
		}
		return null;
	}
	
	
	public static List getListSuperClasses(OntClass ownerClass)throws IdASException {
		List classesList = new ArrayList();
		OntClass superClass = getSuperClass(ownerClass);
		while (superClass != null) {
			classesList.add(superClass);
			superClass = getSuperClass(superClass);
		}
		return classesList;
	}

	public static boolean isPropertyOfClass(OntClass ownerClass, OntProperty property) throws IdASException {
		List classesList = new ArrayList();
		classesList.add(ownerClass);
		classesList.addAll(getListSuperClasses(ownerClass));
		for (int i = 0; i < classesList.size(); i++) {
			OntClass ontClass = (OntClass) classesList.get(i);
			ExtendedIterator ei = property.listDomain();
			while (ei.hasNext()) {
				Object entity = ei.next();
				if (entity instanceof OntClass) {
					OntClass domain = (OntClass) entity;
					if (ontClass.getURI().equals(domain.getURI()))
						return true;
				}
			}
		}
		return false;
	}

	public static ArrayList getPropertiesByDomain(OntModel model, OntClass domainClass) throws IdASException {
		ArrayList list = new ArrayList();
		List subClasses = getListSuperClasses(domainClass);
		Property domain = model.getProperty("http://www.w3.org/2000/01/rdf-schema#domain");
		for (int i = 0, size = subClasses.size(); i < size; i++) {
			OntClass cls = (OntClass) subClasses.get(i);
			if (cls != null) {
				StmtIterator ei = model.listStatements(null, domain, cls);
				while (ei.hasNext()) {
					OntProperty property = model.getOntProperty(ei.nextStatement().getSubject().getURI());
					list.add(property);
				}
			}
		}
		return list;
	}

	public static ArrayList getInheritedPropertiesByDomain(OntModel model, OntClass domainClass) throws IdASException {
		ArrayList list = new ArrayList();
		List subClasses = getListSuperClasses(domainClass);
//		subClasses.add(domainClass);
		for (int i = 0, size = subClasses.size(); i < size; i++) {
			OntClass cls = (OntClass) subClasses.get(i);
			list.addAll(getOwnPropertiesByDomain(model, cls)); 
		}
		return list;
	}

	public static ArrayList getOwnPropertiesByDomain(OntModel model, OntClass domainClass) throws IdASException {
		ArrayList list = new ArrayList();
		Property domain = model.getProperty("http://www.w3.org/2000/01/rdf-schema#domain");
		StmtIterator ei = model.listStatements(null, domain, domainClass);
		while (ei.hasNext()) {
			OntProperty property = model.getOntProperty(ei.nextStatement().getSubject().getURI());
			list.add(property);
		}
		return list;
	}

	public static int getMaxCardinality(OntClass cls, OntProperty attr) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getMaxCardinality");
		for (ExtendedIterator itr = cls.listSuperClasses(); itr.hasNext();) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				if (r.onProperty(attr)) {
					if (r.isCardinalityRestriction()) {
						CardinalityRestriction cr = r.asCardinalityRestriction();
						return cr.getCardinality();
					} else if (r.isMaxCardinalityRestriction()) {
						MaxCardinalityRestriction cr = r.asMaxCardinalityRestriction();
						return cr.getMaxCardinality();
					}
				}
			} else {
				int res = getMaxCardinality(c, attr);
				if (res != -1)
					return res;
			}
		}
		return -1;
	}

	public static int getMinCardinality(OntClass cls, OntProperty attr) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getMinCardinality");
		for (ExtendedIterator itr = cls.listSuperClasses(true); itr.hasNext();) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				if (r.onProperty(attr)) {
					if (r.isCardinalityRestriction()) {
						CardinalityRestriction cr = r.asCardinalityRestriction();
						return cr.getCardinality();
					} else if (r.isMinCardinalityRestriction()) {
						MinCardinalityRestriction cr = r.asMinCardinalityRestriction();
						return cr.getMinCardinality();
					}
				}
			}
		}
		return -1;
	}

	public static boolean isFunctionalProperty(OntModel model, String attributeType) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::isFunctionalProperty");
		ObjectProperty op = model.getObjectProperty(attributeType);
		if (op != null && op.isFunctionalProperty() == true)
			return true;
		else
			return false;
	}

	public static String getHibernateName(OntModel model, OntResource res) {
		String ns = res.getNameSpace();
		String prf = model.getNsURIPrefix(ns);
		String loc = res.getLocalName();
		String val = (prf + "_" + loc).replaceAll("-", "_");
		log.info(val);
		return val;
	}

}
