/**
 * Copyright (c) 2006-2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Tom Doman
 *		Jim Sermersheim
 *		Duane Buss
 */

package org.eclipse.higgins.idas.cp.inmem;

import java.net.URI;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bandit.util.misc.CIStringKey;
import org.eclipse.higgins.audit.api.AuditEvents;
import org.eclipse.higgins.audit.api.AuditException;
import org.eclipse.higgins.audit.api.AuditOutcomes;
import org.eclipse.higgins.audit.api.AuditRecord;
import org.eclipse.higgins.configuration.common.ConfigurableComponentFactoryHelper;
import org.eclipse.higgins.idas.api.ContextNotOpenException;
import org.eclipse.higgins.idas.api.ContextOpenException;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidEntityIDException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.EntityExistsException;
import org.eclipse.higgins.idas.api.UnhandledExtensionException;
import org.eclipse.higgins.idas.spi.BasicContext;
import org.eclipse.higgins.idas.spi.IEntityContainer;
import org.eclipse.higgins.idas.spi.EntityNotification;
import org.eclipse.higgins.idas.common.ForceReauthenticationExtension;

/**
 * An implementation of a <a href="http://www.eclipse.org/higgins/">Higgins</a>.
 * Context Provider which is intended for use when Entities are being
 * manufactored on the fly or need to be cached privatly.
 * 
 * Please see the Samples section for one usage of creating Entities
 * from SAML assertions, then allowing interfaces which require Entities
 * to manipulate the InMemory Entity.
 * 
 * @author jimse@novell.com
 * @author tdoman@novell.com
 * @author dbuss@novell.com
 */

public class InMemoryContext extends BasicContext implements IEntityContainer
{
	private Log	 log = LogFactory.getLog(InMemoryContext.class.getName());
	private Log _auditLog = LogFactory.getLog( AuditEvents.HIGGINS_AUDIT_EVENT_LOGGER);
	private AuditRecord _auditRecord = null;

	private HashMap _idHashMap;
	private IContextId _contextID;
	private String _contextURIs = "";
	private String _contextTypes = "";

	private Map _contextSettings;

	/**
	 */
	public InMemoryContext(
			IContextId contextID) throws IdASException
	{
		// _env = new Hashtable<String, Object>();
		_contextID = contextID;
		_contextSettings = contextID.getConfiguration();

		URI contextURIs[] = _contextID.getUris();
		int iLoop;
		for (iLoop = 0; iLoop < contextURIs.length; ++iLoop)
			_contextURIs = _contextURIs + contextURIs[iLoop].toString() + ", ";

		String contextTypes[] = _contextID.getTypes();
		for (iLoop = 0; iLoop < contextTypes.length; ++iLoop)
			_contextTypes = _contextTypes + contextTypes[iLoop] + ", ";

		String strAuditRecordClass = (String)_contextSettings.get( "AuditRecordClass");
		if (strAuditRecordClass != null)
		{
			try
			{
				_auditRecord = (AuditRecord)ConfigurableComponentFactoryHelper.getInstanceFromClassName( strAuditRecordClass);
			}
			catch (Exception e)
			{
				throw new IdASException( e);
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#open(java.lang.Object, IExtension)
	 */
	public String open(
		Object authentication, 
		IExtension[] extensions) throws IdASException, ContextOpenException {
		if (isOpen(null))
			throw new ContextOpenException();
		
		if (extensions != null) {
			for (int index = 0; index < extensions.length; index++) {
				if (extensions[index].failIfUnsupported() == true) {
					if (!(extensions[index] instanceof ForceReauthenticationExtension)) {
						throw new UnhandledExtensionException(extensions[index].getClass().getName() + " not supported");
					}
				}
			}
		}

		return this.open(authentication);
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#open(java.lang.Object)
	 */
	public String open(
		Object unknown) throws IdASException
	{
		if (this.isOpen(null))
			throw new ContextOpenException();

		this.setOpen(true);

		// TODO: get and return entityID
		_emitAuditRecord(AuditEvents.AE_CREATE_SESSION,
			AuditOutcomes.AUDIT_OUT_SUCCESS, /* _getIdentity()*/ null, null, null);

		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#close()
	 */
	public void close() throws IdASException
	{
		if (!this.isOpen(null))
			throw new ContextNotOpenException();

		super.close();
		if (_idHashMap != null)
			_idHashMap.clear();
		_idHashMap = null;

		_emitAuditRecord(AuditEvents.AE_TERMINATE_SESSION,
			AuditOutcomes.AUDIT_OUT_SUCCESS, /* _getIdentity()*/ null, null, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#getSchema()
	 */
	public String getSchema() throws IdASException
	{
		if (!isOpen(null))
			throw new ContextNotOpenException();

		throw new NotImplementedException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#importData(java.lang.String,
	 *      java.lang.String)
	 */
	public void importData(
		String arg0,
		String arg1) throws IdASException
	{
		throw new NotImplementedException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#getEntity(java.lang.String)
	 */
	public IEntity getEntity(
		String entityID) throws IdASException
	{
		return getEntity(entityID, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#getEntity(java.lang.String,
	 *      java.util.Iterator)
	 */
	public IEntity getEntity(
		String entityID,
		Iterator attrSelectionList) throws IdASException
	{
		if (!isOpen(null))
			throw new ContextNotOpenException();

		CIStringKey idKey = new CIStringKey(entityID);

		// TODO: filter things not in attrSelectionList
		if (_idHashMap == null)
			throw new NoSuchEntityException("No such entity: " + idKey);

		IEntity ds = (IEntity) _idHashMap.get(idKey);
		if (ds == null)
			throw new NoSuchEntityException("No such entity: " + idKey);

		_emitAuditRecord(AuditEvents.AE_QUERY_ACCOUNT, AuditOutcomes.AUDIT_OUT_SUCCESS,
			/* _getIdentity()*/ null, entityID, null);

		return ds;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#getEntities(org.eclipse.higgins.idas.api.IFilter)
	 */
	public Iterator getEntities(
		IFilter filter) throws IdASException
	{
		_emitAuditRecord(AuditEvents.AE_QUERY_ACCOUNT, AuditOutcomes.AUDIT_OUT_SUCCESS,
			/* _getIdentity()*/ null, filter == null ? null : filter.toString(), null);
		return getEntities(filter, null);
	}

	/**
	 */
	public Iterator getEntities(
		IFilter filter,
		Iterator attrSelectionList) throws IdASException
	{
		if (!isOpen(null))
			throw new ContextNotOpenException();

		throw new NotImplementedException();
	}

	/*
	 * 
	 */
	private void _addEntity(
		String entityID,
		IEntity ds)
	{

		CIStringKey idKey = new CIStringKey(entityID);
		if (_idHashMap == null)
			_idHashMap = new HashMap();
		_idHashMap.put(idKey, ds);
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.higgins.idas.spi.BasicContext# addEntity(java.net.URI, java.lang.String)
	 */
	public IEntity addEntity(URI type, String entityID) throws IdASException, InvalidTypeException, InvalidEntityIDException, EntityExistsException 
	{
		if (!isOpen(null))
			throw new ContextNotOpenException();
		
		IEntity ds = new InMemoryEntity(this, type, entityID);

		_addEntity(entityID, ds);
		return ds;
	}
	
	public void applyUpdates() throws IdASException
	{
		// Objects are in-mem and always updated real-time
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#verifyEntityAttributes(java.lang.String,
	 *      java.util.Iterator)
	 */
	public boolean verifyEntityAttributes(
		String arg0,
		Iterator arg1) throws IdASException
	{
		if (!isOpen(null))
			throw new ContextNotOpenException();

		throw new NotImplementedException();
	}

	public void updateNotification(EntityNotification entityNotif) throws IdASException
	{
		if (entityNotif.getAction().equals(EntityNotification.UPDATE_REMOVE))
		{
			CIStringKey idKey = new CIStringKey(entityNotif.getEntity().getEntityID());
			if (_idHashMap != null)
				_idHashMap.remove(idKey);
		}
		// AFAIK, we don't need to deal with any other update notifications
	}

	/**
	 * @throws IdASException
	 */
	private void _emitAuditRecord(
		int iEventNumber,
		int iOutcome,
		String sInitiatorInfo,
		String sTargetInfo,
		String sEventInfo) throws IdASException
	{
		if (_auditRecord != null)
		{
			try
			{
				_auditRecord.clearRecord();
				_auditRecord.setOriginatorInfo( "Higgins IdAS JNDI Context Provider", null, null,
									null, null, null);
				if (sTargetInfo == null)
					sTargetInfo = "";
				if (sEventInfo == null)
					sEventInfo = "";
				_auditRecord.setEventNumber( iEventNumber);
				_auditRecord.setOutcome( iOutcome);
				_auditRecord.setInitiatorInfo( null, null, sInitiatorInfo);
				_auditRecord.setTargetInfo(null, _contextURIs, _contextTypes, null, null, sTargetInfo);
				_auditLog.info( _auditRecord);
			}
			catch (AuditException e)
			{
				throw new IdASException(e);
			}
		}
	}
}
