/**
 * Copyright (c) 2006-2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Tom Doman
 *		Jim Sermersheim
 */

package org.eclipse.higgins.idas.cp.jndi;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Iterator;
import java.util.Vector;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchResult;

import org.apache.log4j.Logger;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.INode;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.spi.AttributeNotification;
import org.eclipse.higgins.idas.spi.BasicNode;
import org.eclipse.higgins.idas.spi.INodeContainer;
import org.eclipse.higgins.idas.spi.MetadataNotification;
import org.eclipse.higgins.idas.spi.BasicAttribute;
import org.eclipse.higgins.idas.spi.BasicAttributeSet;
import org.eclipse.higgins.idas.spi.BasicSimpleValue;
import org.eclipse.higgins.idas.spi.NodeNotification;
import org.eclipse.higgins.idas.api.model.INodeModel;
import org.eclipse.higgins.util.idas.cp.BasicAttributePDP;
import org.eclipse.higgins.util.idas.cp.BasicAttributePDPIter;
import org.eclipse.higgins.util.idas.cp.IAttributePDPs;

/**
 * 
 * @author tdoman@novell.com
 * @author jimse@novell.com
 * 
 */

public class JNDINode extends BasicNode implements INode
{
	private Logger _log = Logger.getLogger(JNDINode.class.getName());
	private JNDIContext _context;
	private SearchResult _jndiResult;
	private IAttributePDPs _attrPDPs;
	private String _nodeID;
	private URI _type;
	private INodeContainer _container;
	private Vector _readListJNDI = new Vector();
	private boolean _bReadAllJNDI = false;
	private String _searchBase;

	/**
	 * 
	 * @param jndiContext
	 * @param jndiResult
	 * @param attrPDPs
	 * @throws IdASException
	 */
	public JNDINode(
		JNDIContext jndiContext,
		SearchResult jndiResult,
		String searchBase,
		IAttributePDPs attrPDPs) throws IdASException
	{
		super(jndiContext);
		_context = jndiContext;
		_container = jndiContext;
		_jndiResult = jndiResult;
		_searchBase = searchBase;
		_attrPDPs = attrPDPs;
	}

	/**
	 * 
	 * @param jndiContext
	 * @param type
	 * @param nodeID
	 * @param attrPDPs
	 * @throws IdASException
	 */
	public JNDINode(
		JNDIContext jndiContext,
		URI type,
		String nodeID,
		IAttributePDPs attrPDPs) throws IdASException
	{
		super(jndiContext, type, nodeID);
		_context = jndiContext;
		_container = jndiContext;
		_nodeID = _context.consumerNodeIDToProvider(nodeID);
		_type = type;
		_attrPDPs = attrPDPs;
	}

	/**
	 * 
	 * @param jndiContext
	 * @param copyFrom
	 * @param attrPDPs
	 * @throws IdASException
	 */
	public JNDINode(
		JNDIContext jndiContext,
		INode copyFrom,
		IAttributePDPs attrPDPs) throws IdASException
	{
		super(jndiContext, copyFrom.getNodeType(), copyFrom.getNodeID());
		_context = jndiContext;
		_container = jndiContext;
		_nodeID = _context.consumerNodeIDToProvider(copyFrom.getNodeID());
		_type = copyFrom.getNodeType();
		_attrPDPs = attrPDPs;
	}

	/**
	 * 
	 */
	private String _getNameInNamespace(
		SearchResult jndiResult) // throws IdASException
	{
		String retStr;
//		try
//		{
			retStr = _jndiResult.getName();
			if (retStr.length() == 0)
				retStr = _searchBase;
			else if (_searchBase.length() > 0)
				retStr = retStr + "," + _searchBase;
			// HACK: composeName does not work to create full names in 1.4 for some reason and
			//		 getNameInNamespace is only in Java 1.5.  LDAP specific hack coded above.
			// retStr = _context.getDirContext().composeName(_jndiResult.getName(), _searchBase);
			
			/* Here's something we should try
			 * NameParser parser = _context.getDirContext().getNameParser(_jndiResult.getName());
			 * parser.add(_searchBase);
			 * return parser.toString();
 			 */
//		}
//		catch (NamingException e)
//		{
//			throw new IdASException(e);
//		}
		return retStr;
	}

	/**
	 */
	public String getNodeID() throws IdASException
	{
		if (_nodeID == null)
		{
			if (_jndiResult != null)
				_nodeID = _context.providerNodeIDToConsumer(_getNameInNamespace(_jndiResult));
			else
				throw new IdASException("No node ID is set.");
		}
		return _nodeID;
	}

	/**
	 */
	public URI getNodeType() throws IdASException
	{
		if (_type == null)
		{
			if (_jndiResult != null)
				_type = _context.getSearchResultType(_jndiResult);
			else
				throw new IdASException("No node type set.");
		}
		return _type; 
	}

	/**
	 */
	public Iterator getAttributes() throws IdASException
	{
		BasicAttributeSet attrSet = this.getBasicAttributeSet();		
		boolean bHasItems = attrSet.getAttributes().hasNext();

		if (!_bReadAllJNDI  && (_jndiResult != null))
		{
			NamingEnumeration jndiEnum;
			Attributes jndiAttrs = _jndiResult.getAttributes();
			jndiEnum = jndiAttrs.getAll();
			try
			{
				while (jndiEnum.hasMore())
				{
					Attribute jndiAttr = (Attribute) jndiEnum.next();
					NamingEnumeration jndiAttrValEnum = jndiAttr.getAll();
					URI attrURI = new URI(jndiAttr.getID());
					BasicAttribute attr = null;
	
					if (!bHasItems || (attrSet.getAttribute(attrURI) == null))
					{
						// TODO: Map based on syntaxes.
						attr = attrSet.addUnnotifiedAttribute(attrURI);
						while (jndiAttrValEnum.hasMore())
						{
							Object jndiValue = jndiAttrValEnum.next();
							if (jndiValue instanceof String)
							{
								_log.debug("Attr: " + jndiAttr.getID() + " is a String");
								attr.addSimpleValue(ISimpleAttrValue.STRING_TYPE_URI, (String) jndiValue);
							}
							else if (jndiValue instanceof byte[])
							{
								_log.debug("Attr: " + jndiAttr.getID() + " is a byte[]");
								attr.addSimpleValue(ISimpleAttrValue.BASE64BINARY_TYPE_URI,
									(byte[]) jndiValue);
							}
							else
							{
								_log.debug(jndiValue.getClass().toString());
								attr.addSimpleValue(new URI(BasicSimpleValue.ATTR_VALUE_TYPE_URI_STR), jndiValue);
							}
						}
						attr.setContainer(attrSet);
						_readListJNDI.add(attrURI);
					}
				}
				_bReadAllJNDI = true;
			}
			catch (NamingException e)
			{
				throw new IdASException(e);
			}
			catch (URISyntaxException e)
			{
				throw new IdASException(e);
			}
		}

		/* No support for metadata PDPs. */
		return new BasicAttributePDPIter(_attrPDPs, null, null, attrSet.getAttributes(), true);
	}

	/**
	 */
	public IAttribute getAttribute(
		URI consumerAttrName) throws IdASException
	{
		IAttribute retAttr = null;
		BasicAttributeSet attrSet = this.getBasicAttributeSet();

		out: try
		{
			Iterator itr = _attrPDPs.consumerIDToProviders(consumerAttrName);
			URI providerAttrName = null;

			while (itr.hasNext())
			{
				providerAttrName = (URI) itr.next();
				if (_readListJNDI.contains(providerAttrName))
				{
					retAttr = attrSet.getAttribute(providerAttrName);
					_log.debug(providerAttrName + " already placed attrSet, container: " + ((BasicAttribute)retAttr).getContainer());
					break out;
				}
				else if ((retAttr = attrSet.getAttribute(providerAttrName)) != null)
				{
					_log.debug(providerAttrName + " already added attrSet, container: " + ((BasicAttribute)retAttr).getContainer());
					break out;
				}
				else if (_jndiResult != null)
				{
					BasicAttribute attr = null;
					Attributes jndiAttrs = _jndiResult.getAttributes();
					NamingEnumeration jndiEnum = jndiAttrs.getAll();
					while (jndiEnum.hasMore())
					{
						Attribute jndiAttr = (Attribute) jndiEnum.next();
						URI attrURI = new URI(jndiAttr.getID());

						if (providerAttrName.compareTo(attrURI) == 0)
						{
							NamingEnumeration jndiAttrValEnum = jndiAttr.getAll();
							attr = attrSet.addUnnotifiedAttribute(attrURI);
							while (jndiAttrValEnum.hasMore())
							{
								Object jndiValue = jndiAttrValEnum.next();
								if (jndiValue instanceof String)
								{
									_log.debug("Attr: " + jndiAttr.getID() + " is a String");
									attr.addSimpleValue(ISimpleAttrValue.STRING_TYPE_URI,
										(String) jndiValue);
								}
								else if (jndiValue instanceof byte[])
								{
									_log.debug("Attr: " + jndiAttr.getID() + " is a byte[]");
									attr.addSimpleValue(ISimpleAttrValue.BASE64BINARY_TYPE_URI,
										(byte[]) jndiValue);
								}
								else
								{
									_log.debug(jndiValue.getClass().toString());
									attr.addSimpleValue(new URI(BasicSimpleValue.ATTR_VALUE_TYPE_URI_STR), jndiValue);
								}
							}
							attr.setContainer(attrSet);
							retAttr = attr;
							_readListJNDI.add(attrURI);
							_log.debug("Placed [" + attr.getAttrID().toString() + "] into attrSet, container: " + ((BasicAttribute)retAttr).getContainer());
							break out;
						}
					}
				}
			}
		}
		catch (NamingException e)
		{
			throw new IdASException(e);
		}
		catch (URISyntaxException e)
		{
			throw new IdASException(e);
		}

		if (retAttr != null)
			return new BasicAttributePDP(_attrPDPs, null, null, retAttr, true);
		else
			return retAttr;
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(org.eclipse.higgins.idas.api.IAttribute)
	 */
	public IAttribute addAttribute(
		IAttribute copyFrom) throws IdASException
	{
		BasicAttributeSet attrSet = this.getBasicAttributeSet();
		BasicAttributePDP attrPDP = new BasicAttributePDP(_attrPDPs, null, null, copyFrom, false);
		IAttribute attr = attrSet.getAttribute(attrPDP.getAttrID()); 
			
		if (attr != null)
			return attr;
		else
			return super.addAttribute(attrPDP);
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(java.net.URI)
	 */
	public IAttribute addAttribute(
		URI type) throws IdASException, InvalidTypeException
	{
		Iterator itr = _attrPDPs.consumerIDToProviders(type);
		BasicAttributeSet attrSet = this.getBasicAttributeSet();
		URI providerType = (URI)itr.next();
		IAttribute attr = attrSet.getAttribute(providerType);

		if (attr != null)
			return attr;
		else
			return super.addAttribute(providerType);
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#removeAttribute(java.net.URI)
	 */
	public void removeAttribute(URI attrID) throws IdASException {
		Iterator itr = _attrPDPs.consumerIDToProviders(attrID);
		if (itr != null) {
			URI providerType = (URI)itr.next();
			super.removeAttribute(providerType);
		} else {
			super.removeAttribute(attrID);
		}
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#removeAttributeValue(IAttribute)
	 */
	public void removeAttributeValue(IAttribute attr) throws IdASException {
		BasicAttributePDP attrPDP = new BasicAttributePDP(_attrPDPs, null, null, attr, false);
		super.removeAttributeValue(attrPDP);
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#removeAttributeValue(URI, Object)
	 */
	public void removeAttributeValue(URI attrID, Object value) throws IdASException {
		Iterator itr = _attrPDPs.consumerIDToProviders(attrID);
		if (itr != null) {
			URI providerType = (URI)itr.next();
			super.removeAttributeValue(providerType, value);
		} else {
			super.removeAttributeValue(attrID, value);
		}
	}


	

	/**
	 */
	public INodeModel getModel() throws IdASException
	{
		throw new NotImplementedException();
	}

	public void remove() throws IdASException
	{
		if (_container != null)
		{
			_container.updateNotification(new NodeNotification(this, NodeNotification.UPDATE_REMOVE,
				null, null));
		}
	}

	public void updateNotification(
		AttributeNotification attrNotif) throws IdASException
	{
		_log.debug("Received attribute update notification - " + attrNotif.getAction() + ", " + attrNotif.getAttr().getAttrID().toString());
		_log.debug("_containter " + ((_container != null) ? _container.toString() : "null"));		
		if (_container != null)
		{
			_container.updateNotification(new NodeNotification(this, NodeNotification.UPDATE_ATTR_NOTIFY,
				attrNotif, null));
		}
	}

	public void updateNotification(
		MetadataNotification metaNotif) throws IdASException
	{
		if (_container != null)
		{
			_container.updateNotification(new NodeNotification(this, NodeNotification.UPDATE_METADATA_NOTIFY,
				null, metaNotif));
		}
	}

}
