/**
 * Copyright (c) 2009 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */
package org.eclipse.higgins.idas.cp.model;

import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.cache.CacheProviderFactory;
import org.eclipse.higgins.cache.api.ICache;
import org.eclipse.higgins.cache.api.key.CacheKey;
import org.eclipse.higgins.cache.nocache.NOCache;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.model.impl.ContextModel;
import org.eclipse.higgins.idas.cp.model.util.HigginsVocabulary;

import com.hp.hpl.jena.ontology.OntDocumentManager;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntModelSpec;
import com.hp.hpl.jena.rdf.model.ModelFactory;

public class ContextModelFactory {

	private static ContextModelFactory instance = new ContextModelFactory();
	private static ICache contextModelCache = CacheProviderFactory.getCacheProvider().getCache("ContextModelFactory.ContextModels");
	private static boolean isNoCache = contextModelCache instanceof NOCache;

	private Object cacheLock = "";

	private static Log log = LogFactory.getLog(ContextModelFactory.class);

	private static synchronized void initSchema(String resourceName, String schemaURI) {
		OntDocumentManager odm = OntDocumentManager.getInstance();
		if (odm.getModel(schemaURI) == null) {
			try {
				OntModel cachedModel = ModelFactory.createOntologyModel(OntModelSpec.OWL_MEM);
				InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(resourceName);
				if (is != null) {
					try {
						cachedModel.read(is, schemaURI);
					} finally {
						is.close();
					}
					if (!cachedModel.isEmpty()) {
						odm.addModel(schemaURI, cachedModel);
					}
				}
			} catch (Exception e) {
				log.error(e, e);
			}
		}
	}

	public static synchronized ContextModelFactory getInstance() {
		if (instance == null)
			instance = new ContextModelFactory();
		return instance;
	}

	public ContextModel getContextModel(URI schemaURI, String fileName) throws IdASException {
		if (isNoCache)
			return initContextModel(fileName);
		else {
			synchronized(cacheLock) {
				CacheKey key = new CacheKey(schemaURI.toString());
				if (contextModelCache.isKeyInCache(key)) {
					return (ContextModel) contextModelCache.get(key);
				} else {
					ContextModel cm = initContextModel(fileName);
					contextModelCache.put(key, cm);
					return cm;
				}
			}
		}
	}

	private ContextModel initContextModel(String fileName) throws IdASException {
		try {
			initSchema("higgins.owl", HigginsVocabulary.HIGGINS_BASE);
			OntModel model = ModelFactory.createOntologyModel(OntModelSpec.OWL_MEM);
			InputStream is = getClass().getResourceAsStream(fileName);
			if (is == null) {
				is = new FileInputStream(fileName);
			}
			try {
//				model.read(is, schemaURI.toString());
				model.read(is, null);
			} finally {
				is.close();
			}
			ContextModel cm = new ContextModel(model);
			model.close();
			return cm;
		} catch (Exception e) {
			throw new IdASException(e);
		}
	}

}
