/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.model.impl;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IHasAttributeModel;
import org.eclipse.higgins.idas.api.model.IValueModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;

public class AttributeModel implements IAttributeModel {

//	private static Log log = LogFactory.getLog(AttributeModel.class);

	private URI type;

	private boolean isSimple;

	private ArrayList<IValueModel> ownValues;

	private String shortType;

	private ArrayList<IHasAttributeModel> owners;

	private String attributeMode;

	private ArrayList<IAttributeModel> subAttributes;

	private IAttributeModel superAttribute;

	public AttributeModel(OntModel model, DatatypeProperty property) throws URISyntaxException {
		this(model, (OntProperty)property);
		this.isSimple = true;
	}

	public AttributeModel(OntModel model, ObjectProperty property) throws URISyntaxException {
		this(model, (OntProperty)property);
		this.isSimple = false;
	}

	private AttributeModel(OntModel model, OntProperty property) throws URISyntaxException {
		this.type = new URI(property.getURI());
		this.shortType = model.shortForm(property.getURI());
		owners = new ArrayList<IHasAttributeModel>();
		subAttributes = new ArrayList<IAttributeModel>();
		ownValues = new ArrayList<IValueModel>();
	}

	private void populateAllSubEntities(ArrayList<IHasAttributeModel> all, IEntityModel superEntity) throws IdASModelException {
		Iterator<IEntityModel> subEntities = superEntity.getSubEntityModels().iterator();
		while (subEntities.hasNext()) {
			IEntityModel subEntity = subEntities.next(); 
			all.add(subEntity);
			populateAllSubEntities(all, subEntity);
		}
	}

	private ArrayList<IHasAttributeModel> populateOwnersHierarchy(Iterator<IHasAttributeModel> hasAttrs) throws IdASModelException {
		ArrayList<IHasAttributeModel> all = new ArrayList<IHasAttributeModel>();
		while (hasAttrs.hasNext()) {
			IHasAttributeModel hasAttr = hasAttrs.next();
			all.add(hasAttr);
			if (hasAttr instanceof IEntityModel) {
				populateAllSubEntities(all, (IEntityModel)hasAttr);
			}
		}
		return all;
	}

	public String getAttributeMode() throws IdASModelException {
		boolean multyValuesMode = false; 
		boolean singleValueMode = false;
		ArrayList<IHasAttributeModel> ownersHierarchy = populateOwnersHierarchy(owners.iterator());
		Iterator<IHasAttributeModel> itr = ownersHierarchy.iterator();
		while (itr.hasNext()) {
			IHasAttributeModel ham = itr.next();
			int max = ham.getMaxCardinality(type);
			if (max == 1) {
				singleValueMode = true;
			}
			else {
				multyValuesMode = true; 
			}
		}
		if (singleValueMode && multyValuesMode)
			return MIXED_MODE;
		if (singleValueMode)
			return SINGLE_VALUED_MODE;
		return MULTY_VALUED_MODE;
	}

	public List<IHasAttributeModel> getOwners() {
		return owners;
	}

	public void addOwner(IHasAttributeModel owner) {
		owners.add(owner);
	}

	public IValueModel getValueModel() throws IdASModelException {
		List<IValueModel> values = getValueModels();
		switch (values.size()) {
			case 0:
				return null;
			case 1:
				return values.get(0);
			default:
				throw new IdASModelException("Attribute model has more than one value model.");
		}
	}

	public List<IValueModel> getValueModels() throws IdASModelException {
		HashSet<IValueModel> hs = new HashSet<IValueModel>();
		hs.addAll(getInheretedValueModels());
		hs.addAll(getOwnValueModels());
		return new ArrayList<IValueModel>(hs);
	}

	public boolean isSimple() {
		return isSimple;
	}

	public String getShortFormType() {
		return shortType;
	}

	public URI getType() {
		return type;
	}

	public List<IAttributeModel> getSubAttributes() {
		return subAttributes;
	}

	public IAttributeModel getSuperAttribute() {
		return superAttribute;
	}

	public List<IValueModel> getInheretedValueModels() throws IdASModelException {
		if (superAttribute != null)
			return superAttribute.getValueModels();
		else
			return Collections.emptyList();
	}

	public List<IValueModel> getOwnValueModels() {
		return ownValues;
	}

	public void addOwnValueModel(IValueModel model) {
		ownValues.add(model);
	}

	public void addSubAttributeModel(IAttributeModel model) {
		subAttributes.add(model);
	}

	public void setSuperAttributeModel(IAttributeModel model) {
		superAttribute = model;
	}

	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + ((type == null) ? 0 : type.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final AttributeModel other = (AttributeModel) obj;
		if (type == null) {
			if (other.type != null)
				return false;
		} else if (!type.equals(other.type))
			return false;
		return true;
	}

}
