/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.model.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;

import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;

public class EntityModel implements IEntityModel {
//	private static Log log = LogFactory.getLog(EntityModel.class);

	private URI type;

	private HashMap<URI, IAttributeModel> ownAttributes;

	private HashMap<URI, Integer> minCardinalityMap;

	private HashMap<URI, Integer> maxCardinalityMap;

	private String shortType;

	private EntityModel superModel;

	private ArrayList<IEntityModel> subModels;

	public EntityModel(OntModel model, OntClass entityClass) throws IdASException {
		type = URI.create(entityClass.getURI());
		shortType = model.shortForm(entityClass.getURI());
		ownAttributes = new HashMap<URI, IAttributeModel>();
		minCardinalityMap = new HashMap<URI, Integer>();
		maxCardinalityMap = new HashMap<URI, Integer>();
		subModels = new ArrayList<IEntityModel>();
	}

	public List<IEntityModel> getSubEntityModels() {
		return subModels;
	}
	
	public IEntityModel getSuperEntityModel() {
		return superModel;
	}

	public boolean isSimple() {
		return false;
	}

	public String getShortFormType() {
		return shortType;
	}

	public URI getType() {
		return type;
	}

	public IAttributeModel getAttributeModel(URI attrID) {
		IAttributeModel res = ownAttributes.get(attrID);
		if (res == null && superModel != null) {
			res = superModel.getAttributeModel(attrID);
		}
		return res;
	}

	public List<IAttributeModel> getAttributeModels() {
		HashSet<IAttributeModel> attrs = new HashSet<IAttributeModel>();
		attrs.addAll(getInheritedAttributeModels());
		attrs.addAll(getOwnAttributeModels());
		return new ArrayList<IAttributeModel>(attrs);
	}

	public List<IAttributeModel> getInheritedAttributeModels() {
		if (superModel == null) {
			return Collections.emptyList();
		} else {
			return superModel.getAttributeModels();
		}
	}

	public List<IAttributeModel> getOwnAttributeModels() {
		return new ArrayList<IAttributeModel>(ownAttributes.values());
	}

	public Integer getCurrentEntityMaxCardinality(URI attributeType) {
		return maxCardinalityMap.get(attributeType);
	}

	public Integer getCurrentEntityMinCardinality(URI attributeType) {
		return minCardinalityMap.get(attributeType);
	}

	public int getMaxCardinality(URI attributeType) {
		Integer crd = getCurrentEntityMaxCardinality(attributeType);
		if (crd == null && superModel != null) {
			crd = superModel.getCurrentEntityMaxCardinality(attributeType);
		}
		return (crd != null) ? crd.intValue() : 0;
	}

	public int getMinCardinality(URI attributeType) {
		Integer crd = getCurrentEntityMinCardinality(attributeType);
		if (crd == null && superModel != null) {
			crd = superModel.getCurrentEntityMinCardinality(attributeType);
		}
		return (crd != null) ? crd.intValue() : 0;
	}

	public void addSubEntityModel(EntityModel subEntity) {
		subModels.add(subEntity);
	}

	public void setSuperEntityModel(EntityModel superEntity) {
		superModel = superEntity;
	}

	public void addAttribute(IAttributeModel attribute) throws IdASModelException {
		ownAttributes.put(attribute.getType(), attribute);
	}

	public void addMaxCardinality(URI attrType, int maxCardinality) {
		maxCardinalityMap.put(attrType, new Integer(maxCardinality));
	}

	public void addMinCardinality(URI attrType, int minCardinality) {
		minCardinalityMap.put(attrType, new Integer(minCardinality));
	}

	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + ((type == null) ? 0 : type.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final EntityModel other = (EntityModel) obj;
		if (type == null) {
			if (other.type != null)
				return false;
		} else if (!type.equals(other.type))
			return false;
		return true;
	}

	// public EntityModel(ContextModel contextModel, OntClass entityClass)
	// throws Exception {
	// log.debug("TOP CLASS " + entityClass.getURI());
	// this.ctx = contextModel;
	// this.entityClass = entityClass;
	// initialize();
	// }
	//
	// public EntityModel(ContextModel contextModel, AttributeModel attrOwner,
	// OntClass entityClass) throws IdASException {
	// if (log.isDebugEnabled()) {
	// log.debug("ENTITY OWNER : " + attrOwner.getOwner().getType().toString() +
	// ", ATTRIBUTE: " + attrOwner.getType().toString() + ", ENTITY VALUE: " +
	// entityClass.getURI());
	// }
	//		
	// this.ctx = contextModel;
	// this.entityClass = entityClass;
	// this.attrOwner = attrOwner;
	// initialize();
	// }
	//
	// private void initialize() throws IdASException {
	// modelID = ModelUtils.getModelShortName(ctx.getOntModel(), entityClass);
	// initType();
	// initSubType();
	// initAttributes();
	// }
	//
	// private void initSubType() throws IdASException {
	// OntClass superClass = ModelUtils.getSuperClass(entityClass);
	// if (superClass != null) {
	// parentType = (superClass != null) ? superClass.getURI() :
	// ctx.getHigginsVocabulary().OWL_Thing;
	// }
	// }
	//
	// private void initType() throws IdASException {
	// try {
	// type = new URI(entityClass.getURI());
	// } catch (URISyntaxException e) {
	// log.error(e);
	// throw new IdASModelException("The entity class " + entityClass.getURI() +
	// " is not valid URI", e);
	// }
	// }
	//
	// private void initAttributes() throws IdASException {
	// attrMap = new Hashtable();
	// allAttrList = new ArrayList();
	// ArrayList own = ModelUtils.getOwnPropertiesByDomain(ctx.getOntModel(),
	// entityClass);
	// ownAttrList = initAttributes(own);
	// ArrayList inherited =
	// ModelUtils.getInheritedPropertiesByDomain(ctx.getOntModel(),
	// entityClass);
	// inheritedAttrList = initAttributes(inherited);
	// allAttrList.addAll(ownAttrList);
	// allAttrList.addAll(inheritedAttrList);
	// checkPPIDAttribute();
	// }
	//
	// private void checkPPIDAttribute() throws IdASException {
	// if (attrMap.containsKey(ModelConstants.PPID_PROPERTY_URI)) {
	// Object obj = attrMap.get(ModelConstants.PPID_PROPERTY_URI);
	// if (ownAttrList.contains(obj)) {
	// AttributeModel attrPPID = (AttributeModel) obj;
	// if (ctx.getPPIDAttribute() != null)
	// throw new IdASException("Attribute with type " +
	// ModelConstants.PPID_PROPERTY + " should be associated only for one entity
	// class.");
	// else
	// ctx.setPPIDAttribute(attrPPID);
	// }
	// }
	// }
	//
	// private ArrayList initAttributes(ArrayList properties) throws
	// IdASException {
	// ArrayList attributeModels = new ArrayList();
	// OntModel mdl = ctx.getOntModel();
	// for (int i = 0, size = properties.size(); i < size; i++) {
	// OntProperty p = (OntProperty) properties.get(i);
	// if (! ctx.existsEntityRelation(entityClass.getURI(), p.getURI())) {
	// ctx.addEntityRelation(entityClass.getURI(), p.getURI());
	// IAttributeModel m = null;
	// if (p.isDatatypeProperty()) {
	// DatatypeProperty dp = p.asDatatypeProperty();
	// m = new AttributeModel(ctx, dp, this);
	// attributeModels.add(m);
	// attrMap.put(m.getType(), m);
	// } else if (p.isObjectProperty()) {
	// ObjectProperty op = p.asObjectProperty();
	// try {
	// m = new AttributeModel(ctx, op, this);
	// attributeModels.add(m);
	// attrMap.put(m.getType(), m);
	// }
	// catch(ValueInitializationException e) {
	// log.warn("Attribute " + p.getURI() + " will be skipped. " +
	// e.getMessage(), e);
	// }
	// }
	// }
	// }
	// return attributeModels;
	// }
	//
	// public void initRelation() {
	//
	// }
	//
	// public void initCorelation() {
	//
	// }
	//
	// public Iterator getAttributeModels() {
	// return allAttrList.iterator();
	// }
	//
	// public Iterator getOwnAttributeModels() {
	// return ownAttrList.iterator();
	// }
	//
	// public Iterator getInheritedAttributeModels() {
	// return inheritedAttrList.iterator();
	// }
	//
	// public URI getType() {
	// return type;
	// }
	//
	// public IAttributeModel getAttributeModel(URI type) {
	// if (type == null)
	// return null;
	// if (attrMap.containsKey(type))
	// return (IAttributeModel) attrMap.get(type);
	// else
	// return null;
	// }
	//
	// public OntClass getOntClass() {
	// return entityClass;
	// }
	//
	// public String getId() {
	// return modelID;
	// }
	//
	// public boolean isSimple() {
	// return false;
	// }
	//
	// public List getSubEntityTypes() {
	// String tp = type.toString();
	// HashMap map = ctx.getEntitiesMap();
	// if (map.containsKey(tp))
	// return (List) map.get(tp);
	// else
	// return new ArrayList();
	// }
	//
	// public IDisplayData getDisplayData() throws IdASModelException {
	// return null;
	// }
	//
	// public String getParentType() {
	// return parentType;
	// }
	//
	// public AttributeModel getAttributeOwner() {
	// return attrOwner;
	// }

	// private class AttributeEntry {
	// AttributeModelExt model;
	// Integer maxCardinality;
	// Integer minCardinality;
	//
	// public AttributeEntry(AttributeEntry entry) {
	// this(entry.getModel(), entry.getMaxCardinality(),
	// entry.getMinCardinality());
	// }
	//
	// public AttributeEntry(AttributeModelExt model, int maxCardinality, int
	// minCardinality) {
	// this.model = model;
	// this.maxCardinality = maxCardinality;
	// this.minCardinality = minCardinality;
	// }
	//
	// public int getMaxCardinality() {
	// return maxCardinality;
	// }
	//
	// public void setMaxCardinality(Integer maxCardinality) {
	// this.maxCardinality = maxCardinality;
	// }
	//
	// public int getMinCardinality() {
	// return minCardinality;
	// }
	//
	// public void setMinCardinality(Integer minCardinality) {
	// this.minCardinality = minCardinality;
	// }
	//
	// public AttributeModelExt getModel() {
	// return model;
	// }
	//
	// public void setModel(AttributeModelExt model) {
	// this.model = model;
	// }
	// }

}
