/**
 * Copyright (c) 2009 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */
package org.eclipse.higgins.idas.cp.model.util;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;
import org.eclipse.higgins.idas.cp.model.impl.AttributeModel;
import org.eclipse.higgins.idas.cp.model.impl.EntityModel;
import org.eclipse.higgins.idas.cp.model.impl.SimpleValueModel;

import com.hp.hpl.jena.datatypes.RDFDatatype;
import com.hp.hpl.jena.ontology.CardinalityRestriction;
import com.hp.hpl.jena.ontology.DataRange;
import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.MaxCardinalityRestriction;
import com.hp.hpl.jena.ontology.MinCardinalityRestriction;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.ontology.Restriction;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.RDFList;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;

public class ContextBuilder {
	private static Log log = LogFactory.getLog(ContextBuilder.class);

	OntModel model;
	HigginsVocabulary vocabulary;
	HashMap<String, EntityModel> entityByType = new HashMap<String, EntityModel>();
	HashMap<String, AttributeModel> attributeByType = new HashMap<String, AttributeModel>();

	public ContextBuilder(OntModel model) throws IdASException, URISyntaxException {
		this.model = model;
		this.vocabulary = new HigginsVocabulary(model);
	}

	public void buildContext() throws IdASException, URISyntaxException {
		log.trace("ContextBuilder.buildContext()");
		parseEntities();
		buildEntiesHierarchy();
		parseSimpleAttributes();
		parseComplexAttributes();
		buildAttributesHierarchy();
		linkEntiesToAttributes();
		if (log.isDebugEnabled()) {
			printEntitiesDebugInfo();
		}
	}

	public HashMap<String, EntityModel> getEntityModels() {
		return entityByType;
	}

	public HashMap<String, AttributeModel> getAttributeModels() {
		return attributeByType;
	}

	private void parseEntities() throws IdASException {
		Iterator classes = model.listNamedClasses();
		log.debug("Initialize entities : ");
		while (classes.hasNext()) {
			OntClass cls = (OntClass) classes.next();
			log.debug("	" + cls.getURI());
			EntityModel em = new EntityModel(model, cls);
			entityByType.put(cls.getURI(), em);
		}
	}

	private void buildEntiesHierarchy() throws URISyntaxException, IdASException {
		Iterator classes = model.listNamedClasses();
		while (classes.hasNext()) {
			OntClass cls = (OntClass) classes.next();
			EntityModel model = entityByType.get(cls.getURI());
			setSuperEntityModel(cls, model);
			setSubEntityModels(cls, model);
			initCardinality(cls, model);
		}
	}

	private void setSuperEntityModel(OntClass cls, EntityModel model) throws IdASException {
		OntClass parent = ModelUtils.getSuperClass(cls);
		if (parent != null && !vocabulary.OWL_Thing.equals(parent.getURI())) {
			EntityModel superEntity = entityByType.get(parent.getURI());
			model.setSuperEntityModel(superEntity);
		}
	}

	private void setSubEntityModels(OntClass cls, EntityModel model) {
		Iterator subItr = cls.listSubClasses();
		while (subItr.hasNext()) {
			OntClass subClass = (OntClass) subItr.next();
			String subType = subClass.getURI();
			EntityModel subModel = entityByType.get(subType);
			model.addSubEntityModel(subModel);
		}
	}

	private void initCardinality(OntClass cls, EntityModel model) throws URISyntaxException, IdASModelException {
		Iterator itr = cls.listSuperClasses();
		while (itr.hasNext()) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				OntProperty onProp = r.getOnProperty();
				if (onProp == null) {
					throw new IdASModelException("Class " + cls.getURI() + " contains restriction to incorrect property.");
				}
				URI attrType = new URI(onProp.getURI());
				if (r.isCardinalityRestriction()) {
					CardinalityRestriction cr = r.asCardinalityRestriction();
					int crdValue = cr.getCardinality();
					model.addMaxCardinality(attrType, crdValue);
					model.addMinCardinality(attrType, crdValue);
				} else if (r.isMaxCardinalityRestriction()) {
					MaxCardinalityRestriction cr = r.asMaxCardinalityRestriction();
					model.addMaxCardinality(attrType, cr.getMaxCardinality());
				} else if (r.isMinCardinalityRestriction()) {
					MinCardinalityRestriction cr = r.asMinCardinalityRestriction();
					model.addMinCardinality(attrType, cr.getMinCardinality());
				}
			}
		}
	}

	private void parseComplexAttributes() throws URISyntaxException {
		log.debug("Initialize complex attributes:");
		Iterator objProps = model.listObjectProperties();
		while (objProps.hasNext()) {
			ObjectProperty prop = (ObjectProperty) objProps.next();
			if (log.isDebugEnabled()) {
				log.debug("		" + prop.getURI());
			}
			AttributeModel attr = new AttributeModel(model, prop);
			attributeByType.put(prop.getURI(), attr);
		}
	}

	private void parseSimpleAttributes() throws URISyntaxException {
		log.debug("Initialize simple attributes:");
		Iterator dtProps = model.listDatatypeProperties();
		while (dtProps.hasNext()) {
			DatatypeProperty prop = (DatatypeProperty) dtProps.next();
			if (log.isDebugEnabled()) {
				log.debug("		" + prop.getURI());
			}
			AttributeModel attr = new AttributeModel(model, prop);
			attributeByType.put(prop.getURI(), attr);
		}
	}

	private void buildAttributesHierarchy() {
		Iterator props = model.listOntProperties();
		while (props.hasNext()) {
			OntProperty prop = (OntProperty) props.next();
			AttributeModel attr = attributeByType.get(prop.getURI());
			OntProperty superProp = prop.getSuperProperty();
			if (superProp != null) {
				AttributeModel superAttr = attributeByType.get(superProp.getURI());
				attr.setSuperAttributeModel(superAttr);
			}
		}
	}

	private void setAttributeOwners(Iterator props) throws IdASModelException {
		while (props.hasNext()) {
			OntProperty prop = (OntProperty) props.next();
			String attrType = prop.getURI();
			AttributeModel attr = attributeByType.get(attrType);
			Iterator domens = prop.listDomain();
			while (domens.hasNext()) {
				Object obj = domens.next();
				if (obj instanceof OntClass) {
					OntClass domain = (OntClass) obj;
					String entityType = domain.getURI();
					EntityModel ownerEntity = entityByType.get(entityType);
					if (ownerEntity == null) {
						throw new IdASModelException("Property " + attrType + " has non-existent domain class " + entityType);
					}
					attr.addOwner(ownerEntity);
					ownerEntity.addAttribute(attr);
				}
			}
		}
	}

	private void linkEntiesToAttributes() throws IdASModelException {
		setAttributeOwners(model.listDatatypeProperties());
		setAttributeOwners(model.listObjectProperties());
		linkComplexValues();
		buildSimpleValues();
	}

	private void linkComplexValues() throws IdASModelException {
		Iterator props = model.listObjectProperties();
		while (props.hasNext()) {
			ObjectProperty prop = (ObjectProperty) props.next();
			AttributeModel attr = attributeByType.get(prop.getURI());
			Iterator ranges = prop.listRange();
			while (ranges.hasNext()) {
				OntResource range = (OntResource) ranges.next();
				if (range.isClass()) {
					String entityType = range.getURI();
					EntityModel em = entityByType.get(entityType);
					attr.addOwnValueModel(em);
				} else
					throw new IdASModelException("Property " + prop.getURI() + " has wrong range.");
			}
		}
	}

	private void buildSimpleValues() throws IdASModelException {
		Iterator props = model.listDatatypeProperties();
		while (props.hasNext()) {
			OntProperty prop = (OntProperty) props.next();
			String attrType = prop.getURI();
			if (log.isDebugEnabled()) {
				log.debug("Build simple value(s) for attribute " + attrType);
			}
			AttributeModel attr = attributeByType.get(attrType);
			Iterator ranges = prop.listRange();
			while (ranges.hasNext()) {
				OntResource range = (OntResource) ranges.next();
				SimpleValueModel sv = initByRange(prop.getURI(), range);
				attr.addOwnValueModel(sv);
			}
		}
	}

	private SimpleValueModel initByRange(String attrType, OntResource range) throws IdASModelException {
		if (range.isDataRange()) {
			return initAsOWL1_0(attrType, range);
		} else {
			Resource res = range.getRDFType();
			if (res != null) {
				return initAsOWL1_1(attrType, range);
			} else {
				SimpleValueModel sv = new SimpleValueModel();
				initType(attrType, sv, range.getURI());
				return sv;
			}
		}
	}

	private SimpleValueModel initAsOWL1_0(String attrType, OntResource r) throws IdASModelException {
		SimpleValueModel sv = new SimpleValueModel();
		sv.setOneOf(true);
		ArrayList<Object> oneOf = new ArrayList<Object>();
		sv.setOneOf(oneOf);
		DataRange dr = r.asDataRange();
		RDFList rdfList = dr.getOneOf();
		for (Iterator itr = dr.listOneOf(); itr.hasNext();) {
			Literal l = (Literal) itr.next();
			RDFDatatype t = l.getDatatype();
			initType(attrType, sv, t.getURI());
			oneOf.add(l.getValue());
		}
		return sv;
	}

	private SimpleValueModel initAsOWL1_1(String attrType, OntResource r) throws IdASModelException {
		SimpleValueModel sv = new SimpleValueModel();
		StmtIterator itr = r.listProperties();
		while (itr.hasNext()) {
			Statement st = itr.nextStatement();
			RDFNode node = st.getObject();
			Literal val = node.isLiteral() ? (Literal) node.as(Literal.class) : null;
			String prop = st.getPredicate().getURI();
			if (HigginsVocabulary.OWL11_onDataRange.equals(prop)) {
				if (node.isResource()) {
					Resource res = (Resource) node.as(Resource.class);
					initType(attrType, sv, res.getURI());
				}
			} else if (HigginsVocabulary.OWL11_pattern.equals(prop)) {
				sv.setPattern(val.getString());
			} else if (HigginsVocabulary.OWL11_maxLength.equals(prop)) {
				try {
					sv.setMaxLength(val.getInt());
				} catch (Exception e) {
					log.error("Can not parse maxLength for property " + attrType, e);
				}
			} else if (HigginsVocabulary.OWL11_minLength.equals(prop)) {
				try {
					sv.setMinLength(val.getInt());
				} catch (Exception e) {
					log.error("Can not parse maxLength for property " + attrType, e);
				}

			} else if (HigginsVocabulary.OWL11_length.equals(prop)) {
				try {
					sv.setLength(val.getInt());
				} catch (Exception e) {
					log.error("Can not parse maxLength for property " + attrType, e);
				}

			} else if (HigginsVocabulary.OWL11_totalDigits.equals(prop)) {
				try {
					sv.setTotalDigits(val.getInt());
				} catch (Exception e) {
					log.error("Can not parse totalDigits for property " + attrType, e);
				}
			} else if (HigginsVocabulary.OWL11_fractionDigits.equals(prop)) {
				try {
					sv.setFractionDigits(val.getInt());
				} catch (Exception e) {
					log.error("Can not parse fractionDigits for property " + attrType, e);
				}
			}
		}
		return sv;
	}

	private void initType(String attrType, SimpleValueModel sv, String typeUri) throws IdASModelException {
		URI type = sv.getType();
		if (type == null) {
			try {
				type = new URI(typeUri);
				sv.setType(type);
				String shortType = model.shortForm(typeUri);
				sv.setShortFormType(shortType);
			} catch (Exception e) {
				throw new IdASModelException(e);
			}
		}
		else if (! typeUri.equals(type.toString()))
			throw new IdASModelException("Datatype mismatch for enumerated values: Property=" + attrType + " Datatype1=" + type.toString() + " Datatype2=" + typeUri);
	}

	private class ModelComparator implements Comparator<IModel> {
		public int compare(IModel o1, IModel o2) {
			try {
				String type1 = o1.getType().toString();
				String type2 = o1.getType().toString();
				return type2.compareTo(type1);
			} catch (IdASException e) {
				log.error(e, e);
				return -1;
			}
		}
	}

	private void printEntitiesDebugInfo() throws IdASException {
		ArrayList<EntityModel> entityList = new ArrayList<EntityModel>(entityByType.values());
		Collections.sort(entityList, new ModelComparator());
		for (int i = 0, size = entityList.size(); i < size; i++) {
			EntityModel em = entityList.get(i);
			log.debug("ENTITY # " + (i+1));
			log.debug("	type: " + em.getType().toString());
			log.debug("	super entity : " + ((em.getSuperEntityModel() == null) ? "none" : em.getSuperEntityModel().getType().toString()));
			log.debug("	sub entities :");
			Iterator<IEntityModel> itrSub = em.getSubEntityModels().iterator();
			while (itrSub.hasNext()) {
				log.debug("		" + itrSub.next().getType().toString());
			}
			log.debug("	attributes :");
			Iterator<IAttributeModel> itrAttr = em.getOwnAttributeModels().iterator();
			while (itrAttr.hasNext()) {
				log.debug("		" + itrAttr.next().getType().toString());
			}
		}
	}

}
