/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Copied from org.eclipse.higgins.icard.provider.cardspace.zurich project by Sergey Lyakhov
 *
 * Contributors:
 *    Patrick R Wardrop (IBM Corporation)
 *  
 *******************************************************************************/
package org.eclipse.higgins.idas.cp.model.util;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

public class DateUtil {

	public static final String COMPLETE_DATE_HOURS_MINUTES_SECS_MSECS_Z = "yyyy-MM-dd'T'HH':'mm':'ss'.'S'Z'";

	public static String getIso8601Date(Date date) {
		if (date == null) {
			date = new Date();
		}

		SimpleDateFormat sdf = new SimpleDateFormat(COMPLETE_DATE_HOURS_MINUTES_SECS_MSECS_Z);

		sdf.setTimeZone(TimeZone.getTimeZone("GMT"));

		return sdf.format(date);
	}

	public static String getIso8601Date() {
		return getIso8601Date(null);
	}

	/**
	 * This method will parse iso8601 date formats. See
	 * http://www.w3.org/TR/NOTE-datetime
	 * 
	 * @param iso8601
	 * @return
	 */
	public static Date getDateFromIso8601(String iso8601) {
		Date result = null;

		if (iso8601 == null || iso8601.length() < 4) {
			return result;
		}

		int len = iso8601.length();
		int year = 0;
		int month = 0;
		int day = 0;
		int hour = 0;
		int minute = 0;
		int second = 0;
		int millis = 0;
		String timeZone = null;

		if (len == 4) {
			year = parseToInt(iso8601.substring(0, 4), 0);
		} else if (len == 7) {
			year = parseToInt(iso8601.substring(0, 4), 0);
			month = parseToInt(iso8601.substring(5, 7), 0);
		} else if (len == 10) {
			year = parseToInt(iso8601.substring(0, 4), 0);
			month = parseToInt(iso8601.substring(5, 7), 0);
			day = parseToInt(iso8601.substring(8, 10), 0);
		} else if (len > 15) {
			year = parseToInt(iso8601.substring(0, 4), 0);
			month = parseToInt(iso8601.substring(5, 7), 0);
			day = parseToInt(iso8601.substring(8, 10), 0);
			hour = parseToInt(iso8601.substring(11, 13), 0);
			minute = parseToInt(iso8601.substring(14, 16), 0);

			if (iso8601.charAt(16) == ':') {
				second = parseToInt(iso8601.substring(17, 19), 0);
				timeZone = iso8601.substring(19, len);
			} else {
				timeZone = iso8601.substring(16, len);
			}

			millis = parseOutMillisFromTimeZone(timeZone);

			timeZone = trimTimeZone(timeZone);
		}

		// the Calendar class expects January == 0.
		month = Math.max(0, month - 1);

		// days are never zero
		day = Math.max(1, day);

		if (timeZone == null) {
			int offset = TimeZone.getDefault().getRawOffset();
			timeZone = convertOffSetToTimeZone(offset);
		} else if (timeZone.equals("Z")) {
			timeZone = "GMT+00:00";
		} else {
			timeZone = "GMT" + timeZone;
		}

		return makeDate(year, month, day, hour, minute, second, millis, timeZone);
	}

	private static String convertOffSetToTimeZone(int offset) {

		StringBuffer result = new StringBuffer();
		result.append("GMT");
		boolean isPositive = true;
		if (offset < 0) {
			isPositive = false;
		}

		if (isPositive) {
			result.append("+");
		} else {
			result.append("-");
			offset = offset * -1;
		}

		int hour = offset / (1000 * 60 * 60);
		int minutes = offset / (1000 * 60 * 60 * 60);

		result.append(padder(hour, 2));
		result.append(":");
		result.append(padder(minutes, 2));

		return result.toString();
	}

	private static String padder(int value, int pads) {
		String v = Integer.toString(value);

		int numZeros = pads - v.length();

		StringBuffer result = new StringBuffer();
		if (numZeros > 0) {
			for (int i = 0; i < numZeros; i++) {
				result.append("0");
			}
		}

		result.append(v);

		return result.toString();
	}

	private static String trimTimeZone(String timeZone) {
		if (timeZone == null) {
			return null;
		}

		int beginIndex = 0;
		int len = timeZone.length();
		for (int i = 0; i < timeZone.length(); i++) {
			beginIndex = i;
			char c = timeZone.charAt(i);
			if (c == '+' || c == '-' || i > 4 || isAlphaChar(c)) {
				break;
			}
		}

		return timeZone.substring(beginIndex, len);
	}

	/**
	 * This method will take the fraction of a second and convert it to
	 * milliseconds
	 * 
	 * @param timeZone
	 *            this is the remainder of the time string i.e., ".62054Z" or
	 *            ".543Z"
	 * @return
	 */
	private static int parseOutMillisFromTimeZone(String timeZone) {

		if (timeZone == null || !timeZone.startsWith(".") || timeZone.length() < 4) {
			return 0;
		}

		//
		// Since the milliseconds can be specified as 1 to n chars we need to
		// get it manually.
		//
		StringBuffer millis = new StringBuffer();
		for (int i = 1; i < timeZone.length(); i++) {
			char c = timeZone.charAt(i);
			if (c == '+' || c == '-' || i > 4 || isAlphaChar(c)) {
				break;
			}

			millis.append(c);
		}

		int fs = parseToInt(millis.toString(), 0);

		int numDecimals = millis.length();

		int ms = 0;

		if (numDecimals > 0) {
			double x = Math.pow(10, numDecimals);
			float faction = fs / (float) x;
			ms = (int) ((float) 1000 * faction);
		}

		return ms;
	}

	private static boolean isAlphaChar(char c) {
		if (c > 64 && c < 91) {
			return true;
		} else {
			return false;
		}
	}

	public static Date makeDate(int year, int month, int day, int hour, int minute, int sec, int msec, String timeZone) {
		Calendar cal;

		if (timeZone != null) {
			TimeZone tzone = TimeZone.getTimeZone(timeZone);
			cal = Calendar.getInstance(tzone);
		} else {
			cal = Calendar.getInstance();
		}

		cal.set(year, month, day, hour, minute, sec);
		cal.set(Calendar.MILLISECOND, msec);

		return cal.getTime();
	}

	private static int parseToInt(String token, int def) {
		if (token == null) {
			return def;
		}

		try {
			return Integer.parseInt(token, 10);
		} catch (Exception e) {
			// TODO: remove print stack trace once tested.
			e.printStackTrace();
			return def;
		}
	}

	public static void main(String[] args) {
		String s = "1997";
		System.out.println(s);
		Date d = getDateFromIso8601(s);
		System.out.println(d + "\n");

		s = "1997-07";
		System.out.println(s);
		d = getDateFromIso8601(s);
		System.out.println(d + "\n");

		s = "1997-07-16";
		System.out.println(s);
		d = getDateFromIso8601(s);
		System.out.println(d + "\n");

		s = "1997-07-16T19:20-05:00";
		System.out.println(s);
		d = getDateFromIso8601(s);
		System.out.println(d + "\n");

		s = "1997-07-16T19:20:30-05:00";
		System.out.println(s);
		d = getDateFromIso8601(s);
		System.out.println(d + "\n");

		s = "2007-03-26T19:36:28.4375Z";
		System.out.println(s);
		d = getDateFromIso8601(s);
		System.out.println(d + "\n");

		Date date = new Date();
		s = getIso8601Date(date);
		System.out.println("Our ISO8601: " + s);
		System.out.println(getDateFromIso8601(s) + "\n");

		s = getIso8601Date();
		System.out.println("Our ISO8601: " + s);
		System.out.println(getDateFromIso8601(s));

	}
}
