/*******************************************************************************
 * Copyright (c) 2009 Azigo, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.cp.xdi.util;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.codec.binary.Hex;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

/**
 * This class contains methods for creating HINs from simple and complex values
 */
public class IdASHinUtil {

	public static final XRI3Segment XRI_HIN = new XRI3Segment("$value$hash");
	
	public static final String ALGORITHM_SHA1 = "SHA-1";
	public static final String ALGORITHM_SHA256 = "SHA-256";
	public static final String ALGORITHM_MD5 = "MD5";

	public static final String ALGORITHM_DEFAULT = ALGORITHM_SHA256;

	private IdASHinUtil() { }

	public static XRI3Segment makeHin(IValue value, String algorithm) throws IdASException {

		return(new XRI3Segment("$value$hash" + algorithmToSubSegment(algorithm) + "!" + hashhex(makeString(value), algorithm)));
	}

	public static String makeString(IValue value) throws IdASException {

		if (value instanceof ISimpleValue) {

			return(makeSimpleValueString((ISimpleValue) value));
		} else if (value instanceof IEntity) {

			return(makeComplexValueString((IEntity) value));
		} else {

			return(null);
		}
	}

	public static String makeSimpleValueString(ISimpleValue simpleValue) throws IdASException {

		return(simpleValue.getLexical());
	}

	public static String makeComplexValueString(IEntity complexValue) throws IdASException {

		StringBuffer buffer = new StringBuffer();

		// get all attribute IDs and sort them

		List attributeIds = new ArrayList();

		for (Iterator i = complexValue.getAttributes(); i.hasNext(); ) {

			IAttribute attribute = (IAttribute) i.next();
			attributeIds.add(IdASUtil.attrIDToXri(attribute.getType()));
		}

		Collections.sort(attributeIds);

		// iterate through the attribute IDs

		for (Iterator i = attributeIds.iterator(); i.hasNext(); ) {

			// append the attribute ID to the buffer

			XRI3Segment attributeId = (XRI3Segment) i.next();

			buffer.append(attributeId.toString());
			buffer.append(' ');

			// iterate through the values

			List simpleValuesStrings = new ArrayList();
			List complexValuesStrings = new ArrayList();

			IAttribute attribute = complexValue.getAttribute(IdASUtil.xriToAttrID(attributeId));
			Iterator ii = attribute.getValues();

			if (ii != null) while (ii.hasNext()) {

				IValue value = (IValue) ii.next();

				if (value instanceof ISimpleValue) {

					simpleValuesStrings.add('"' + makeSimpleValueString((ISimpleValue) value) + '"' + ' ');
				} else if (value instanceof IEntity) {

					complexValuesStrings.add(makeComplexValueString((IEntity) value) + ' ');
				}
			}

			// sort the values and add them to the buffer

			Collections.sort(simpleValuesStrings);
			Collections.sort(complexValuesStrings);

			for (Iterator iii = simpleValuesStrings.iterator(); iii.hasNext(); ) buffer.append((String) iii.next());
			for (Iterator iii = complexValuesStrings.iterator(); iii.hasNext(); ) buffer.append((String) iii.next());
		}

		// done

		return(buffer.toString());
	}

	public static String hashhex(String string, String algorithm) {

		MessageDigest digest;

		try {

			digest = MessageDigest.getInstance(algorithm);
		} catch (NoSuchAlgorithmException ex) {

			throw new RuntimeException(ex);
		}

		byte[] bytes;
		String hash;

		try {

			digest.update(string.getBytes("UTF-8"));
			bytes = digest.digest();
			hash = new String(Hex.encodeHex(bytes));
		} catch (UnsupportedEncodingException ex) {

			throw new RuntimeException(ex);
		}

		return(hash);
	}

	public static String algorithmToSubSegment(String algorithm) {

		if (algorithm.equals(ALGORITHM_SHA1)) return("$sha$1");
		if (algorithm.equals(ALGORITHM_SHA256)) return("$sha$256");
		if (algorithm.equals(ALGORITHM_MD5)) return("$md$5");

		return(null);
	}

	public static String segmentToAlgorithm(XRI3Segment xri) {

		if (! isHin(xri)) return(null);

		if (xri.getNumSubSegments() < 3) return(ALGORITHM_DEFAULT);
		if (xri.getSubSegment(2).toString().equals("$sha$1")) return(ALGORITHM_SHA1);
		if (xri.getSubSegment(2).toString().equals("$sha$256")) return(ALGORITHM_SHA256);
		if (xri.getSubSegment(2).toString().equals("$md$5")) return(ALGORITHM_MD5);

		return(null);
	}
	
	public static boolean isHin(XRI3Segment xri) {
		
		if (xri.getNumSubSegments() < 3) return(false);
		if (! xri.getSubSegment(0).toString().equals("$value")) return(false);
		if (! xri.getSubSegment(1).toString().equals("$hash")) return(false);
		
		return(true);
	}
}
