/*******************************************************************************
 * Copyright (c) 2009 Azigo, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.cp.xdi;

import java.net.URI;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.common.SimpleValueSerializer;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.GraphComponent;
import org.eclipse.higgins.xdi4j.Literal;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Reference;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.addressing.Addressing;
import org.eclipse.higgins.xdi4j.constants.DictionaryConstants;
import org.eclipse.higgins.xdi4j.constants.TypeConstants;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageEnvelope;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.client.XDIClient;
import org.eclipse.higgins.xdi4j.messaging.error.ErrorMessageResult;
import org.eclipse.higgins.xdi4j.signatures.Signatures;
import org.eclipse.higgins.xdi4j.util.CopyUtil;
import org.eclipse.higgins.xdi4j.util.XdiHinUtil;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class XDIAttribute implements IAttribute {

	private XDIContext context;

	private XDIClient xdiClient;
	private XRI3Segment senderXri;
	private PrivateKey privateKey;

	private XRI3 address;

	private URI attrID;
	private IAttributeModel model;

	public XDIAttribute(XDIContext context, XDIClient xdiClient, XRI3Segment senderXri, PrivateKey privateKey, XRI3 address, URI attrID) throws IdASException {

		this.context = context;

		this.xdiClient = xdiClient;
		this.senderXri = senderXri;
		this.privateKey = privateKey;

		this.address = address;

		this.attrID = attrID;
		this.model = null;
	}

	public URI getAttrID() throws IdASException {

		return(this.attrID);
	}

	public IAttributeModel getModel() throws IdASException {

		return(this.model);
	}

	public boolean isSingleValued() throws IdASException {

		Iterator values = this.getValues();

		return(values != null && values.hasNext() && ! values.hasNext());
	}

	public Iterator getValues() throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(this.senderXri);
		Operation operation = message.createGetOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		CopyUtil.copyStatement(Addressing.convertAddressToStatement(this.address), operationGraph, null);

		// send the message

		MessageResult messageResult;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.context.setLastMessageEnvelope(messageEnvelope);
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.context.setLastMessageResult(messageResult);
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());
		} catch (IdASException ex) {
			
			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while sending XDI message: " + ex.getMessage(), ex);
		}

		// read values from message result

		GraphComponent[] graphComponents = Addressing.findByAddress(messageResult.getGraph(), this.address);

		List values = new ArrayList();

		// if the predicate has a literal, the IdAS Attribute has a single, simple string value

		if (graphComponents.length == 1 && graphComponents[0] instanceof Literal) {

			Object data = ((Literal) graphComponents[0]).getData();
			XRI3 simpleValueAddress = this.address;

			values.add(new XDISimpleAttrValue(this.context, simpleValueAddress, data, ITypedValue.STRING_TYPE_URI));
		}

		// if the predicate has one or more references, the IdAS Attribute is an Entity Relation

		if (graphComponents.length > 0 && graphComponents[0] instanceof Reference) {

			for (int i=0; i<graphComponents.length; i++) {

				Reference reference = (Reference) graphComponents[i];

				Object data = IdASUtil.xriToEntityID(reference.getReferenceXri());
				XRI3 simpleValueAddress = this.address;

				values.add(new XDISimpleAttrValue(this.context, simpleValueAddress, data, ITypedValue.RELATIVEENTITYUDI_TYPE_URI));
			}
		}

		// if the predicate has an inner graph, each subject corresponds to one value of the IdAS Attribute

		if (graphComponents.length == 1 && graphComponents[0] instanceof Graph) {

			Iterator innerSubjects = ((Graph) graphComponents[0]).getSubjects();
			while (innerSubjects.hasNext()) {

				Subject innerSubject = (Subject) innerSubjects.next();
				if (! XdiHinUtil.isHin(innerSubject.getSubjectXri())) continue;

				if (innerSubject.containsPredicate(TypeConstants.XRI_VALUE)) {

					// if the subject has a $value predicate, then this is a simple value

					Predicate innerPredicate = innerSubject.getPredicate(TypeConstants.XRI_VALUE);
					if (! innerPredicate.containsLiteral()) continue;
					String dataStr = Addressing.findLiteralData(innerSubject, new XRI3(TypeConstants.XRI_VALUE.toString()));

					URI type = null;
					innerPredicate = innerSubject.getPredicate(DictionaryConstants.XRI_INHERITANCE);
					if (innerPredicate != null && innerPredicate.containsReferences()) type = IdASUtil.xriToType(innerPredicate.getReference().getReferenceXri());

					Object data = SimpleValueSerializer.deserialize(dataStr, type.toString());

					XRI3 simpleValueAddress = new XRI3(this.address.toString() + "//" + innerSubject.getSubjectXri().toString() + "/" + innerPredicate.getPredicateXri().toString());

					values.add(new XDISimpleAttrValue(this.context, simpleValueAddress, data, type));
				} else {

					// otherwise this is a complex value

					URI type = null;
					Predicate innerPredicate = innerSubject.getPredicate(DictionaryConstants.XRI_INHERITANCE);
					if (innerPredicate != null && innerPredicate.containsReferences()) type = IdASUtil.xriToType(innerPredicate.getReference().getReferenceXri());

					XRI3 complexValueAddress = new XRI3(this.address.toString() + "//" + innerSubject.getSubjectXri().toString());

					values.add(new XDIBlankEntity(this.context, this.xdiClient, this.senderXri, this.privateKey, complexValueAddress, null, type));
				}
			}
		}

		return(values.iterator());
	}

	public Iterator getValues(IExtension[] extensions) throws IdASException {

		return(this.getValues());
	}

	public void remove() throws IdASException {

		// TODO: send XDI message

		throw new NotImplementedException();
	}

	public IEntity addComplexValue(URI dataType) throws IdASException, InvalidTypeException {

		// TODO: send XDI message

		throw new NotImplementedException();
	}

	public ISimpleValue addSimpleValue(URI dataType, Object data) throws IdASException, InvalidTypeException {

		// TODO: send XDI message

		throw new NotImplementedException();
	}

	public IValue addValue(URI dataType) throws IdASException, InvalidTypeException {

		// TODO: send XDI message

		throw new NotImplementedException();
	}

	public IValue addValue(IValue copyFrom) throws IdASException {

		// TODO: send XDI message

		throw new NotImplementedException();
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {

		throw new NotImplementedException();
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {

		throw new NotImplementedException();
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {

		throw new NotImplementedException();
	}

	public Iterator getAttributes() throws IdASException {

		throw new NotImplementedException();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {

		throw new NotImplementedException();
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {

		throw new NotImplementedException();
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {

		throw new NotImplementedException();
	}

	public void removeAttribute(URI attrID) throws IdASException {

		throw new NotImplementedException();
	}

	public boolean equals(IAttribute attr) throws IdASException {

		throw new NotImplementedException();
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {

		throw new NotImplementedException();
	}

	public IEntity addComplexValue(String entityID) throws IdASException,
			InvalidTypeException {
		// TODO Auto-generated method stub
		return null;
	}

	public URI getType() throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

	public IEntity addAttributeValue(URI attrType, String entityId)
			throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}
}
