/*******************************************************************************
 * Copyright (c) 2009 Azigo, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.cp.xdi;


import java.net.URI;
import java.net.URL;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.higgins.idas.api.AuthenticationException;
import org.eclipse.higgins.idas.api.ContextOpenException;
import org.eclipse.higgins.idas.api.EntityExistsException;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAuthNAttributesMaterials;
import org.eclipse.higgins.idas.api.IAuthNMaterials;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.IFilterEntityIDAssertion;
import org.eclipse.higgins.idas.api.IFilterEntityTypeAssertion;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidEntityIDException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.common.AuthNAnonymousMaterials;
import org.eclipse.higgins.idas.common.AuthNNamePrivateKeyMaterials;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.addressing.Addressing;
import org.eclipse.higgins.xdi4j.constants.MessagingConstants;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageEnvelope;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.client.XDIClient;
import org.eclipse.higgins.xdi4j.messaging.client.http.XDIHttpClient;
import org.eclipse.higgins.xdi4j.messaging.error.ErrorMessageResult;
import org.eclipse.higgins.xdi4j.signatures.Signatures;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class XDIContext implements IContext {

	private URL url;

	private XRI3Segment senderXri;
	private PrivateKey privateKey;
	private XDIClient xdiClient;

	private boolean open;
	private MessageEnvelope lastMessageEnvelope;
	private MessageResult lastMessageResult;

	private MessageEnvelope applyMessageEnvelope;
	private Message applyMessage;

	public XDIContext(URL url) throws IdASException {

		this.url = url;

		this.open = false;
	}

	Message getApplyMessage() {

		return(this.applyMessage);
	}

	public String open(IAuthNMaterials authNMaterials) throws IdASException, ContextOpenException {

		// read the authentication materials

		if (authNMaterials instanceof AuthNNamePrivateKeyMaterials) {

			this.senderXri = new XRI3Segment(((AuthNNamePrivateKeyMaterials) authNMaterials).getName());
			this.privateKey = ((AuthNNamePrivateKeyMaterials) authNMaterials).getPrivateKey();
		} else if (authNMaterials instanceof AuthNAnonymousMaterials) {

			this.senderXri = new XRI3Segment("$");
			this.privateKey = null;
		}

		this.applyMessageEnvelope = MessageEnvelope.newInstance();
		this.applyMessage = this.applyMessageEnvelope.newMessage(this.senderXri);

		// create the XDI client

		try {

			this.xdiClient = new XDIHttpClient(this.url.toString());
		} catch (Exception ex) {

			throw new IdASException("Cannot initialize XDI client.", ex);
		}

		// done

		this.open = true;
		return(null);
	}

	public void close() throws IdASException {

		this.xdiClient.close();
		this.xdiClient = null;

		this.open = false;
	}

	public Iterator getEntities(IFilter filter) throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(this.senderXri);
		message.createGetOperation();

		// send the message

		MessageResult messageResult;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.setLastMessageEnvelope(messageEnvelope);
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.setLastMessageResult(messageResult);
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());
		} catch (IdASException ex) {

			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while sending XDI message: " + ex.getMessage(), ex);
		}

		// read entities from message result

		Iterator subjects = messageResult.getGraph().getSubjects();

		List entities = new ArrayList();

		while (subjects.hasNext()) {

			Subject subject = (Subject) subjects.next();

			String entityID = IdASUtil.xriToEntityID(subject.getSubjectXri());
			XRI3 entityAddress = new XRI3(IdASUtil.entityIDToXri(entityID).toString());
			URI entityType = IdASUtil.xriToType(Addressing.findReferenceXri(subject, new XRI3("$is$a")));

			IEntity entity = new XDIBlankEntity(this, this.xdiClient, this.senderXri, this.privateKey, entityAddress, entityID,  entityType);
			entities.add(entity);
		}

		return(entities.iterator());
	}

	public IEntity getEntity(String entityID) throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(this.senderXri);
		Operation operation = message.createGetExistsOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		operationGraph.createStatement(IdASUtil.entityIDToXri(entityID));

		// send the message

		MessageResult messageResult;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.setLastMessageEnvelope(messageEnvelope);
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.setLastMessageResult(messageResult);
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());
		} catch (IdASException ex) {

			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while sending XDI message: " + ex.getMessage(), ex);
		}

		// read entity from message result

		Iterator subjects = messageResult.getGraph().getSubjects();
		if (! subjects.hasNext()) throw new NoSuchEntityException();
		Subject subject = (Subject) subjects.next();
		if (! subject.getSubjectXri().equals(MessagingConstants.XRI_TRUE)) throw new NoSuchEntityException();

		XRI3 entityAddress = new XRI3(IdASUtil.entityIDToXri(entityID).toString());
		URI entityType = IdASUtil.xriToType(Addressing.findReferenceXri(subject, new XRI3("$is$a")));

		IEntity entity = new XDIBlankEntity(this, this.xdiClient, this.senderXri, this.privateKey, entityAddress, entityID, entityType);

		return(entity);
	}

	public IEntity addEntity(URI entityType, String entityID) throws IdASException, InvalidTypeException, InvalidEntityIDException, EntityExistsException {

		throw new NotImplementedException();
	}

	public IEntity addEntity(IEntity copyFrom) throws IdASException, EntityExistsException {

		throw new NotImplementedException();
	}

	public void applyUpdates() throws IdASException {

		// send the message

		MessageResult messageResult;

		try {

			if (this.privateKey != null) Signatures.sign(this.applyMessageEnvelope.getGraph(), this.privateKey);
			this.setLastMessageEnvelope(this.applyMessageEnvelope);
			messageResult = this.xdiClient.send(this.applyMessageEnvelope, null);
			this.setLastMessageResult(messageResult);
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());
		} catch (IdASException ex) {

			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while sending XDI message: " + ex.getMessage(), ex);
		}

		// reset the message

		this.applyMessageEnvelope = MessageEnvelope.newInstance();
		this.applyMessage = this.applyMessageEnvelope.newMessage(this.senderXri);
	}

	public void cancelUpdates() throws IdASException {

		throw new NotImplementedException();
	}

	public IAttribute buildAttribute(URI attrID) throws IdASException {

		throw new NotImplementedException();
	}

	public IEntity buildComplexAttrValue(URI dataType) throws IdASException, InvalidTypeException {

		throw new NotImplementedException();
	}

	public IFilter buildFilter() throws IdASException {

		throw new NotImplementedException();
	}

	public IFilterEntityIDAssertion buildEntityIDAssertion() throws IdASException {

		throw new NotImplementedException();
	}

	public IFilterEntityTypeAssertion buildEntityTypeAssertion() throws IdASException {

		throw new NotImplementedException();
	}

	public ISimpleValue buildSimpleAttrValue(URI dataType, Object value) throws IdASException, InvalidTypeException {

		throw new NotImplementedException();
	}

	public Iterator getRelationships() throws IdASException {

		throw new NotImplementedException();
	}

	public boolean verifyEntityAttributes(String entityID, Iterator attributes) throws IdASException {

		throw new NotImplementedException();
	}

	public String getSchema() throws IdASException {

		throw new NotImplementedException();
	}

	public void setSchema(String schema) throws IdASException {

		throw new NotImplementedException();
	}

	public void importData(String filter, String representationFormat) throws IdASException {

		throw new NotImplementedException();
	}

	public String exportData(String filter, String representationFormat) throws IdASException {

		throw new NotImplementedException();
	}

	/*
	 * Model methods
	 */

	public IContextModel getContextModel() throws IdASException {

		throw new NotImplementedException();
	}

	public boolean isOpen(Object identity) throws IdASException {

		return(this.open);
	}

	/*
	 * For debugging purposes we always remember the last XDI message envelope and
	 * XDI message result
	 */

	public MessageEnvelope getLastMessageEnvelope() {

		return(this.lastMessageEnvelope);
	}

	public MessageResult getLastMessageResult() {

		return(this.lastMessageResult);
	}

	void setLastMessageEnvelope(MessageEnvelope messageEnvelope) {

		this.lastMessageEnvelope = messageEnvelope;
	}

	void setLastMessageResult(MessageResult messageResult) {

		this.lastMessageResult = messageResult;
	}

	public IFilterAttributeAssertion buildAttributeAssertion() throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public IAuthNAttributesMaterials buildAuthNAttributesMaterials() throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public URI getContextID() throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public Iterator getEntities(IFilter filter, Iterator attrSelectionList) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public Iterator getEntities(IFilter filter, Iterator attrSelectionList, IExtension[] extensions) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public IEntity getEntity(String entityID, Iterator attrSelectionList) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public String open(IAuthNMaterials authentication, IExtension[] extensions) throws IdASException, ContextOpenException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public void reopen() throws IdASException, AuthenticationException {
		// TODO Auto-generated method stub

	}

	public void setComponentSetting(String key, Object value, boolean failUnsupported) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public void setComponentSetting(String key, Object value) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public Iterator getAttributes() throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public void removeAttribute(URI attrID) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {

		// TODO Auto-generated method stub
		throw new NotImplementedException();
	}

	public Iterator getEntities(String query) throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

	public Iterator getEntities(String query, Iterator attrSelectionList) throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

	public Iterator getEntities(String query, Iterator attrSelectionList, IExtension[] extensions) throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

	public IEntity addAttributeValue(URI attrType, String entityId) throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}
}
