/*******************************************************************************
 * Copyright (c) 2009 Azigo, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.cp.xdi;

import java.net.URI;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IValueModel;
import org.eclipse.higgins.idas.spi.BasicEntityModel;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.GraphComponent;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.addressing.Addressing;
import org.eclipse.higgins.xdi4j.constants.MessagingConstants;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageEnvelope;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.client.XDIClient;
import org.eclipse.higgins.xdi4j.messaging.error.ErrorMessageResult;
import org.eclipse.higgins.xdi4j.signatures.Signatures;
import org.eclipse.higgins.xdi4j.util.CopyUtil;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class XDIBlankEntity implements IEntity {

	private XDIContext context;
	
	private XDIClient xdiClient;
	private XRI3Segment senderXri;
	private PrivateKey privateKey;

	private XRI3 address;

	private String entityID;
	private URI type;
	private IEntityModel model;
	
	public XDIBlankEntity(XDIContext context, XDIClient xdiClient, XRI3Segment senderXri, PrivateKey privateKey, XRI3 address, String entityID, URI type) throws IdASException {

		this.context = context;

		this.xdiClient = xdiClient;
		this.senderXri = senderXri;
		this.privateKey = privateKey;
	
		this.address = address;
	
		this.entityID = entityID;
		this.type = type;
		this.model = new BasicEntityModel(type);
	}

	public IContext getContext() throws IdASException {
		
		return(this.context);
	}

	public String getEntityID() throws IdASException {
		
		return(this.entityID);
	}

//	public URI getEntityType() throws IdASException {
//		
//		return(this.type);
//	}

	public IEntityModel getModel() {
		
		return(this.model);
	}

	public URI getDataType() throws IdASException {
		
		return(this.model.getType());
	}

	public URI getValueType() throws IdASException {

		return(this.model.getType());
	}

	public boolean isSimple() throws IdASException {

		return(false);
	}

	public Iterator getAttributes() throws IdASException {

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(this.senderXri);
		Operation operation = message.createGetOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		CopyUtil.copyStatement(Addressing.convertAddressToStatement(this.address), operationGraph, null);

		// send the message

		MessageResult messageResult;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.context.setLastMessageEnvelope(messageEnvelope);
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.context.setLastMessageResult(messageResult);
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());
		} catch (IdASException ex) {
			
			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while sending XDI message: " + ex.getMessage(), ex);
		}

		// read attributes from message result

		GraphComponent[] graphComponents = Addressing.findByAddress(messageResult.getGraph(), this.address);
		if (graphComponents == null || graphComponents.length < 1 || ! (graphComponents[0] instanceof Subject)) throw new IdASException("Cannot retreive attributes.");
		Subject subject = (Subject) graphComponents[0];
		Iterator predicates = subject.getPredicates();

		List attributes = new ArrayList();

		while (predicates.hasNext()) {

			Predicate predicate = (Predicate) predicates.next();

			URI attrID = IdASUtil.xriToAttrID(predicate.getPredicateXri());
			XRI3 attributeAddress = new XRI3(this.address + "/" + IdASUtil.attrIDToXri(attrID).toString());

			IAttribute attribute = new XDIAttribute(this.context, this.xdiClient, this.senderXri, this.privateKey, attributeAddress, attrID);
			attributes.add(attribute);
		}

		return(attributes.iterator());
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {

		XRI3 attributeAddress = new XRI3(this.address + "/" + IdASUtil.attrIDToXri(attrID));

		// prepare an XDI message

		MessageEnvelope messageEnvelope = MessageEnvelope.newInstance();
		Message message = messageEnvelope.newMessage(this.senderXri);
		Operation operation = message.createGetExistsOperation();
		Graph operationGraph = operation.createOperationGraph(null);
		CopyUtil.copyStatement(Addressing.convertAddressToStatement(attributeAddress), operationGraph, null);

		// send the message

		MessageResult messageResult;

		try {

			if (this.privateKey != null) Signatures.sign(messageEnvelope.getGraph(), this.privateKey);
			this.context.setLastMessageEnvelope(messageEnvelope);
			messageResult = this.xdiClient.send(messageEnvelope, null);
			this.context.setLastMessageResult(messageResult);
			if (messageResult instanceof ErrorMessageResult) throw new IdASException("Error received from IdAS Proxy: " + ((ErrorMessageResult) messageResult).getErrorString());
		} catch (IdASException ex) {
			
			throw ex;
		} catch (Exception ex) {

			throw new IdASException("Problem while sending XDI message: " + ex.getMessage(), ex);
		}

		// read attribute from message result

		Iterator subjects = messageResult.getGraph().getSubjects();
		if (! subjects.hasNext()) throw new NoSuchEntityException();
		Subject subject = (Subject) subjects.next();
		if (! subject.getSubjectXri().equals(MessagingConstants.XRI_TRUE)) return(null);

		IAttribute attribute = new XDIAttribute(this.context, this.xdiClient, this.senderXri, this.privateKey, attributeAddress, attrID);

		return(attribute);
	}

	public void remove() throws IdASException {
		
		throw new NotImplementedException();
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {

		throw new NotImplementedException();
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {

		throw new NotImplementedException();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {

		throw new NotImplementedException();
	}

	public void removeAttribute(URI attrID) throws IdASException {

		throw new NotImplementedException();
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {

		throw new NotImplementedException();
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {

		throw new NotImplementedException();
	}

	public boolean equals(IValue value) throws IdASException {
		
		throw new NotImplementedException();
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {

		throw new NotImplementedException();
	}

	public boolean isProxy() throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public URI getType() throws IdASException {
		return(this.type);
	}

	public IValueModel getValueModel() {
		return getModel();
	}

	public IEntity addAttributeValue(URI attrType, String entityId) throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}
}
