/*******************************************************************************
 * Copyright (c) 2009 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.proxy;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.as.IdASContextMessagingTarget;
import org.eclipse.higgins.xdi4j.messaging.server.EndpointRegistry;
import org.eclipse.higgins.xdi4j.messaging.server.MessagingTarget;

public class CleanupThread extends Thread {

	private static final Log log = LogFactory.getLog(CleanupThread.class);

	private EndpointRegistry endpointRegistry;
	private List<IdASContextMessagingTarget> messagingTargets;
	private long checkIntervalMillis;
	private long expireIntervalMillis;

	public CleanupThread(EndpointRegistry endpointRegistry, long checkIntervalMillis, long expireIntervalMillis) {

		this.endpointRegistry = endpointRegistry;
		this.messagingTargets = new ArrayList<IdASContextMessagingTarget> ();
		this.checkIntervalMillis = checkIntervalMillis;
		this.expireIntervalMillis = expireIntervalMillis;
	}

	@Override
	public void run() {

		// run until we are interrupted

		while (true) {

			// sleep a bit

			try {

				Thread.sleep(this.checkIntervalMillis);
			} catch (InterruptedException ex) {

				break;
			}

			// find out which of the messaging targets expired

			List<MessagingTarget> expiredMessagingTargets = new ArrayList<MessagingTarget> ();

			for (IdASContextMessagingTarget messagingTarget : this.messagingTargets) {

				long lastActivityMillis = ((IdASContextMessagingTarget) messagingTarget).getLastActivityMillis();

				if (System.currentTimeMillis() - lastActivityMillis > this.expireIntervalMillis) {

					expiredMessagingTargets.add(messagingTarget);
				}
			}

			// unregister them from the endpoint

			log.info("Unregistering " + expiredMessagingTargets.size() + " expired messaging targets.");

			for (MessagingTarget expiredMessagingTarget : expiredMessagingTargets) {

				this.endpointRegistry.unregisterMessagingTarget(expiredMessagingTarget);

				try {

					expiredMessagingTarget.shutdown();
				} catch (Exception ex) {

					log.warn("Cannot shutdown expired messaging target: " + ex.getMessage(), ex);
				}
			}
			
			// and from ourselves
			
			this.messagingTargets.removeAll(expiredMessagingTargets);
		}
	}

	public synchronized void addMessagingTarget(IdASContextMessagingTarget messagingTarget) {

		this.messagingTargets.add(messagingTarget);
	}
}
