/*******************************************************************************
 * Copyright (c) 2009 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.proxy;

import java.net.URI;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.as.IdASContextMessagingTarget;
import org.eclipse.higgins.idas.api.IAuthNMaterials;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.common.AuthNMaterialsSerializer;
import org.eclipse.higgins.idas.udi.IContextUDI;
import org.eclipse.higgins.idas.udi.IContextUDIMetadata;
import org.eclipse.higgins.idas.udi.UDIResolver;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.addressing.Addressing;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.EndpointRegistry;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class ContextUDIResourceHandler extends AbstractResourceHandler {

	private static final Log log = LogFactory.getLog(ContextUDIResourceHandler.class);

	public static final XRI3Segment XRI_AUTHNMATERIALSTYPES = new XRI3Segment("$authn$type");
	public static final XRI3Segment XRI_AUTHNMATERIALSTYPE = new XRI3Segment("$authn$type");
	public static final XRI3Segment XRI_AUTHNMATERIALS = new XRI3Segment("$authn");
	public static final XRI3Segment XRI_CONTEXTTYPES = new XRI3Segment("$is$a");
	public static final XRI3Segment XRI_CONTEXTURI = new XRI3Segment("$uri$http");
	public static final XRI3Segment XRI_SESSIONKEY = new XRI3Segment("$key$session");

	protected EndpointRegistry endpointRegistry;
	protected CleanupThread cleanupThread;
	protected String baseUri;
	protected UDIResolver udiResolver;

	protected IContextUDIMetadata contextUDIMetadata;
	protected String contextUri;
	protected String sessionKey;

	public ContextUDIResourceHandler(Message message, Subject subject, EndpointRegistry endpointRegistry, CleanupThread cleanupThread, String baseUri, UDIResolver udiResolver) {

		super(message, subject);

		this.endpointRegistry = endpointRegistry;
		this.cleanupThread = cleanupThread;
		this.baseUri = baseUri;
		this.udiResolver = udiResolver;
	}

	@Override
	public boolean executeGet(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		// read message fields

		Graph outsideGraph = this.operationSubject.getContainingGraph().getContainingGraph();
		Subject outsideSubject = outsideGraph.getSubject(this.operationSubject.getSubjectXri());
		if (outsideSubject == null) throw new MessagingException("Incomplete message");

		String udiStr = Addressing.findLiteralData(outsideSubject, new XRI3("$value"));;
		if (udiStr == null) throw new MessagingException("No UDI found in message.");

		String authNMaterialsType = Addressing.findLiteralData(outsideSubject, new XRI3(XRI_AUTHNMATERIALSTYPE.toString()));
		String authNMaterialsStr = Addressing.findLiteralData(outsideSubject, new XRI3(XRI_AUTHNMATERIALS.toString()));

		// populate the message result

		try {

			this.resolveUDI(udiStr);

			Subject subject = messageResult.getGraph().createSubject(this.operationSubject.getSubjectXri());

			if (this.operationSubject.containsPredicate(XRI_CONTEXTTYPES) && this.contextUDIMetadata.getTypes() != null) {

				Predicate predicate = subject.createPredicate(XRI_CONTEXTTYPES);
				for (String string : this.contextUDIMetadata.getTypes()) predicate.createReference(new XRI3Segment(string));
			}

			if (this.operationSubject.containsPredicate(XRI_AUTHNMATERIALSTYPES) && this.contextUDIMetadata.getAuthNMaterialsTypes() != null) {

				Predicate predicate = subject.createPredicate(XRI_AUTHNMATERIALSTYPES);
				for (String string : this.contextUDIMetadata.getAuthNMaterialsTypes()) predicate.createReference(IdASUtil.typeToXri(URI.create(string)));
			}

			if (this.operationSubject.containsPredicate(XRI_AUTHNMATERIALS) && this.contextUDIMetadata.getAuthNMaterials() != null) {

				Predicate predicate = subject.createPredicate(XRI_AUTHNMATERIALS);
				predicate.createLiteral(this.contextUDIMetadata.getAuthNMaterials());
			}

			if (this.operationSubject.containsPredicate(XRI_CONTEXTURI) || this.operationSubject.containsPredicate(XRI_SESSIONKEY)) {

				IAuthNMaterials authNMaterials = AuthNMaterialsSerializer.deserialize(authNMaterialsType, authNMaterialsStr);

				this.setupProxyUri(udiStr, authNMaterials);

				Predicate predicate = subject.createPredicate(XRI_CONTEXTURI);
				predicate.createLiteral(this.contextUri);

				predicate = subject.createPredicate(XRI_SESSIONKEY);
				predicate.createLiteral(this.sessionKey);
			}
		} catch (IdASException ex) {

			throw new MessagingException(ex.getMessage(), ex);
		}

		log.info("Successfully answered resultion request for Context UDI.");
		
		return(true);
	}

	protected void resolveUDI(String udiStr) throws MessagingException {

		// resolve the Context UDI

		try {

			IContextUDI contextUDI = this.udiResolver.parseContextUDI(udiStr);
			if (contextUDI == null) throw new MessagingException("Cannot parse the Context UDI: " + udiStr);

			this.contextUDIMetadata = contextUDI.getContextUDIMetadata();
			if (this.contextUDIMetadata == null) throw new MessagingException("Cannot retrieve the Context Metadata: " + udiStr);
		} catch (MessagingException ex) {

			throw ex;
		} catch (Exception ex) {

			throw new MessagingException("Cannot resolve the Context UDI " + udiStr + ": " + ex.getMessage(), ex);
		}
	}

	private void setupProxyUri(String udiStr, IAuthNMaterials authNMaterials) throws MessagingException {

		// open the context

		IContext context;

		try {

			context = this.udiResolver.resolve(this.contextUDIMetadata);
			if (context == null) throw new MessagingException("Cannot instantiate context.");

			context.open(authNMaterials);
		} catch (MessagingException ex) {

			throw ex;
		} catch (Exception ex) {

			throw new MessagingException("Cannot open context: " + ex.getMessage(), ex);
		}

		// randomly generate a session key for the Attribute Service

		this.sessionKey = generateSessionKey();

		// need this to support sync://
		
		String contextUdi = null;
		
		if (udiStr.startsWith("sync://")) {
			
			contextUdi = udiStr;
			if (contextUdi.indexOf('#') > 1) {
				
				contextUdi = udiStr.substring(0, contextUdi.indexOf('#'));
			} else {
				
				contextUdi = udiStr;
			}
		}
		
		// dynamically set up a new IdASResourceMessagingTarget for the requested context

		IdASContextMessagingTarget messagingTarget = new IdASContextMessagingTarget();

		try {

			messagingTarget.setContext(context);
			messagingTarget.setContextUdi(contextUdi);
			messagingTarget.setSessionKey(this.sessionKey);
			messagingTarget.init(this.endpointRegistry);
		} catch (Exception ex) {

			throw new MessagingException("Cannot set up proxy URI: " + ex.getMessage(), ex);
		}

		// randomly generate a URI for the Attribute Service

		String path = "dynamic-" + UUID.randomUUID().toString();

		this.contextUri = this.baseUri + path;

		// register the IdASContextMessagingTarget

		this.endpointRegistry.registerMessagingTarget(path, messagingTarget);
		this.cleanupThread.addMessagingTarget(messagingTarget);
	}

	private static String generateSessionKey() {

		int length = 32;
		byte b[] = new byte[length];
		for (int i=0; i<length; i++) b[i] = (byte)((int)(Math.random() * ('Z' - 'A') + 'A'));

		return new String(b);
	}
}
