/*******************************************************************************

 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.proxy;

import java.io.File;
import java.util.Map;

import javax.servlet.ServletContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.configuration.xml.ConfigurationHandler;
import org.eclipse.higgins.idas.udi.UDIResolver;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.server.EndpointRegistry;
import org.eclipse.higgins.xdi4j.messaging.server.impl.ResourceHandler;
import org.eclipse.higgins.xdi4j.messaging.server.impl.ResourceMessagingTarget;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class IdASProxyMessagingTarget extends ResourceMessagingTarget {

	private static final Log log = LogFactory.getLog(IdASProxyMessagingTarget.class);

	private String higginsConfigurationFile;
	private String baseUri;
	private int checkInterval;
	private int expireInterval;

	private UDIResolver udiResolver;
	private CleanupThread cleanupThread;

	public IdASProxyMessagingTarget() {

		super(true);

		this.higginsConfigurationFile = "./WEB-INF/higgins.xml";
		this.baseUri = "";
		this.checkInterval = 10000;		// 10 seconds
		this.expireInterval = 600000;	// 10 minutes
	}

	@Override
	public void init(EndpointRegistry endpointRegistry) throws Exception {

		super.init(endpointRegistry);

		// load the Higgins UDIResolver

		ServletContext servletContext = endpointRegistry.getServletContext();
		String higginsConfigurationFileRealPath = servletContext.getRealPath(this.higginsConfigurationFile);

		log.debug("Trying to load Higgins configuration file " + higginsConfigurationFileRealPath + ".");

		ConfigurationHandler configurationHandler;
		Map<?, ?> configurationSettings;

		configurationHandler = new ConfigurationHandler();
		configurationHandler.setConfigurationBase("");
		configurationHandler.setFileName(higginsConfigurationFileRealPath);
		configurationHandler.configure(null);
		configurationSettings = configurationHandler.getSettings();

		this.udiResolver = (UDIResolver) configurationSettings.get("UDIResolver");
		if (this.udiResolver == null) throw new RuntimeException("Cannot find UDIResolver component in Higgins configuration.");

		log.info("Loaded Higgins configuration.");

		// set up the cleanup thread

		this.cleanupThread = new CleanupThread(endpointRegistry, this.checkInterval, this.expireInterval);
		this.cleanupThread.start();

		// done

		log.info("Successfully initialized with CWD " + new File(".").getAbsolutePath());
	}

	@Override
	public void shutdown() throws Exception {

		super.shutdown();

		// interrupt the cleanup thread

		do {

			this.cleanupThread.interrupt();

			try {

				Thread.sleep(500);
			} catch (InterruptedException ex) {

				ex.printStackTrace();
			}
		} while (this.cleanupThread.isAlive());

		// done

		log.info("Successfully shut down.");
	}

	@Override
	public ResourceHandler getResource(Message message, Subject subject) throws MessagingException {

		// find the type of this subject by looking at the $is$a statement
		// in the outer graph

		Graph outsideGraph = subject.getContainingGraph().getContainingGraph();
		Subject outsideSubject = outsideGraph.getSubject(subject.getSubjectXri());
		if (outsideSubject == null) return(null);

		// check the type of this subject, then return the appropriate ResourceHandler

		if (subject.getSubjectXri().equals(new XRI3Segment("$"))) {

			return new ContextUDIResourceHandler(message, subject, this.endpointRegistry, this.cleanupThread, this.baseUri, this.udiResolver);
		} else if (subject.getSubjectXri().equals(new XRI3Segment("="))) {

			return new EntityUDIResourceHandler(message, subject, this.endpointRegistry, this.cleanupThread, this.baseUri, this.udiResolver);
		} else if (subject.getSubjectXri().equals(new XRI3Segment("+"))) {

			return new AttributeUDIResourceHandler(message, subject, this.endpointRegistry, this.cleanupThread, this.baseUri, this.udiResolver);
		} else {

			return null;
		}
	}

	public String getHigginsConfigurationFile() {

		return this.higginsConfigurationFile;
	}

	public void setHigginsConfigurationFile(String higginsConfigurationFile) {

		this.higginsConfigurationFile = higginsConfigurationFile;
	}

	public String getBaseUri() {

		return this.baseUri;
	}

	public void setBaseUri(String baseUri) {

		this.baseUri = baseUri;
	}

	public int getCheckInterval() {

		return this.checkInterval;
	}

	public void setCheckInterval(int checkInterval) {

		this.checkInterval = checkInterval;
	}

	public int getExpireInterval() {

		return this.expireInterval;
	}

	public void setExpireInterval(int expireInterval) {

		this.expireInterval = expireInterval;
	}
}
