/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.registry.discovery;

import java.io.IOException;
import java.io.InputStream;

import org.apache.xerces.parsers.DOMParser;
import org.openxri.util.DOMUtils;
import org.openxri.xml.XRDS;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * A Discovery implementation that reads an XRDS document from a stream, then closes the stream.
 * @author msabadello at parityinc dot net
 */
public class StreamDiscovery extends AbstractDiscovery {

	private InputStream stream;
	private XRDS xrds;

	public StreamDiscovery(InputStream stream) {

		if (stream == null) throw new NullPointerException();
		
		this.stream = stream;
		this.xrds = null;
	}

	public XRDS discoverXRDS() throws DiscoveryException {

		// if we have the XRDS already, simply return it

		if (this.xrds != null) return(this.xrds);

		// parse XML document from input stream

		Document document; 

		try {

			DOMParser domParser = DOMUtils.getDOMParser();

			domParser.parse(new InputSource(this.stream));
			document = domParser.getDocument();
		} catch (Exception ex) {

			throw new DiscoveryException("Cannot parse XML.", ex);
		}

		// build XRDS object from XML document

		try {

			this.xrds = new XRDS(document.getDocumentElement(), true);
		} catch (Exception ex) {

			throw new DiscoveryException("Cannot parse XRDS.", ex);
		}

		// close the stream

		try {

			this.stream.close();
		} catch (IOException ex) {

			throw new DiscoveryException("Cannot close the file.", ex);
		}

		// done

		return(this.xrds);
	}
	
	public String toString() {
		
		return("stream");
	}
}
