/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.registry;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.higgins.configuration.api.IConfigurableComponent;
import org.eclipse.higgins.configuration.api.IConfigurableComponentFactory;
import org.eclipse.higgins.configuration.xrds.ConfigurationHandler;
import org.eclipse.higgins.configuration.api.ISettingDescriptor;
import org.eclipse.higgins.configuration.common.SettingDescriptor;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IContextFactory;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.registry.contextid.ContextIdFactory;
import org.eclipse.higgins.idas.registry.discovery.DiscoveryException;
import org.eclipse.higgins.idas.registry.discovery.FileDiscovery;
import org.eclipse.higgins.idas.registry.discovery.IDiscovery;
import org.eclipse.higgins.util.jscript.JScriptExec;
import org.eclipse.higgins.util.jscript.JScriptExecHelper;
import org.openxri.xml.SEPType;
import org.openxri.xml.Service;
import org.w3c.dom.CharacterData;
import org.w3c.dom.Element;

/**
 * Manages context factories.
 * 
 * @author msabadello at parityinc dot net
 * @see http://wiki.eclipse.org/ContextDiscoveryComponents
 * @see http://wiki.eclipse.org/ContextDiscoveryComponents_withoutXRDS
 */
public class IdASRegistry implements IConfigurableComponent, IConfigurableComponentFactory {

	private static IdASRegistry impl = null;

	public static final String HIGGINS_CONF_NS = "http://higgins.eclipse.org/Configuration";

	private static final String DEFAULT_CONTEXTFACTORIES_XRDS = "contextfactories.xrds";
	
	/**
	 * Discovery object used for looking up Context Factories
	 */
	protected IDiscovery discovery;

	/**
	 * Classloader to use for instantiating Context Factories
	 */
	protected ClassLoader classLoader;

	/**
	 * List of currently-known context factories (IContextFactory instances).
	 */
	protected List contextFactoriesList;	// <IContextFactory>

	/**
	 * Map of context types to currently-known context factories (IContextFactory instances).
	 */
	protected Map contextFactoriesMap;	// <String, List<IContextFactory>>

	/**
	 * Map of context ids to context configuration.
	 * This is populated during configure() and is an alternative to constructing context ids from XRDS documents.
	 */
	protected Map contextIdsMap;	// <String, Map>
	
	/**
	 * Javascript function for mapping context Ids.  Found in the component setting "JSContextIdMapper"
	 * This is populated during configure().  Note that it is an optional configuration parameter, in
	 * which case, contextIdMapperFunc will be set to null.  The name of the parameter to be passed into
	 * the function is "contextId", and it is expected to be a string.  The JScriptExecHelper.transformString method
	 * is called to convert the incoming context id to another context id.
	 */
	protected JScriptExec contextIdMapperFunc;

	/**
	 * In-memory list of context factories. It is part of the Map of global settings.
	 */
	protected List _contextFactoryInstances = null; 
	/**
	 * In-memory list of context Ids. It is part of the global settings.
	 */
	protected List _contextIds = null;
	/**
	 * Map of global settings.
	 */
	protected Map _mapGlobalSettings = null;
	/**
	 * Global setting descriptor
	 */
	protected ISettingDescriptor _globalDescriptor = null;
	/**
	 * Setting descriptor for this component.
	 */
	protected ISettingDescriptor _componentDescriptor = null;
	/**
	 * Setting descriptor for this component.
	 */
	protected ISettingDescriptor _idasDescriptor = null;
	/**
	 * Get the singleton instance.
	 */
	public static synchronized IdASRegistry getInstance() {

		if (impl == null) impl = new IdASRegistry();

		return(impl);
	}

	public IConfigurableComponent getNewInstance() {
		
		return(new IdASRegistry());
	}

	public IConfigurableComponent getSingletonInstance() {
		
		return(getInstance());
	}

	/**
	 * Use getInstance rather than the constructor.
	 */
	public IdASRegistry() {

		// initialize private variables

		this.contextFactoriesList = new ArrayList();	// <IContextFactory>
		this.contextFactoriesMap = new HashMap();		// <String, List<IContextFactory>>
		this.contextIdsMap = new HashMap();		// <String, Map>

		this.discovery = new FileDiscovery(new File(DEFAULT_CONTEXTFACTORIES_XRDS));
		this.classLoader = IdASRegistry.class.getClassLoader();
	}

	/**
	 * Configure the IdASRegistry using XRDS documents (an IDiscovery object).
	 * 
	 * @see http://wiki.eclipse.org/ContextDiscoveryComponents
	 */
	public void discover() throws IdASException {

		if (this.discovery == null) throw new NullPointerException();

		// clear all currently known context factories

		this.contextFactoriesList.clear();
		this.contextFactoriesMap.clear();

		// use the discovery object to get all service endpoints

		List seps;

		try {

			seps = this.discovery.discoverAllServices();
		} catch (DiscoveryException ex) {

			throw new IdASException(ex);
		}

		// iterate through all service endpoints of the XRDS and build cache

		for (Iterator i = seps.iterator(); i.hasNext(); ) {

			Service sep = (Service) i.next();

			// instantiate IContextFactory and register it under its context types

			try {

				IContextFactory contextFactory = this.instantiateFactory(sep);

				this.registerContextFactory(contextFactory, false);
			} catch (IdASException ex) {

				throw(ex);
			} catch (Exception ex) {

				throw new IdASException("Cannot instantiate Context Factory.", ex);
			}
		}
	}

	/**
	 * Configure the IdASRegistry using XRDS documents (an IDiscovery object).
	 * 
	 * @see http://wiki.eclipse.org/ContextDiscoveryComponents
	 */
	public void discover(String type) throws IdASException {

		if (discovery == null) throw new NullPointerException();

		// use the discovery object to get only the service endpoints for the type we need

		List seps;

		try {

			seps = this.discovery.discoverServices(type, null, null);
		} catch (DiscoveryException ex) {

			throw new IdASException(ex);
		}

		// iterate through all service endpoints of the XRDS

		List typeFactoriesList = new ArrayList();

		for (Iterator i = seps.iterator(); i.hasNext(); ) {

			Service sep = (Service) i.next();

			// instantiate IContextFactory and add it to the map (for every <Type> element of the service endpoint)

			try {

				IContextFactory factory = this.instantiateFactory(sep);
				typeFactoriesList.add(factory);
			} catch (IdASException ex) {

				throw(ex);
			} catch (Exception ex) {

				throw new IdASException("Cannot instantiate Context Factory.", ex);
			}
		}

		// add the constructed IContextFactory objects to the cache, and return them

		this.contextFactoriesList.removeAll(typeFactoriesList);
		this.contextFactoriesList.addAll(typeFactoriesList);
		this.contextFactoriesMap.put(type, typeFactoriesList);
	}

	/**
	 * Configure the IdASRegistry using the Configuration API. This is an alternative way to using 
	 * XRDS documents / XRI resolution.
	 * 
	 * @see http://wiki.eclipse.org/ContextDiscoveryComponents_withoutXRDS
	 */
	public void configure(Map mapGlobalSettings, String strComponentName, 
			Map mapComponentSettings, ISettingDescriptor componentDescriptor,
			ISettingDescriptor globalDescriptor) throws Exception {

		// clear all currently known context factories and context id configurations
		this._mapGlobalSettings = mapGlobalSettings;
		this.contextFactoriesList.clear();
		this.contextFactoriesMap.clear();
		this.contextIdsMap.clear();
		
		_globalDescriptor = globalDescriptor;
		_componentDescriptor = componentDescriptor;
		_idasDescriptor = componentDescriptor.getSubSetting(strComponentName);
		
		// register the context factories from the configuration

		_contextFactoryInstances = (List) mapComponentSettings.get("ContextFactoryInstancesList");

		this.contextIdMapperFunc = (JScriptExec) mapComponentSettings.get( "JSContextIdMapper");

		for (Iterator i = _contextFactoryInstances.iterator(); i.hasNext(); ) {

			Map contextFactoryConfiguration = (Map) i.next();

			String contextFactoryInstanceName = (String) contextFactoryConfiguration.get("Instance");
			
			// Look for the context factory first in the component settings, then, if not there, in the global settings.
			
			IContextFactory contextFactory;
			
			if ((contextFactory = (IContextFactory)mapComponentSettings.get( contextFactoryInstanceName)) == null)
			{
				contextFactory = (IContextFactory) mapGlobalSettings.get(contextFactoryInstanceName);
			}

			this.registerContextFactory(contextFactory, false);
		}

		// register the context id configurations

		_contextIds = (List) mapComponentSettings.get("ContextIdsList");

		for (Iterator i = _contextIds.iterator(); i.hasNext(); ) {

			Map contextIdConfiguration = (Map) i.next();

			String contextId = (String) contextIdConfiguration.get("ContextId");

			ISettingDescriptor contextDescriptor = _idasDescriptor.getSubSetting("ContextIdsList").getSubSetting(contextId);
			
			this.registerContextId(contextId, contextIdConfiguration, contextDescriptor, false);
		}
	}
	
	public ISettingDescriptor getComponentDescriptor() { 
		return this._componentDescriptor;
	}

	public Map getContextIdConfiguration(String contextId) {

		String	mappedContextId;
		
		if (this.contextIdMapperFunc == null)
		{
			mappedContextId = contextId;
		}
		else
		{
			try
			{
				mappedContextId = JScriptExecHelper.transformString( this.contextIdMapperFunc, "contextId", contextId);
			}
			catch (Exception e)
			{
				mappedContextId = contextId;
			}
		}
		return((Map) this.contextIdsMap.get( mappedContextId));
	}

	/**
	 * Return all registered IContextFactory objects.
	 * 
	 * @return Iterator of {@link IContextFactory} objects
	 */
	public List getContextFactories() throws IdASException {

		// if we have no context factories, try to discover them

		if (this.contextFactoriesList.isEmpty()) this.discover();

		// done

		return(this.contextFactoriesList);
	}

	/**
	 * Find suitable IContextFactory objects for a context type.
	 * 
	 * @param type The context type to look for
	 * @throws IdASException 
	 */
	public List getContextFactories(String type) throws IdASException {

		// if we have no context factory for this type, try to discover it

		if (! this.contextFactoriesMap.containsKey(type)) this.discover(type);

		if (! this.contextFactoriesMap.containsKey(type)) return(new ArrayList());

		return((List) this.contextFactoriesMap.get(type));
	}

	/**
	 * Find a suitable IContextFactory for a context type.
	 * 
	 * @param type The context type to look for
	 * @return An IContextFactory or null if none was found
	 * @throws IdASException 
	 */
	public IContextFactory getContextFactory(String type) throws IdASException {

		List list = this.getContextFactories(type);
		if (list == null || list.size() < 1) return(null); else return((IContextFactory) list.get(0));
	}

	/**
	 * Find suitable IContextFactory objects for a ContextId
	 * @param contextId The IContextId of the context to be instantiated
	 * @return Iterator of {@link IContextFactory} objects
	 * @throws IdASException 
	 */
	public List getContextFactories(IContextId contextId) throws IdASException {

		List allFactories = new ArrayList();

		// get a list of types for this ContextId

		String[] types = contextId.getTypes();

		for (int i=0; i<types.length; i++) {

			// get a list of factories for this type

			List factories = this.getContextFactories(types[i]);

			allFactories.addAll(factories);
		}

		// return all factories for any of the ContextId's types

		return(allFactories);
	}

	/**
	 * Find a suitable IContextFactory for a ContextId
	 * @param contextId The IContextId of the context to be instantiated
	 * @return Iterator of {@link IContextFactory} objects
	 * @throws IdASException 
	 */
	public IContextFactory getContextFactory(IContextId contextId) throws IdASException {

		List list = this.getContextFactories(contextId);
		if (list == null || list.size() < 1) return(null); else return((IContextFactory) list.get(0));
	}

	/**
	 * Convenience method that creates an IContext based on an IContextId
	 * @param contextId The IContextId for which an IContext needs to be found.
	 * @return A configured context.
	 * @throws IdASException
	 * @see http://wiki.eclipse.org/ContextId
	 */
	public IContext createContext(IContextId contextId) throws IdASException {

		IContextFactory factory = this.getContextFactory(contextId);
		IContext context = factory.createContext(contextId);

		return(context);
	}

	/**
	 * Convenience method that creates an IContext based on a string (which must be a valid context ID)
	 * @param contextId The IContextId for which an IContext needs to be found.
	 * @return A newly created context.
	 * @throws IdASException
	 * @see http://wiki.eclipse.org/ContextId
	 */
	public IContext createContext(String contextIdStr) throws IdASException {

		IContextId contextId = ContextIdFactory.fromString(contextIdStr);
		IContextFactory factory = this.getContextFactory(contextId);
		IContext context = factory.createContext(contextId);

		return(context);
	}

	/**
	 * Add context factory to the registered map.
	 * @param types Types of contexts this factory can create
	 * @param factory IContextFactory object to be registered
	 * @param bSave If true, the settings are saved in the global map
	 */

	public void registerContextFactory(IContextFactory factory, boolean bSave) throws IdASException {

		if (factory == null) throw new NullPointerException();
		
		// add the factory to the map (under each context type)
		List types = factory.getTypes();

		for (int i=0; i< types.size(); i++) {

			List typeFactoriesList = (List) this.contextFactoriesMap.get(types.get(i));
			if (typeFactoriesList == null) typeFactoriesList = new ArrayList();
			typeFactoriesList.add(factory);

			this.contextFactoriesMap.put(types.get(i), typeFactoriesList);
		}

		// add the factory to the list (once)

		this.contextFactoriesList.add(factory);
		
		if ( bSave == false ) {
			return;
		}
		
		/* Add this context factory to the component map */
		ISettingDescriptor factoryDescriptor = factory.getComponentDescriptor();
		Map contextFactoryMap = new HashMap();
		contextFactoryMap.put("Instance", factoryDescriptor.getName()); 
		contextFactoryMap.put("ContextTypes", types);	
		
		this._contextFactoryInstances.add(contextFactoryMap);
		this._idasDescriptor.getSubSetting("ContextFactoryInstancesList").addSubSetting(factoryDescriptor);
		this._mapGlobalSettings.put(factoryDescriptor.getName(), factory);
		this._globalDescriptor.addSubSetting(factoryDescriptor.getName(), "htf:classinstance");
	}
	
	/**
	 * Get the descriptor for any type of context factory listed under
	 * 'ContextFactoryInstancesList'. If it is not a registered context factory,
	 * then a new setting descriptor is created and returned.
	 */
	public ISettingDescriptor getContextFactoryDescriptor(String factoryClassName, String[] factoryTypes) { 
		
		if (this.contextFactoriesMap.containsKey(factoryClassName)) { 
			/* This is a registered context factory */ 
			return _idasDescriptor.getSubSetting("ContextFactoryInstancesList").getSubSetting(factoryClassName);
		} else { 
			ISettingDescriptor factoryDescriptor = new SettingDescriptor(factoryClassName, "htf:map");
			factoryDescriptor.addSubSetting("Instance", "xsd:string");
			ISettingDescriptor typeDescriptor = factoryDescriptor.addSubSetting("ContextTypes", "htf:list");
			for ( int i = 0; i < factoryTypes.length; i++ ) {
				typeDescriptor.addSubSetting(factoryTypes[i], "xsd:string");
			}
			return factoryDescriptor;
		}
	}
	
	/**
	 * Get the setting descriptor for the specified context id. If it is a 
	 * registered context id, its setting descriptor is returned from the component's
	 * setting descriptor. Else depending on the type of the context id, a new
	 * setting descriptor is created and returned. 
	 */
	
	public ISettingDescriptor getContextIdDescriptor(String contextIdName, String[] types) throws IdASException { 
		if ( this.contextIdsMap.containsKey(contextIdName)) { 
			return _idasDescriptor.getSubSetting("ContextIdsList").getSubSetting(contextIdName);
		} else { 
			/* TODO: Assumption one context Id can be of only one type */ 
			if ( this.contextFactoriesMap.containsKey(types[0]) ) { 
				IContextFactory factory = (IContextFactory)((List)contextFactoriesMap.get(types[0])).get(0);
				return factory.getContextDescriptor(contextIdName);
			} else { 
				/* TODO: Throw exception - Attempt to create context id of type that is
				 * not associated with any context factory */
				return null;
			}
		}
	}

	/**
	 * Add context factory to the registered map.
	 * @param types Types of contexts this factory can create
	 * @param factory IContextFactory object to be registered
	 * @param factoryDescriptor Setting descriptor for the factory to register
	 * @param bSave If true, the settings are saved in the global map
	 */
	
	public void registerContextFactory(String[] types, String factoryClassName, ISettingDescriptor factoryDescriptor, boolean bSave ) throws IdASException {

		if (types == null || factoryClassName == null) throw new NullPointerException();
		
		/* Add this context factory to the component map */
		Map contextFactoryMap = new HashMap();
		contextFactoryMap.put("Instance", factoryDescriptor.getName()); 
		List typeList = new ArrayList();
		for ( int i = 0; i < types.length; i++ ) { 
			typeList.add(types[i]);
		}
		contextFactoryMap.put("ContextTypes", typeList);	
	
		// instantiate the Context Factory

		IContextFactory factory;

		try {

			factory = instantiateFactory(factoryClassName, contextFactoryMap, factoryDescriptor);
		} catch (IdASException ex) {

			throw(ex);
		} catch (Exception ex) {

			throw new IdASException(ex);
		}

		this.registerContextFactory(factory, bSave);
	}

	/**
	 * Add context factory to the registered map.
	 * @param types Type of context this factory can create
	 * @param factory IContextFactory object to be registered
	 * @param configuration Configuration for the Context Factory (may be null)
	 * @param bSave If true, the settings are saved in the global map
	 */
	public void registerContextFactory(String type, String factoryClassName, 
			ISettingDescriptor factoryDescriptor, boolean bSave ) throws IdASException {

		this.registerContextFactory(new String[] { type }, factoryClassName, factoryDescriptor, bSave);
	}
	/**
	 * Remove context factory from the cache.
	 * @param factory IContextFactory object to be removed
	 */
	public void removeContextFactory(IContextFactory factory) throws IdASException {

		if (factory == null) throw new NullPointerException();

		this.contextFactoriesList.remove(factory);

		for (Iterator i = this.contextFactoriesMap.values().iterator(); i.hasNext(); ) {

			List typeFactoriesList = (List) i.next();

			typeFactoriesList.remove(factory);
		}
		
		for ( int j = 0; j < this._contextFactoryInstances.size(); j++ ) {
			Map contextFactoryMap = (Map)this._contextFactoryInstances.get(j);
			if ( contextFactoryMap.get("Instance").toString().equals(factory.getName()) ) { 
				this._contextFactoryInstances.remove(j);
				this._idasDescriptor.getSubSetting("ContextFactoryInstancesList").removeSubSetting(j);
			}
		}
		
		this._mapGlobalSettings.remove(factory.getName());
		this._globalDescriptor.removeSubSetting(this._globalDescriptor.getSubSetting(factory.getName()));
	}

	public void registerContextId(String contextId, Map contextIdConfiguration, 
			ISettingDescriptor contextDescriptor, boolean bSave) throws Exception { 
		this.contextIdsMap.put(contextId, contextIdConfiguration);
		if ( bSave == false ) {
			return;
		}
		
		this._contextIds.add(contextIdConfiguration);
		this._idasDescriptor.getSubSetting("ContextIdsList").addSubSetting(contextDescriptor);
	}
	
	public void removeContextId(String contextId) { 
		if ( contextId == null ) throw new NullPointerException();
		this.contextIdsMap.remove(contextId);
		
		for ( int i = 0; i < this._contextIds.size(); i++ ) { 
			if ( ((Map)this._contextIds.get(i)).get("ContextId").toString().equalsIgnoreCase(contextId) ) { 
				this._contextIds.remove(i);
				this._idasDescriptor.getSubSetting("ContextIdsList").removeSubSetting(i);
				break;
			}
		}
	}
	
	/**
	 * Instantiates a Context Factory based on a class name and configuration (may be null).
	 * The method checks if the Context Factory implements the IConfigurableComponent interface; if yes, it is
	 * automatically configured.
	 * @param factoryClass The Context Factory class
	 * @param configuration Configuration for the Context Factory
	 */
	private IContextFactory instantiateFactory(String factoryClass, Map configuration, ISettingDescriptor factoryDescriptor) throws Exception {

		Class clazz = this.classLoader.loadClass(factoryClass);

		// check if the class implements the IContextFactory interface

		boolean foundInterface = false;
		for (Class c = clazz; c != null; c = c.getSuperclass()) {

			Class[] interfaces = c.getInterfaces();
			for (int j = 0; j < interfaces.length; j++) {

				if (interfaces[j] == IContextFactory.class) {

					foundInterface = true;
					break;
				}
			}
		}

		if (! foundInterface) return(null);

		// instantiate and configure the Context Factory, and add it to the registry

		IContextFactory newFactory = (IContextFactory) clazz.newInstance();

		if (configuration != null) {

			newFactory.configure(
					null, 
					factoryClass, 
					configuration,
					factoryDescriptor,
					_globalDescriptor);
		}
		
		return(newFactory);
	}

	/**
	 * Instantiates and configures a Context Factory based on a class name.
	 * @param factoryClass The Context Factory class
	 */
	private IContextFactory instantiateFactory(String factoryClass) throws Exception {

		return(instantiateFactory(factoryClass, null, null));
	}

	/**
	 * Instantiates and configures a Context Factory based on an XRDS service endpoint.
	 * @return
	 */
	private IContextFactory instantiateFactory(Service sep) throws Exception {

		// read class name from the service endpoint

		String factoryClass;

		Vector values = sep.getOtherTagValues("Class");
		if (values.size() < 1) throw new IdASException("No Context Factory class name found in XRDS service endpoint.");
		factoryClass = ((CharacterData) ((Element) values.get(0)).getFirstChild()).getData();

		// create configuration map from the service endpoint using the XRDS ConfigurationHandler

		Map configuration;

		ConfigurationHandler handler = new ConfigurationHandler();
		handler.setSEP(sep);

		try {

			handler.configure(null);
			configuration = handler.getSettings();
		} catch (Exception ex) {

			throw new IdASException("Cannot read configuration from XRDS service endpoint.", ex);
		}

		ISettingDescriptor settingDescriptor = handler.getSettingDescriptor();
		// instantiate Context Factory

		return(this.instantiateFactory(factoryClass, configuration, settingDescriptor));
	}

	/**
	 * Clears all currently known context factories and context id configurations.
	 * If after this call you want to use the IdASRegistry again, you need to do one of the following
	 * - Call configure() again to initialize it using the Configuration API.
	 * - Have a valid and working IDiscovery object (by default pointing to the local file "contextfactories.xrds")
	 */
	public void clear() {

		this.contextFactoriesList.clear();
		this.contextFactoriesMap.clear();
		this.contextIdsMap.clear();
	}

	/*
	 * Getters and setters
	 */

	public ClassLoader getClassLoader() {
		return classLoader;
	}

	public void setClassLoader(ClassLoader classLoader) {
		this.classLoader = classLoader;
	}

	public IDiscovery getDiscovery() {
		return discovery;
	}

	public void setDiscovery(IDiscovery discovery) {
		this.discovery = discovery;
	}	
}
