/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.iss;

import java.util.Iterator;

import javax.security.auth.callback.CallbackHandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.registry.HigginsRegistry;
import org.eclipse.higgins.registry.IIORegistryExtension;
import org.eclipse.higgins.registry.IRegistryExtension;
import org.eclipse.higgins.registry.SecurityRegistryExtension;

public class ICardSelectorService extends HigginsRegistry {
	protected static Log log = LogFactory.getLog(ICardSelectorService.class);
	
	protected static ICardSelectorService instance = null;
	
	/**
	 * Set up the valid service provider categories and automatically register
	 * all available service providers.
	 * 
	 * <p>
	 * The constructor is protected in order to prevent creation of additional
	 * instances.
	 */
	protected ICardSelectorService() {
		super(IICardSelectorFactory.class);
		log.trace("Constructor");
		initialize();
	}

	/**
	 * @param configPath
	 */
	public ICardSelectorService(String configPath) {
		super(IICardSelectorFactory.class, configPath);
		log.trace("Config Path is " + configPath);
		initialize();
	}
	
	private void initialize() {
		addExtension(new IIORegistryExtension(IICardSelectorFactory.class));
		addExtension(new SecurityRegistryExtension(IICardSelectorFactory.class));
		try {
			Class c = Class.forName("org.eclipse.higgins.iss.plugin.ICardSelectorRegistryExtension");
			IRegistryExtension e = (IRegistryExtension) c.newInstance();
			addExtension(e);
		} catch (ClassNotFoundException e) {
			log.warn(e);
		} catch (Throwable e) {
			log.warn(e);
			//e.printStackTrace();
		}
		loadProviders();
	}

	public Iterator getICardSelectorFactories() {
		return getServiceProviders();
	}

	public IICardSelectorFactory getICardSelectorFactory(String extID) {
		if (extID == null) {
			throw new IllegalArgumentException("id == null!");
		}
		return (IICardSelectorFactory) getServiceProvider(extID);
	}

	protected String getConfigFolder() {
		return ".iss";
	}

	public synchronized static ICardSelectorService getInstance() {
		if (instance == null) {
			instance = new ICardSelectorService();
		}
		return instance;
	}
	
	public IPolicy parsePolicy(String policy) throws PolicyParseException {
		log.trace("ICardSelectorService::parsePolicy(String policy)");
		log.trace("policy: " + policy);
		IPolicy pol;
		for (Iterator itr = getICardSelectorFactories(); itr.hasNext(); ) {
			IICardSelectorFactory f = (IICardSelectorFactory) itr.next();
			log.trace("Current IICardSelectorFactory = "+f);
			try {
				pol = f.parsePolicy(policy);
				if (pol != null) {
					log.debug("Parsed policy: " + pol);
					return pol;
				}
			} catch (PolicyParseException e) {
				log.error(e);
				//e.printStackTrace();
			}
		}
		throw new PolicyParseException("Unable to find proper policy parser.");
	}
	
	public IICardSelector getICardSelector(CallbackHandler handler, IPolicy policy) throws UnsatisfiablePolicyException {
		log.trace("ICardSelectorService::getICardSelector(CallbackHandler handler, IPolicy policy)");
		log.trace("policy: " + policy);
		IICardSelector selector;
		for (Iterator itr = getICardSelectorFactories(); itr.hasNext(); ) {
			IICardSelectorFactory f = (IICardSelectorFactory) itr.next();
			try {
				selector = f.getICardSelector(handler, policy);
				if (selector != null) {
					log.trace("I-Card Selector found: " + selector);
					return selector;
				}
			} catch (Exception e) {
				log.error(e);
				//e.printStackTrace();
			}
		}
		throw new UnsatisfiablePolicyException("Unable to find proper i-card selector.");
	}
	
	public IICardSelector getICardSelector(CallbackHandler handler, String policy) throws PolicyParseException, UnsatisfiablePolicyException {
		log.trace("ICardSelectorService::getICardSelector(CallbackHandler handler, String policy)");
		log.trace("policy: " + policy);
		IICardSelector selector;
		for (Iterator itr = getICardSelectorFactories(); itr.hasNext(); ) {
			IICardSelectorFactory f = (IICardSelectorFactory) itr.next();
			log.trace("Current IICardSelectorFactory = "+f);
			try {
				selector = f.getICardSelector(handler, policy);
				if (selector != null) {
					log.trace("I-Card Selector found: " + selector);
					return selector;
				}
			} catch (Exception e) {
				log.error(e);
				//e.printStackTrace();
			}
		}
		throw new PolicyParseException("Unable to find proper policy parser.");
	}

}
