package org.eclipse.higgins.keystore.common;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertPathValidatorResult;
import java.security.cert.CertStore;
import java.security.cert.CertStoreException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateParsingException;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.x500.X500Principal;

import org.apache.axiom.om.util.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.keystore.IKeyStoreService;
import org.eclipse.higgins.keystore.common.X509.AccessDescription;
import org.eclipse.higgins.keystore.common.X509.AuthorityInformationAccessExtension;
import org.eclipse.higgins.keystore.common.X509.ObjectIdentifier;
import org.eclipse.higgins.keystore.common.X509.PolicyInformation;
import org.eclipse.higgins.keystore.common.X509.URIName;
import org.eclipse.higgins.keystore.common.X509.X509HigginsCertificate;
import org.eclipse.higgins.keystore.common.utils.ConvertHelper;
import org.eclipse.higgins.keystore.common.utils.DNParser;

public abstract class KeyStoreService implements IKeyStoreService {

	private static Log log = LogFactory.getLog(KeyStoreService.class);
	
	private static Hashtable signsTable = new Hashtable();

	private static int maxCertCacheSize = 300;

	/**
	 * Contains certificate validity sign <code>Boolean</code> by SHA-256 hash of public key of certificate
	 */
	private static Hashtable certCache = new Hashtable();

	/**
	 * Contains certificate validity sign <code>Boolean</code> by SHA-256 hash of public key of certificate
	 */
	private static Hashtable certValidCache = new Hashtable();

	/**
	 * Contains certificate extended validity sign <code>Boolean</code> by SHA-256 hash of public key of certificate
	 */
	private static Hashtable certExtValidCache = new Hashtable();

	public X509Certificate[] getValidateCertificateChain(CallbackHandler handler,X509Certificate[] certPath, URI uri, CertStore certStore) throws CertificateException, CertStoreException {
		X509Certificate[] chain;
		try {
			chain = getNormalizedCertificateChain(certPath, certStore);
			if (validateCertificateChain(handler,chain, uri, certStore)) {
				putCertChain(chain);
				return chain;
			}
			else {
				if (chain != null && chain.length >0) {
					try {
						chain = getChainFromHTTPS(chain[0]);
						chain = getNormalizedCertificateChain(chain, certStore);
						if (validateCertificateChain(handler,chain, uri, certStore)) {
							putCertChain(chain);
							return chain;
						}
					}
					catch(Exception e) {
						log.error(e, e);
					}
				}
			}
		} catch (GeneralSecurityException e) {
			log.error(e, e);
			CertificateException ce = new CertificateException(e.getMessage());
			ce.setStackTrace(e.getStackTrace());
			throw ce;
		}
		return null;
	}

	private X509Certificate[] getChainFromHTTPS(X509Certificate cert) throws Exception {
		X509Certificate[] chain = null;
		String certHost = getHostNameFromDN(cert);
		if (certHost != null) {
			certHost = "https://" + certHost;
			chain = getChainFromHTTPS(certHost);
			if (chain != null) {
				return chain; 
			}
		}
		Collection altNamesC = null;
		try {
			altNamesC = cert.getSubjectAlternativeNames();
		} catch (CertificateParsingException e) {
			log.error(e, e);
		}
		if (altNamesC != null) {
			Iterator altNames = altNamesC.iterator();
			while (altNames.hasNext()) {
				Collection altNameCol = (Collection)altNames.next();
				if (altNameCol.size() == 2) {
					certHost = (String)altNameCol.toArray()[1];
					if (certHost.startsWith("*.")) {
						certHost = certHost.substring(2);
					}	
					chain = getChainFromHTTPS(certHost);
					if (chain != null) {
						return chain; 
					}
				}
			}
		}
		return null;
	}

	private static X509Certificate[] getChainFromHTTPS(String url) throws Exception {
		X509Certificate[] chain = null;
		URI uri;
		try {
			uri = new URI(url);
		} catch (URISyntaxException e) {
			throw e;
		}
		String host = uri.getHost();
		int port = uri.getPort();
		if (port == -1) {
			port = 443;
		}
		SSLSocketFactory factory = HttpsURLConnection.getDefaultSSLSocketFactory();
		SSLSocket socket = null;
		try {
			socket = (SSLSocket) factory.createSocket(host, port);
			socket.startHandshake();
			chain = (java.security.cert.X509Certificate[]) socket.getSession().getPeerCertificates();
		} catch (Exception e) {
			throw  e;
		} finally {
			if (socket != null) {
				try {
					socket.close();
				} catch (IOException e) {
					throw e;
				}
			}
		}
		return chain;
	}

	
		
	protected X509Certificate [] getNormalizedCertificateChain(X509Certificate [] certPath, CertStore certStore) throws CertificateException, CertStoreException
	{
		X509Certificate[] chain;
		try {
			chain = getParent(certPath, certStore);
			return chain;
		} catch (GeneralSecurityException e) {
			log.error(e, e);
			CertificateException ce = new CertificateException(e.getMessage());
			ce.setStackTrace(e.getStackTrace());
			throw ce;
		} catch (IOException e) {
			log.error(e, e);
			CertificateException ce = new CertificateException(e.getMessage());
			ce.setStackTrace(e.getStackTrace());
			throw ce;
		}
	}
	
	public boolean extendendValidationCertificateChain(CallbackHandler handler,X509Certificate[] certChain,URI uri, CertStore certStore) {
		boolean result = false;
		X509Certificate[] validCertChain = null;
		try {
			validCertChain = getValidateCertificateChain(handler,certChain,uri,certStore);
		} catch (Exception e) {
			log.error(e, e);
		}
		if (validCertChain != null)
		{
			result=extendendValidationCertificate(certChain);
		}
		return result;
	}

	protected boolean extendendValidationCertificate(X509Certificate[] certChain) {
		if (certChain == null || certChain.length == 0)
			return false;
		boolean result = false;
		Boolean cacheValidity = checkExtendedValidationInCache(certChain);
		if (cacheValidity != null)
			return cacheValidity.booleanValue();
		
		boolean isExtValid = true;

		X509Certificate rootCert=certChain[certChain.length-1];
		X509HigginsCertificate rootHigginsCert=null;
		
		if (rootCert instanceof X509HigginsCertificate)
			rootHigginsCert = (X509HigginsCertificate)rootCert;
		else
			rootHigginsCert = new X509HigginsCertificate(rootCert);
		String policyOID =  (String)signsTable.get(rootHigginsCert.getFingerprintSHA1());
		if (policyOID == null) {
			isExtValid = false;
		}
		else {
			result = policyValidationCertificate(certChain[0],policyOID,false);
			if (!result) {
				isExtValid = false;
			}
			else {
				for (int i=1;i<certChain.length-1;i++)
				{
					result = policyValidationCertificate(certChain[i],policyOID,true);	        
					if (!result) {
						isExtValid = false;
						break;
					}
				}
			}
		}
		addExtendedValiditySignToCache(certChain, isExtValid);
		return isExtValid;
	}

//	private String extractCnAttribute(String subj) {
//		if (subj == null)
//			return null;
//		Matcher m = commonNameAttributePattern.matcher(subj);
//		if (m.find()) {
//			return subj.substring(m.start() + 3, m.end());
//		}
//		return null;
//	}

	private boolean checkHostEquivalence(String realHostName, String certHostName) {
		if (certHostName.startsWith("*.")) {
			certHostName = certHostName.substring(2);
			if (realHostName.endsWith(certHostName)) {
				return true;
			}
		} else {
			if (realHostName.equals(certHostName)) {
				return true;
			}
		}
		return false;
	}

	private String getHostNameFromDN(X509Certificate cert)  {
		X509Certificate[] chain = null;
		Map map;
		try {
			map = DNParser.parse(cert.getSubjectDN().getName());
		} catch (Exception e) {
			log.error(e, e);
			return null;
		}
		String certHost = null;
		if (map.containsKey("CN")) {
			certHost = (String)map.get("CN");
		}
		else if (map.containsKey("cn")) {
			certHost = (String)map.get("cn");
		}
		return  certHost;
	}
	
	protected boolean uriValidationCertificate(X509Certificate cert, URI uri) {
		String realHost = uri.getHost();
		if (realHost != null)
			realHost = realHost.toString();
		else
			realHost = uri.toString();
		String certHost = getHostNameFromDN(cert);
		if (checkHostEquivalence(realHost, certHost)) {
			return true;
		}
		Collection altNamesC = null;
		try {
			altNamesC = cert.getSubjectAlternativeNames();
		} catch (CertificateParsingException e) {
			log.error(e, e);
		}
		if (altNamesC != null) {
			Iterator altNames = altNamesC.iterator();
			while (altNames.hasNext()) {
				Collection altNameCol = (Collection)altNames.next();
				if (altNameCol.size() == 2) {
					String altname = (String)altNameCol.toArray()[1];
					if (checkHostEquivalence(realHost, altname)) {
						return true;
					}
				}
			}
		}
		return false;
	}

	protected boolean policyValidationCertificate(X509Certificate cert,String policyOID, boolean isAnyPolicy ) {
		X509HigginsCertificate higginsCert=null;
		if (cert instanceof X509HigginsCertificate)
        	higginsCert = (X509HigginsCertificate)cert;
  		else
  			higginsCert = new X509HigginsCertificate(cert);
        
		if (higginsCert.getCertificatePolicies() == null)
        	return false;
		
		for (Iterator iter = higginsCert.getCertificatePolicies().getCertificatePolicies().iterator();iter.hasNext();)
        {
        	PolicyInformation policy = (PolicyInformation)iter.next();
        	if ( policy.getPolicyIdentifier().equals(policyOID))
        		return true;
        	if (isAnyPolicy && policy.getPolicyIdentifier().equals(ObjectIdentifier.id_ce_anyPolicy))
        		return true;
        }
		return false;
	}

	private Object getChainKey(X509Certificate[] certChain) throws NoSuchAlgorithmException {
		if (certChain == null)
			return null;
		int size = certChain.length;
		if (size == 0)
			return null;
		MessageDigest md = MessageDigest.getInstance("SHA-1");
		int hashSize = md.getDigestLength();
		byte[] res = new byte[hashSize*size];
		for (int i = 0; i < size; i++) {
			byte[] hash = md.digest(certChain[i].getPublicKey().getEncoded());
			System.arraycopy(hash, 0, res, i*hashSize, hash.length);
		}
		return Base64.encode(res);
	}

	private Boolean checkValidationInCache(X509Certificate[] certChain) {
		if (certChain == null)
			return null;
		Object key;
		try {
			key = getChainKey(certChain);
		} catch (NoSuchAlgorithmException e) {
			log.error(e, e);
			return null;
		}
		if (certCache.containsKey(key)) {
			X509Certificate[] cachedChain = (X509Certificate[])certCache.get(key);
			if (cachedChain.length >= certChain.length) {
				if (certValidCache.containsKey(key)) {
					Boolean isValid = (Boolean)certValidCache.get(key);
					log.info("Certificate validity sign found in cache : " + isValid);
					return isValid;
				}
			}
		}
		return null;
	}

	private synchronized void addValiditySignToCache(X509Certificate[] certChain, boolean isValid) {
		if (certValidCache.size() > maxCertCacheSize) {
			log.warn("Valid certificate cash size exhausted - " + maxCertCacheSize);
			return;
		}
		if (certChain == null)
			return;
		Object key;
		try {
			key = getChainKey(certChain);
		} catch (NoSuchAlgorithmException e) {
			log.error(e, e);
			return;
		}
		if (!certValidCache.containsKey(key)) {
			certValidCache.put(key, new Boolean(isValid));
			log.info("Certificate validity sign added");
		}
	}

	private Boolean checkExtendedValidationInCache(X509Certificate[] certChain) {
		if (certChain == null)
			return null;
		Object key;
		try {
			key = getChainKey(certChain);
		} catch (NoSuchAlgorithmException e) {
			log.error(e, e);
			return null;
		}
		if (certExtValidCache.containsKey(key)) {
			Boolean isValid = (Boolean)certExtValidCache.get(key);
			log.info("Certificate extended validity sign found in cache : " + isValid);
			return isValid;
		}
		else {
			log.info("Certificate extended valididy found in cache");
			return null;
		}
	}

	private synchronized void addExtendedValiditySignToCache(X509Certificate[]  certChain, boolean isExtValid) {
		if (certExtValidCache.size() > maxCertCacheSize) {
			log.warn("Extended validity certificate cash size exhausted - " + maxCertCacheSize);
			return;
		}
		if (certChain == null)
			return;
		Object key;
		try {
			key = getChainKey(certChain);
		} catch (NoSuchAlgorithmException e) {
			log.error(e, e);
			return;
		}
		if (!certExtValidCache.containsKey(key)) {
			certExtValidCache.put(key, new Boolean(isExtValid));
			log.info("Certificate extended validity sign added");
		}
	}

	public boolean validateCertificateChain(CallbackHandler handler,X509Certificate[] certChain, URI uri, CertStore certStore) {
		if (certChain == null )
			return false;
		if (certChain.length == 0)
			return false;
		if (uri != null) {
			if (!uriValidationCertificate(certChain[0],uri))
				return false;
		}
		Boolean cacheValidity = checkValidationInCache(certChain);
		if (cacheValidity != null)
			return cacheValidity.booleanValue();
		boolean isValid = false;
		try {
			// Create the parameters for the validator
			HashSet trustAnchors = new HashSet();
			Collection colSert=certStore.getCertificates(null);			
			for (Iterator iter=colSert.iterator();iter.hasNext();)
				trustAnchors.add(new TrustAnchor((X509Certificate)iter.next(), null));
            
            PKIXParameters params = new PKIXParameters(trustAnchors);

			// Disable CRL checking since we are not supplying any CRLs
			params.setRevocationEnabled(false);

			// create list which contains only last element
			List ll = new ArrayList();
			for(int i = 0, s = certChain.length; i < s; i++) {
				X509Certificate cert = certChain[i];
				// TODO temporary fix, should be fixed in cain population algorithm.
				//Chain should not contain non CA certificates, see http://forums.sun.com/thread.jspa?threadID=5244243
				// some root CA are v.1 certificates and should not be encluded into chain, otherwise CertPathValidatorException will be thrown
				if (cert.getVersion() != 1)
					ll.add(cert);
			}
			CertificateFactory certFact = CertificateFactory.getInstance("X.509");

			CertPath certPath = certFact.generateCertPath(ll);

			// Create the validator and validate the path
			CertPathValidator certPathValidator = CertPathValidator.getInstance(CertPathValidator.getDefaultType());
			CertPathValidatorResult result = certPathValidator.validate(certPath, params);
			isValid = (result != null); 
		} catch (CertPathValidatorException e) {
			log.debug(e, e);
		} catch (Exception e) {
			log.error(e, e);
		}
		addValiditySignToCache(certChain, isValid);
		return isValid;
	}

	private Object getLeafKey(X509Certificate leaf) {
		byte[] publicKey = leaf.getPublicKey().getEncoded();
		return Base64.encode(publicKey);
	}

	private X509Certificate[] getNormaizedChainFromCache(X509Certificate leaf) {
		if (leaf == null)
			return null;
		Object key = getLeafKey(leaf);
		if (certCache.containsKey(key)) {
			log.info("Certificate chain found in cache");
			return (X509Certificate[])certCache.get(key);
		}
		else
			return null;
	}

	private synchronized void addNormaizedChainToCache(X509Certificate [] certPath) {
		if (certCache.size() > maxCertCacheSize) {
			log.warn("Normalized certificate chain cash size exhausted - " + maxCertCacheSize);
			return;
		}
		if (certPath == null || certPath.length == 0)
			return;
		X509Certificate leaf = certPath[0];
		Object key = getLeafKey(leaf);
		if (!certCache.containsKey(key)) {
			certCache.put(key, certPath);
			log.info("Certificate chain added");
		}
	}

	private X509Certificate[] normalizeAsIs(X509Certificate[] certPath) throws Exception {
		if (certPath == null)
			return null;
		int size = certPath.length;
		if (size == 0)
			return null;
		if (size == 1)
			return certPath;
		HashMap certByIssuer = new HashMap();
		HashMap certBySubj = new HashMap();
		for (int i = 0; i < size; i++) {
			X509Certificate cert = certPath[i];
			String subj = cert.getSubjectDN().getName();
			String issuer = cert.getIssuerDN().getName();
			certBySubj.put(subj, cert);
			if (issuer != null && !issuer.equals(subj)) //do not add self-signed certificate
				certByIssuer.put(issuer, cert);
		}
		X509Certificate leaf = null;
		for (int i = 0; i < size; i++) {
			String subj = certPath[i].getSubjectDN().getName();
			if (!certByIssuer.containsKey(subj)) {
				leaf = certPath[i];
				break;
			}
		}
		ArrayList norm = new ArrayList();
		X509Certificate last = leaf;
		norm.add(last);
		for (int i = 1; i < size; i++) {
			if (log.isDebugEnabled()) {
				log.debug(i + " certificate of normalized chain : "  + last.getSubjectDN().getName());
			}
			String issuerPrevious = last.getIssuerDN().getName();
			if (certBySubj.containsKey(issuerPrevious)) {
				last = (X509Certificate)certBySubj.get(issuerPrevious);
				norm.add(last);
			}
			else {
				log.error("Could not normalize certificates, issuer not found");
				break;
			}
		}
		X509Certificate[] normalized = new X509Certificate[norm.size()];
		for (int i = 0, s = norm.size(); i < s; i++) {
			normalized[i] = (X509Certificate)norm.get(i);
		}
		normalized[0] = leaf;
		return normalized;
	}

	protected X509Certificate[] getParent(X509Certificate[] certificates, CertStore certStore) throws InvalidAlgorithmParameterException, NoSuchAlgorithmException, IOException, CertStoreException {
		List path = new ArrayList();
		List parentPath = null;
		if (certificates == null)
			return null;
		X509Certificate cert = null;
		X509Certificate[] preNormalized = null;
		try {
			preNormalized = normalizeAsIs(certificates);
		}
		catch(Exception e) {
			log.error(e, e);
		}
		if (preNormalized != null) {
			cert = preNormalized[0];
			X509Certificate[] cachedChain = getNormaizedChainFromCache(cert);
			if (cachedChain != null &&  cachedChain.length >= certificates.length)
				return cachedChain;
		}
		if (cert == null) {
			cert = findLeafFromCertificates(certificates);
		}
		List inputList = Arrays.asList(certificates);
		CollectionCertStoreParameters collCSP = new CollectionCertStoreParameters(inputList);
		CertStore tempStore = CertStore.getInstance("Collection", collCSP);		
		X509CertSelector selector = new X509CertSelector();
		path.add(cert);
		while (!isRootCertificate(cert)) {
			parentPath = getParent(cert, certStore, path);
			if (parentPath != null)
				break;
			selector.setSubject(cert.getIssuerX500Principal().getName());
			Collection collCertificates = tempStore.getCertificates(selector);
			if (!collCertificates.isEmpty()) {
				cert = (X509Certificate) collCertificates.iterator().next();
				if (!path.contains(cert))
					path.add(cert);
			} else
				break;
		}

		if (parentPath != null) {
			X509Certificate[] chain = new X509Certificate[parentPath.size()];
			parentPath.toArray(chain);
			addNormaizedChainToCache(chain);
			return chain;
		}
		else {
			return preNormalized;
		}
	}

	protected List getParent(X509Certificate certificate, CertStore certStore, List path) {
		if (certificate != null) {
			try {
				String subj = certificate.getSubjectDN().getName();
				String issuer = certificate.getIssuerDN().getName();
				if (subj.equals(issuer)) {
					return path;
				}
			}
			catch(Exception e) {
				log.warn(e, e);
				return path;
			}
			X509Certificate cert=null;
			X509HigginsCertificate higginsCert=ConvertHelper.getX509HigginsCertificate(certificate);			
			
			X509Certificate [] certs=getCertChainFromStore(higginsCert.getFingerprintSHA1());
			if (certs!=null && certs.length>0)
			{
				List localStoreList=Arrays.asList(certs);
				for (Iterator iter=localStoreList.iterator();iter.hasNext();)
				{
					cert=(X509Certificate)iter.next();
					if (!path.contains(cert))
						path.add(cert);
				}
				cert = (X509Certificate)path.get(path.size()-1);
			}
			if (cert == null)
				cert = extendExternalPath(certificate);
			if (cert == null)
				cert = extendLocalPath(certificate, certStore);

			if (cert != null) {
				try {
					String subj = certificate.getSubjectDN().getName();
					String issuer = cert.getSubjectDN().getName();
					if (subj.equals(issuer)) {
						return path;
					}
				}
				catch(Exception e) {
					log.warn(e, e);
					return path;
				}
				if (!path.contains(cert))
					path.add(cert);
				if (!isRootCertificate(cert))
					return getParent(cert, certStore, path);
				else
					return path;
			}
		}
		return null;
	}

	protected X509Certificate extendExternalPath(X509Certificate certificate) {
		X509HigginsCertificate certHiggins = null;
		if (certificate instanceof X509HigginsCertificate)
			certHiggins = (X509HigginsCertificate)certificate;
		else
			certHiggins = new X509HigginsCertificate(certificate);
		AuthorityInformationAccessExtension extention = certHiggins.getAuthorityInformationAccess();
		if (extention != null) {
			URI uri = null;
			for (Iterator iter = extention.getAccessDescriptions().iterator(); iter.hasNext();) {
				AccessDescription accessDescription = (AccessDescription) iter.next();
				if (accessDescription.getAccessMethod().equals(ObjectIdentifier.id_ad_caIssuers)) {
					if (accessDescription.getAccessLocation().getValue() instanceof URIName) {
						uri = ((URIName) accessDescription.getAccessLocation().getValue()).getURI();
					}
				}
			}
			if (uri != null) {
				InputStream stream = null;
				try {
					stream = uri.toURL().openStream();

					BufferedInputStream bis = new BufferedInputStream(stream);
					CertificateFactory cf = CertificateFactory.getInstance("X.509");
					X509Certificate cert = (X509Certificate) cf.generateCertificate(bis);
					return new X509HigginsCertificate(cert);
				} catch (Exception e) {
					log.error(e, e);
				} finally {
					if (stream != null)
						try {
							stream.close();
						} catch (IOException e) {
							log.error(e, e);
						}
				}
			}
		}
		return null;
	}

	protected X509Certificate extendLocalPath(X509Certificate certificate, CertStore certStore) {
		try {
			X509Certificate parentCert = null;
			X509CertSelector selector = new X509CertSelector();
			selector.setSubject(certificate.getIssuerX500Principal().getName(X500Principal.CANONICAL));

			if (certStore != null) {
				Collection list = certStore.getCertificates(selector);

				if (!list.isEmpty())
				{
					parentCert = (X509Certificate) list.toArray()[0];
					return new X509HigginsCertificate(parentCert);
				}
			}

		} catch (Exception e) {
			log.error(e, e);
		}
		return null;
	}

	protected boolean isRootCertificate(X509Certificate cert) {

		if (cert == null)
			return false;
		X500Principal issuer_dn = null;
		X500Principal subject_dn = null;
		issuer_dn = cert.getIssuerX500Principal();
		subject_dn = cert.getSubjectX500Principal();
		return subject_dn.equals(issuer_dn);
	}

	public X509Certificate findLeafFromCertificates(X509Certificate [] listCerts) {
		if (listCerts == null)
			return null;
		if (0 == listCerts.length)
			return null;
		if (1 == listCerts.length)
			return (X509Certificate) listCerts[0];

		// Find the leaf certificate (the one that no other certificate points to as its issuer
		HashMap issuerMap = new HashMap();
		ArrayList subjectList = new ArrayList();

		for (int iCert = 0; iCert < listCerts.length; iCert++) {
			X509Certificate x509Certificate = (X509Certificate) listCerts[iCert];
			String strSubject = x509Certificate.getSubjectX500Principal().getName();
			String strIssuer = x509Certificate.getIssuerX500Principal().getName();
			issuerMap.put(strIssuer, strSubject);
			subjectList.add(strSubject);
		}
		for (int iCert = 0; iCert < subjectList.size(); iCert++) {
			String strSubject = (String) subjectList.get(iCert);
			if (null == issuerMap.get(strSubject)) {
				// log.trace("Found LEaf Certificate for Subject: " + strSubject);
				return (X509Certificate) listCerts[iCert];
			}
		}
		return null;
	}
	
	public static CertStore joinCertStore (CertStore certStore1, CertStore certStore2)
	{
		Collection certificates = new ArrayList();
		if (certStore1 == null)
			return certStore2;
		if (certStore2 == null)
			return certStore1;
		try {
			certificates.addAll(certStore1.getCertificates(null));
		} catch (CertStoreException e) {
			log.error(e,e);
		}
		try {
			certificates.addAll(certStore2.getCertificates(null));
		} catch (CertStoreException e) {
			log.error(e,e);
		}
		CollectionCertStoreParameters collCSP = new CollectionCertStoreParameters(certificates);
		try {
			return CertStore.getInstance("Collection", collCSP);
		} catch (InvalidAlgorithmParameterException e) {
			log.error(e,e);
		} catch (NoSuchAlgorithmException e) {
			log.error(e,e);
		}
		return null;
	}
	
	public static X509Certificate[] getCertificatesFromByteArray(final byte [] byteArray) throws CertificateException
	{
		ByteArrayInputStream bis = new ByteArrayInputStream(byteArray);

		CertificateFactory cf = CertificateFactory.getInstance("X.509");
		Collection colSert = cf.generateCertificates(bis);
		int index = 0;
		X509Certificate [] certs = new X509Certificate[colSert.size()];
		for (Iterator iter = colSert.iterator(); iter.hasNext();) {
			certs[index] = (X509Certificate) iter.next();
			index++;
		}
		return certs;
	}
	
	static
	{		
		// companies names from http://cabforum.org/forum.html and http://en.wikipedia.org/wiki/Extended_Validation_Certificate
		
//		A-Trust GmbH
//		Certum
		signsTable.put("66 31 bf 9e f7 4f 9e b6 c9 d5 a6 0c ba 6a be d1 f7 bd ef 7b", "1.3.6.1.4.1.6449.1.2.1.5.1");// Comodo CA Ltd 
		signsTable.put("5f 43 e5 b1 bf f8 78 8c ac 1c c7 ca 4a 9a c6 22 2b cc 34 c6", "1.3.6.1.4.1.6334.1.100.1");// Cybertrust 
		signsTable.put("5f b7 ee 06 33 e2 59 db ad 0c 4c 9a e6 d3 8f 1a 61 c7 dc 25", "2.16.840.1.114412.2.1");// DigiCert, Inc.
		signsTable.put("f0 bb 01 d1 d8 31 98 51 6a 4f f1 6d 8a b5 ef ab 74 c7 cc a3", "2.16.528.1.1001.1.1.1.12.6.1.1.1");// DigiNotar				
//		Echoworx Corporation
		signsTable.put("b3 1e b1 b7 40 e3 6c 84 02 da dc 37 d4 4d f5 d4 67 49 52 f9", "2.16.840.1.114028.10.1.2");// Entrust, Inc. 
		signsTable.put("32 3c 11 8e 1b f7 b8 b6 52 54 e2 e2 10 0d d6 02 90 37 f0 96", "1.3.6.1.4.1.14370.1.6");// GeoTrust, Inc.
//		Getronics PinkRoccade
		signsTable.put("75 e0 ab b6 13 85 12 27 1c 04 f8 5f dd de 38 e4 b7 24 2e fe", "1.3.6.1.4.1.4146.1.1");// GlobalSign 
		signsTable.put("27 96 ba e6 3f 18 01 e2 77 26 1b a0 d7 77 70 02 8f 20 ee e4", "2.16.840.1.114413.1.7.23.3");// GoDaddy.com, Inc. 
//		IdenTrust, Inc. 
//		ipsCA, IPS Certification Authority s.l.
//		Izenpe S.A.
		signsTable.put("74 f8 a3 c3 ef e7 b3 90 06 4b 83 90 3c 21 64 60 20 e5 df ce", "1.3.6.1.4.1.782.1.2.1.8.1");// Network Solutions, LLC 
		signsTable.put("ca 3a fb cf 12 40 36 4b 44 b2 16 20 88 80 48 39 19 93 7c f7", "1.3.6.1.4.1.8024.0.2.100.1.2");// QuoVadis Ltd. (https://www.quovadis.bm) 
//		RSA Security, Inc.
//		SECOM Trust Systems CO., Ltd.
//		Skaitmeninio sertifikavimo centras (SSC)
		signsTable.put("ad 7e 1c 28 b0 64 ef 8f 60 03 40 20 14 c3 d0 e3 37 0e b5 8a", "2.16.840.1.114414.1.7.23.3");// Starfield Technologies
//		SwissSign AG
//		TC TrustCenter GmbH
//		TDC Certification Authority
		signsTable.put("87 82 c6 c3 04 35 3b cf d2 96 92 d2 59 3e 7d 44 d9 34 ff 11", "2.16.840.1.114404.1.1.2.4.1");// Trustwave
		signsTable.put("91 c6 d6 ee 3e 8a c8 63 84 e5 48 c2 99 29 5c 75 6c 81 7b 81", "2.16.840.1.113733.1.7.48.1");// Thawte, Inc. 
//		Trustis Limited
		signsTable.put("4e b6 d5 78 49 9b 1c cf 5f 58 1e ad 56 be 3d 9b 67 44 a5 e5", "2.16.840.1.113733.1.7.23.6");// VeriSign, Inc. 
//		Wells Fargo Bank, N.A.		
	}
}
