/*******************************************************************************
 * Copyright (c) 2009 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.registry;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.registry.IConfiguration;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Load/save provider configuration from properties file.
 */
public class PropertiesConfiguration implements IConfiguration {
	private Log log = LogFactory.getLog(PropertiesConfiguration.class);

	private HashMap defaults = null;

	private HashMap properties = null;

	private String fileName = null;

	private String providerId = null;

	public PropertiesConfiguration(String providerId, String providerConfigFile) throws RegistryConfigurationException {
		this.providerId = providerId;
		this.fileName = providerConfigFile;
		loadDefaults();
		loadProperties();
	}

	/**
	 * @throws RegistryConfigurationException
	 */
	private void loadProperties() throws RegistryConfigurationException {
		properties = new HashMap();
		log.info("config=\"" + fileName + "\"");
		FileInputStream is = null;
		try {
			File f = new File(fileName);
			if (f.exists() && f.isFile()) {
				is = new FileInputStream(f);
				properties.putAll(loadProperties(is));
			} else {
				log.warn("Configuration file not found for provider " + providerId);
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new RegistryConfigurationException(e);
		} finally {
			if (is != null) {
				try {
					is.close();
				} catch (Exception e) {
					log.error(e, e);
				}
			}
		}
		log.info("properties=\"" + properties + "\"");
	}

	/**
	 * @throws RegistryConfigurationException
	 */
	private void loadDefaults() throws RegistryConfigurationException {
		defaults = new HashMap();
		InputStream is = null;
		try {
			is = Thread.currentThread().getContextClassLoader().getResourceAsStream(providerId + ".ini");
			if (is != null) {
				defaults.putAll(loadProperties(is));
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new RegistryConfigurationException(e);

		} finally {
			if (is != null) {
				try {
					is.close();
				} catch (IOException e) {
					log.error(e, e);
				}
			}
		}
		log.info("defaults=\"" + defaults + "\"");
	}

	/**
	 * @param is
	 * @return
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */
	private HashMap loadProperties(InputStream is) throws ParserConfigurationException, SAXException, IOException {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		Document doc = null;
		DocumentBuilder db = dbf.newDocumentBuilder();
		doc = db.parse(new InputSource(is));
		Element root = doc.getDocumentElement();
		return loadProperties(root);
	}

	/**
	 * @param e
	 * @return
	 */
	private HashMap loadProperties(Element e) {
		HashMap props = new HashMap();
		NodeList nl = e.getChildNodes();
		for (int i = 0; i < nl.getLength(); i++) {
			Node node = nl.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE && node.getNodeName().equals("parameter")) {
				Element el = (Element) node;
				String name = el.getAttribute("name");
				if (name.length() > 0) {
					String value = el.getAttribute("value");
					props.put(name, value);
				}
			}
		}
		return props;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.registry.IConfiguration#getPropertyNames()
	 */
	public Iterator getPropertyNames() {
		HashSet set = new HashSet();
		set.addAll(defaults.keySet());
		set.addAll(properties.keySet());
		return set.iterator();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.registry.IConfiguration#getProperty(java.lang.String)
	 */
	public Object getProperty(String key) {
		if (properties.containsKey(key)) {
			return properties.get(key);
		}
		if (defaults.containsKey(key)) {
			return defaults.get(key);
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.registry.IConfiguration#getProperty(java.lang.String,
	 * java.lang.Object)
	 */
	public Object getProperty(String key, Object defaultValue) {
		Object prop = getProperty(key);
		if (prop != null) {
			return prop;
		}
		return defaultValue;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.registry.IConfiguration#setProperty(java.lang.String,
	 * java.lang.Object)
	 */
	public void setProperty(String key, Object property) throws RegistryConfigurationException {
		properties.put(key, property);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.registry.IConfiguration#save()
	 */
	public void save() {
		throw new UnsupportedOperationException();
	}

}
