/*******************************************************************************
 * Copyright (c) 2006-2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.registry;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

/**
 * A registry for service provider instances.
 * 
 * A service is an interface which extends
 * {@link org.eclipse.higgins.registry.IServiceProvider} interface. A service
 * provider is a specific implementation of a service. The classes in a provider
 * implement the interface defined by the service.
 * <p>
 * 
 * Higgins Registry uses registry extensions to lookup for available service
 * providers.
 * <p>
 * 
 * When base implementation is constructed it doesn't contain any registry
 * extensions and so, requires additional configuration. For example:
 * 
 * <code>
 * <pre>
 * HigginsRegistry&lt;IServiceProvider&gt; r = new HigginsRegistry&lt;IServiceProvider&gt;();
 * r.addExtension(new IIORegistryExtension&lt;IServiceProvider&gt;(
 * 		IServiceProvider.class));
 * r.loadProviders();
 * </pre>
 * </code>
 * 
 * @param <E>
 *            the type of the service provider
 * 
 * @see org.eclipse.higgins.registry.IServiceProvider
 * @see org.eclipse.higgins.registry.IRegistryExtension
 */
public abstract class HigginsRegistry {

	protected List extensions = new ArrayList();

	protected HashMap providers = new HashMap();
	
	protected Class type;

	protected String configPath = System.getProperty("user.home")
			+ File.separator + ".higgins";

	public HigginsRegistry(Class type) {
		this(type, System.getProperty("user.home") + File.separator + ".higgins");
	}

	public HigginsRegistry(Class type, String configPath) {
		this.type = type;
		this.configPath = configPath;
	}

	public void addExtension(IRegistryExtension ext) {
		if (!extensions.contains(ext)) {
			extensions.add(ext);
		}
	}

	public void loadProviders() {
		for (int i = 0; i < extensions.size(); i++) {
			IRegistryExtension e = (IRegistryExtension) extensions.get(i);
			loadProviders(e);
		}
	}

	protected void loadProviders(IRegistryExtension e) {
		for (Iterator itr = e.getServiceProviders(); itr.hasNext(); ) {
			Object o = itr.next();
			if (o instanceof IServiceProvider) {
				add((IServiceProvider) o);
			} else {
				add(o);
			}
			//IServiceProvider p = (IServiceProvider) itr.next();
			//add(p);
		}
	}

	public synchronized Iterator getServiceProviders() {
		return providers.values().iterator();
	}

	public synchronized Object getServiceProvider(String id) {
		if (id == null) {
			throw new IllegalArgumentException("Argument \"id\" can't be null.");
		}
		return providers.get(id);
	}

	protected synchronized void add(Object provider) {
		if (provider == null) {
			new IllegalArgumentException("Argument \"provider\" can't be null.");
		}
		
		if (type.isInstance(provider)) {
			
			if (provider instanceof IServiceProvider) {
				add((IServiceProvider) provider); 
			} else {
				String id = provider.getClass().getName();
				
				if (providers.get(id) == null) {
					providers.put(id, provider);
				}
			}
		} else {
			System.err.println(this.getClass().getName() + "::add(Object provider) provider=\"" + provider + "\" is not instance of \"" + type + "\"");
		}
	}

	protected synchronized void add(IServiceProvider provider) {
		/*
		if (provider == null) {
			new IllegalArgumentException("Argument \"provider\" can't be null.");
		}
		*/

		String id = provider.getID();

		if (id == null) {
			new IllegalArgumentException("Provider's ID can't be null.").printStackTrace();
			return;
		}

		if (providers.get(id) == null) {
			IConfiguration config = provider.getConfiguration();
			File f = getConfigFile(id);
			if (config != null) {
				config.load(f);
			}
			providers.put(id, provider);
		}
	}

	public synchronized void remove(IServiceProvider provider) {
		if (provider == null) {
			new IllegalArgumentException("Argument \"provider\" can't be null.");
		}

		String id = provider.getID();

		if (id == null) {
			new IllegalArgumentException("Provider's ID can't be null.");
		}

		providers.remove(id);
	}

	public File getConfigFile(String id) {
		return new File(getConfigPath() + File.separator + id + ".ini");
	}

	public String getConfigPath() {
		return configPath + File.separator + getConfigFolder();
	}

	protected abstract String getConfigFolder();
}
