/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Suresh Chari (IBM Corporation) - initial implementation
 *******************************************************************************/ 
package org.eclipse.higgins.rp.icard;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.CertificateException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.rp.AuthProtocolHandler;
import org.eclipse.higgins.rp.DispatchCallback;
import org.eclipse.higgins.rp.ResultCallback;
import org.eclipse.higgins.rp.SessionContext;
import org.eclipse.higgins.rp.icard.server.ICardConstants;
import org.eclipse.higgins.rp.icard.token.identity.DigitalIdentity;
import org.eclipse.higgins.sts.api.IElement;
import org.eclipse.higgins.sts.common.Constants;
import org.eclipse.higgins.sts.spi.IXMLSecurityExtension;
import org.eclipse.higgins.sts.xmlsecurity.apache.XMLSecurityApacheExtension;

public class ICardProtocolHandler extends AuthProtocolHandler {

	private static final long serialVersionUID = 1L;
    protected static Log log = LogFactory.getLog(ICardProtocolHandler.class.getName());
    
	// States	
	private static final int ERRORSTATE = 100;
	private static final int INITIALSTATE = 0;
	private static final int REDIRECTED = 1;
	private static final int AUTHENTICATED = 2;
	
	
	private static boolean init_errors = false;
	private static String _errorpage = null;
	
	// Parameter name for the XML Token
	private static final String TOKENPARAMETERNAME = "xmlToken";	
	
	private ResultCallback	_resultCallback = null;
	private DispatchCallback  _dispatchCallback = null;
	private SessionContext	_sessionContext = null;
	
	List _reqClaims = null;
	List _optClaims = null;
	
	List _reqClaimValues = null;
	List _optClaimValues = null;
	
	private int state = INITIALSTATE;

	private static String _loginFormLoc = null;
	private static KeyStore keyStore=null;
	private boolean keystoreInited = false;
	private static String keyStorePwd = null;
	private static String keyStoreAlias = null;
	private static String keyStoreName = null;
	private static String keyStoreType = null;
	
	private static Map ppidToDigitalIdentityMap = null;
	private static String _xmlSecConfigFile = "config.xml";
	private static final String LOGIN_PAGE_ATTRIBUTE = "icardLoginPage";
	private static final String XMLSEC_CONFIG_ATTRIBUTE = "xmlsecconfig";
	
	
	public void registerCallbacks(ResultCallback resultCallback,
			DispatchCallback dispatchCallback, SessionContext ctxt) {
		_resultCallback = resultCallback;
		_dispatchCallback= dispatchCallback;
		_sessionContext = ctxt;
		
	}	

	public void authenticate( Map requestHeaders, Map requestParams) {
		if(init_errors)
			state = ERRORSTATE;

		switch (state){
		case INITIALSTATE:
			log.debug("Current state is INITIALSTATE");
			// SNC: why is this here?
			_sessionContext.storeAttribute("ErrorPage", _errorpage);
			_dispatchCallback.redirect( addAuthSessionToLocation(_loginFormLoc ) );
			state = REDIRECTED;
			return;

		case REDIRECTED:
			log.debug("Current state is REDIRECTED");
			processUserToken( requestParams );
			return;

		case AUTHENTICATED:
		case ERRORSTATE:
			log.debug("Current state is AUTHENTICATED or ERRORSTATE");
			state= ERRORSTATE;
			_resultCallback.handleFailure( ResultCallback.RP_AUTHENTICATION_INTERNAL_ERROR );
			return;

		}		
	}

	
	// Routine which parses the token from the user.
	public void processUserToken( Map reqParams ){
		
		DigitalIdentity di = null;
		String strToken = null;
		String[] strTokenValues = (String[])reqParams.get( TOKENPARAMETERNAME );
		if (strTokenValues !=null )
			strToken = strTokenValues[0];
		
		try	{
			if (strToken != null) {
				
				org.eclipse.higgins.sts.api.IElement elemToken = new org.eclipse.higgins.sts.common.Element();
				elemToken.set(strToken);
				log.debug("Setting system property org.apache.xml.security.resource.config to " + _xmlSecConfigFile);
				System.setProperty("org.apache.xml.security.resource.config", _xmlSecConfigFile);
				IXMLSecurityExtension secext = new XMLSecurityApacheExtension(); 
				IElement ie = null;
				secext.configure(new HashMap(), null, new HashMap());
				if (!keystoreInited ){
					keystoreInited = true;
					keyStore = getKeyStore(keyStoreType, _sessionContext.getRealPath( keyStoreName ), keyStorePwd.toCharArray());
				}
				PrivateKey key = (PrivateKey)keyStore.getKey(keyStoreAlias,keyStorePwd.toCharArray());
				log.info("Decrypt token using key " + key + " key algorithm " + key.getAlgorithm());
				ie = secext.DecryptElement(elemToken, (PrivateKey)(keyStore.getKey(keyStoreAlias,keyStorePwd.toCharArray())));
				log.info("Decrypted token looks like\n"+ie.getAs(java.lang.String.class));
				// verify that it's signed correctly
				secext.VerifyEnveloped(ie, new Constants());
				di = DigitalIdentity.fromXml(ie);
				log.info(di);
				ppidToDigitalIdentityMap.put(di.getPpid(), di);
				
			}
		} catch (final Exception e)	{
			log.error("Unable to proecess token");
			e.printStackTrace(new java.io.PrintStream(System.out));
		}
		
		if (di != null) {
			log.info("Sucessfully authenticated token");
			_resultCallback.handleSuccess( _sessionContext.getProtectedResource(), new ICardRPToken( di ));
			state = AUTHENTICATED;
			// Now we are done.	
		} else {
			log.info("Error authenticating token");
			_resultCallback.handleFailure(ResultCallback.RP_AUTHENTICATION_FAILURE);
			state = ERRORSTATE;
		}		
	}
	
	
	public static String getSupportedTokenName() {
		return "urn:oasis:names:tc:SAML:1.0:assertion";
	}

	public void init(List requiredClaims, List optionalClaims) {
		_reqClaims = requiredClaims;
		_optClaims = optionalClaims;
	}

	public List getOptionalClaims() {
		return _optClaims;
	}

	public List getRequiredClaims() {
		return _reqClaims;
	}

	private static KeyStore getKeyStore(String type, String name, char[] pw) {
		KeyStore answer = null;
		try {
			if (type == null)
				type = KeyStore.getDefaultType();
			log.info("name: " + name + " type: " +type);
			answer = KeyStore.getInstance(type);
			answer.load(new FileInputStream(name), pw);
		}  catch ( KeyStoreException kse ){
			log.info("KeyStore Exception while loading keystore ");
			kse.printStackTrace();
			throw new RuntimeException();
		} catch ( CertificateException ce){
			log.info("Certificate Exception while loading keystore ");
			ce.printStackTrace();
			throw new RuntimeException();		
		} catch ( FileNotFoundException fnfe){
			log.info("KeyStore File not found "+name);
			fnfe.printStackTrace();
			throw new RuntimeException();			
		} catch ( IOException ioe){
			log.info(" IOException while reading file "+name);
			ioe.printStackTrace();
			throw new RuntimeException();
		} catch ( NoSuchAlgorithmException nsae ){
			log.info("No such algorithm");
			nsae.printStackTrace();
			throw new RuntimeException();
		}
		return answer;
	}	
	
	public static void init(Map initParams) {
		// NOTE: This is assumed to be the absolute path to the keystore.
		keyStoreName = (String)initParams.get( ICardConstants.KEYSTORENAME );
		
		String kspwd = (String)initParams.get( ICardConstants.KEYSTOREPW );
		keyStoreType = (String)initParams.get( ICardConstants.KEYSTORETYPE );
		String ksalias = (String)initParams.get( ICardConstants.KEYSTOREKEYALIAS );
		// PKA - still need to figure out best way to set error page
		_errorpage = (String)initParams.get("icardErrorPage");
		log.info("initializing");
		
		// check that params are set. 
		if ( ( keyStoreName== null ) || ( kspwd == null )|| ( ksalias == null ) ){
			init_errors = true;
			if(keyStoreName== null)
				log.warn(ICardConstants.KEYSTORENAME+"not set");
			if(kspwd == null)
				log.warn(ICardConstants.KEYSTOREPW+"not set");
			if(ksalias == null)
				log.warn(ICardConstants.KEYSTOREKEYALIAS+"not set");
			log.error("returning error state");
			return;
		}
		// it's not an error if KEYSTORETYPE is not set since a default value will be used.
		if ( keyStoreType == null )
			log.warn(ICardConstants.KEYSTORETYPE+"not set");
		
		
		keyStoreAlias = ksalias;
		keyStorePwd = kspwd;			
		ppidToDigitalIdentityMap = new HashMap();
		
		_loginFormLoc = (String)initParams.get(LOGIN_PAGE_ATTRIBUTE);
		_xmlSecConfigFile = (String)initParams.get(XMLSEC_CONFIG_ATTRIBUTE);	
		
	
	}

	public void setOptionalClaims(List optionalClaims) {
		_optClaims = optionalClaims;
	}

	public void setRequiredClaims(List requiredClaims) {
		_reqClaims = requiredClaims;
	}

}
