/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Suresh Chari (IBM Corporation) - initial implementation
 *******************************************************************************/ package org.eclipse.higgins.rp;


 import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;



 /**
  * 
  * The top level interface for the protocol finite state machine handler.
  * 
  * At each step of the authentication process the relying party application can
  * call authenticate to continue the authentication process passing in the
  * parameters (POST and query) in the current request for the user.
  */
 public abstract class AuthProtocolHandler  implements Serializable{

	 
	 /**
	 * query parameter which contains the value of the authentication session
	 * for the current request
	 */
	public static final String AUTH_SESSION_PARAM = "authsession";
	 
	 /**
	 * String to identify the authentication session this instance of the handler is
	 * processing. 
	 *
	 */
	private String _sessionId = null;
     /**
      * Returns a session ID which is bound to the authentication session 
      */
     public String getSessionId(){
    	 return _sessionId;
     }

     /**
      * Sets the sessionID bound to this authentication session
      */
     
     public  void setSessionId(String sessionId){
    	 _sessionId = sessionId;
     }
     
     
     /**
      * Register callbacks to be be used in the authentication process
      * 
      * @param resultCallback
      *                to notify the application at the end of the process
      * @param dispatchCallback
      *                used to have the app server redirect to other resources
      * @param sCtxt
      *                sessionContext used to store attributes and obtain other 
      *                context parameters
      */
     public abstract void registerCallbacks(ResultCallback resultCallback,
             DispatchCallback dispatchCallback, SessionContext sCtxt);

     /**
      * Call to initiate or continue the authentication process.
      * 
      * @request requestHeaders headers in the request. Assume that multiple
      * headers with the same name are merged as allowed by the RFC
      * 
      * @param requestParams both the POST as well as query params in the string
      */
     public abstract void authenticate(Map requestHeaders,
             Map requestParams);

     /**
      * 
      * Calls to initialize this object with the list of required and optional
      * claim types. This is done on a per protected resource basis to support
      * the use case that different resources require different claims
      * 
      */
     public abstract void setRequiredClaims(List requiredClaims);

     public abstract void setOptionalClaims(List optionalClaims);

     /**
      * Protocol handler specific initialization. Subclasses SHOULD override
      * this method
      * 
      * @param initParams
      */
     public static void init(Map initParams){
     
     }

     /**
      * Static token method which MUST be over-ridden by the subclass
      */
     public static String getSupportedTokenName(){
         return null;
     }

     /**
      * Getter method for the list of required Claims
      */
     public abstract List getRequiredClaims();

     /**
      * Getter method for the list of optional Claims
      */
     public abstract List getOptionalClaims();
     
     /**
 	 * Utility function which adds the authentication context as a request parameter
 	 * 
 	 * @param location 
 	 * 
 	 */
 	public String addAuthSessionToLocation( String location ){
 		String result= null;
 		URI uri=null;
 		try{
 			 uri= new URI(location);
 		} catch (URISyntaxException exc1){
 			// Not sure what do here.
 			result=location;	
 		}
 		String queryStr = uri.getQuery();
 		if ( queryStr == null ){
 			result=location+"?"+AUTH_SESSION_PARAM+"="+getSessionId();
 		} else
 			if ( queryStr.indexOf( AUTH_SESSION_PARAM ) == -1 )
 				result= location+"&"+AUTH_SESSION_PARAM+"="+getSessionId();
 		return result;
 	}
 	
 }
