/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Bruce Rich (IBM Corporation) - initial API and implementation
 *    Paula Austel (IBM Corporation) - adaptation to new Relying Party Interface
 *******************************************************************************/

package org.eclipse.higgins.rp.servlet.server;

import java.io.IOException;
import java.security.SecureRandom;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.rp.AuthProtocolHandler;

/**
 * Authentication filter This is an authentication filter for relying party
 * applications. The filter checks to see if a token is found in the session
 * (indicating that a sucessful authentication has already taken place.) The
 * filter and the web pages that require authentication are specified in the
 * web.xml file for the application. The URL of the protected resource is
 * invoked a few times during the authentication process. Each time the
 * authenticaton process is in a different state. The filter sets the initial
 * authentication state. The state is advanced inside the invoker class.
 * 
 * @author Paula Austel
 * 
 */
public class AuthNFilter implements Filter {

	protected static Log log = LogFactory.getLog(AuthNFilter.class.getName());
	private static final int BYTE_ARRAY_SIZE = ((23 * 5) / 8) + 1;
	private static final SecureRandom SECURE_RANDOM = new SecureRandom();

	public static final String getNextAuthSession() {
		byte[] genBytes = new byte[BYTE_ARRAY_SIZE];
		String randomID = null;
		SECURE_RANDOM.nextBytes(genBytes);
		try {
			randomID = new String
				(org.apache.commons.codec.binary.Base64.encodeBase64
					(genBytes)).replace
					('+', 'P');
		} catch (Exception exc) {
			throw new RuntimeException(" Base64 encoding error ");
		}
		return randomID;
	}

	public void destroy() {

	}

	public void init(FilterConfig arg0) throws ServletException {

	}

	public void doFilter(ServletRequest request, ServletResponse response,
			FilterChain fChain) throws IOException, ServletException {

		HttpServletRequest req = (HttpServletRequest) request;
		HttpServletResponse resp = (HttpServletResponse) response;
		HttpSession httpsession = req.getSession();
		boolean runFilter = true;
		boolean concurrentRequest = false;
		String authSession = null;

		log.info(req.getRemoteHost() + " tried to access "
				+ req.getRequestURL() + " on " + new Date());
		// See if we have a token in the session, if so then the user is already
		// authenticated
		String token_id = (String) httpsession
				.getAttribute(CxtConstants.RET_TOKEN_ID);

		// get ongoing authentication session
		authSession = req.getParameter(AuthProtocolHandler.AUTH_SESSION_PARAM);

		if (null == token_id)
			runFilter = false;
		else {
			log.info("User is already authenticated");
			if (authSession != null) {
				runFilter = false; // need to redirect to the protectedResource
				concurrentRequest = true;
			}
		}
		// if user is already authenticated then run the next filter in the
		// chain.
		if (runFilter)
			fChain.doFilter(request, response);
		else {
			if (concurrentRequest)
				handleConcurrentRequest(authSession, req, resp);
			else {
				// User is NOT authenticated. Check if this is the first time
				// we're getting here for this request
				Map _mp = null;
				if (authSession == null) {

					// start a new context
					authSession = getNextAuthSession();
					// Save the request and response to the session
					_mp = new HashMap();

					_mp.put(CxtConstants.PROTECTED_RESOURCE, req
							.getRequestURL().toString()
							+ ((req.getQueryString() == null) ? "" : "?"
									+ req.getQueryString()));
					_mp.put(CxtConstants.AUTHENTICATION_STATE_URI,
							CxtConstants.INIT_STATE);
					httpsession.setAttribute(authSession, _mp);
				} else {
					_mp = (Map) httpsession.getAttribute(authSession);
					_mp.put(CxtConstants.AUTHENTICATION_STATE_URI,
							CxtConstants.PROCESS_STATE);
				}
				_mp.put(CxtConstants.REQUEST, req);
				_mp.put(CxtConstants.RESPONSE, resp);
				Invoker.invoke(authSession, httpsession);
			}
		}
	}

	/*
	 * Handles the case when there were multiple requests which triggered
	 * authentication flows. If one succeeds then this routine assumes that the
	 * other authentication requests can immediately proceed to completion since
	 * the user has already authenticated. NOTE: this may NOT be the best way to
	 * handle but since we are using the RP interface as a federation not an
	 * authorization API this is correct
	 */
	public static void handleConcurrentRequest(String authSession,
			HttpServletRequest req, HttpServletResponse resp) {

		HttpSession httpsession = req.getSession();

		Map _mp = (Map) httpsession.getAttribute(authSession);

		if (_mp != null) {
			String redirect = (String) _mp.get(CxtConstants.PROTECTED_RESOURCE);
			httpsession.removeAttribute(authSession);
			// We can remove this authSession since we'll no longer need this.
			try {
				resp.sendRedirect(redirect);
			} catch (IOException exc) {
				throw new RuntimeException(" Unable to redirect user to "
						+ redirect);
			}

		} else {
			// case where we have an authenticated user who makes a request
			// with an non-existent session. Do nothing.
			return;
		}

	}

	
}
