/** 
 * Copyright (c) 2009 Parity Communications, Inc.  
 * All rights reserved. This program and the accompanying materials  
 * are made available under the terms of the Eclipse Public License v1.0  
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 *  
 * Contributors: 
 * Alexander Yuhimenko - implementation 
 * 
 */

package org.eclipse.higgins.rpps.webservices;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.rmi.AccessException;
import java.util.ArrayList;
import java.util.StringTokenizer;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServlet;
import javax.xml.rpc.soap.SOAPFaultException;

import org.apache.axis.MessageContext;
import org.apache.axis.transport.http.HTTPConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.rpps.core.IRppsService;
import org.eclipse.higgins.rpps.core.RppsException;
import org.eclipse.higgins.rpps.core.impl.RppsService;
import org.eclipse.higgins.rpps.core.tobj.AccessTokenTO;
import org.eclipse.higgins.rpps.core.tobj.AuthCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.FormatDescriptorTO;
import org.eclipse.higgins.rpps.core.tobj.ICardResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ImportCardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.MapTO;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.tobj.SelectorAuthCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.SelectorClientTO;
import org.eclipse.higgins.rpps.core.tobj.StsFaultException;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;
import org.eclipse.higgins.rpps.core.tobj.UsernamePasswordAuthCredentialTO;
import org.eclipse.higgins.rpps.utilities.SoapUtils;
import org.eclipse.higgins.sync.auth.entity.AccessTokenEntity;
import org.eclipse.higgins.user.login.UserAccount;
import org.eclipse.higgins.user.profile.entity.UserProfile;

/**
 * RPPS WS endpoint for Serialized Client server side implementation.
 * 
 * @see http://wiki.eclipse.org/CardSync_Authentication
 * @see https://camelot.parityinc.net/mediawiki/index.php/Serialized_Client_Rpps
 * @see WWM-1820: Add axis Rpps WS endpoint for supporting new authentication workflow
 *      http://graceland.parityinc.net:8090/browse/WWM-1820
 * 
 * @author Alexander Yuhimenko
 * 
 */
public class SCRPPSServiceImpl implements ISCRPPSServiceSE {

	/**
	 * 
	 */
	protected static final String AUTHORIZATION_HEADER = "Authorization";

	private static final Log LOG = LogFactory.getLog(SCRPPSServiceImpl.class);

	private final SoapUtils soapUtils = new SoapUtils();

	protected IRppsService rppsService;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.SCRPPSService#addCategory(org.eclipse.higgins.rpps.core.tobj.CategoryTO)
	 */
	public CategoryTO addCategory(CategoryTO category) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().addCategory(account, category);

		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/**
	 * @return the soapUtils
	 */
	public SoapUtils getSoapUtils() {
		return soapUtils;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#addClaimValuesMRU(java.lang.String, java.lang.String)
	 */
	public void addClaimValuesMRU(String claimType, String claimValue) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			getRppsService().addClaimValuesMRU(account, claimType, claimValue);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#addUrlCardRelation(java.lang.String, java.lang.String)
	 */
	public boolean addUrlCardRelation(String cuid, String url) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().addUrlCardRelation(account, cuid, url);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	public SelectorClientTO addUserProfile(UserProfileTO userProfileTO, SelectorClientTO selectorClientTO,
			UsernamePasswordAuthCredentialTO authCredentialTO) throws SOAPFaultException {
		try {
			if (authCredentialTO instanceof UsernamePasswordAuthCredentialTO) {
				userProfileTO.setLoginName(((UsernamePasswordAuthCredentialTO) authCredentialTO).getUsername());
				UserProfile userProfile = getRppsService().addUserProfileBase(userProfileTO,
						((UsernamePasswordAuthCredentialTO) authCredentialTO).getPassword());
				UserAccount userAccount = new UserAccount().setUserId(userProfile.getUserIdentifier());
				return getRppsService().addSelectorClient(userAccount, selectorClientTO);
			} else {
				throw new RuntimeException("Unsupported  AuthCredentialTO type");
			}
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	public SelectorClientTO addSelectorClient(SelectorClientTO selectorClientTO, AuthCredentialTO authCredentialTO)
			throws SOAPFaultException {
		try {
			return getRppsService()
					.addSelectorClient(getRppsService().authenticate(authCredentialTO), selectorClientTO);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#addUserProfileFromHBX(java.lang.String, java.lang.String)
	 */
	public void addUserProfileFromHBX(String userId, String password) throws SOAPFaultException {
		try {
			UserProfileTO userProfile = new UserProfileTO();
			userProfile.setLoginName(userId);
			getRppsService().addUserProfile(userProfile, password);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#checkImportStream(byte[])
	 */
	public FormatDescriptorTO checkImportStream(byte[] crd) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().checkImportStream(account, crd);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#clearCardCredential(java.lang.String)
	 */
	public ICardTO clearCardCredential(String cuid) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().clearCardCredential(account, cuid);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#clearCardHistory(java.lang.String)
	 */
	public boolean clearCardHistory(String cuid) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().clearCardHistory(account, cuid);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#createDuplicateICard(java.lang.String, java.lang.String,
	 * java.lang.String)
	 */
	public ICardTO createDuplicateICard(String cardName, String cuid, String pinCode) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().createDuplicateICard(account, cardName, cuid, pinCode);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.higgins.rpps.webservices.SCRPPSService#createICardByTemplate(org.eclipse.higgins.rpps.core.tobj.
	 * ICardTemplateTO)
	 */
	public ICardTO createICardByTemplate(ICardTemplateTO template) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().createICard(account, template);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#createICardFromHBX(java.lang.String)
	 */
	public String createICardFromHBX(String cardname) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().createICardFromHBX(account, cardname);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#deleteAllICard(java.lang.String, java.lang.String)
	 */
	public void deleteAllICard() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			getRppsService().deleteAllICard(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#deleteCardHistoryRecord(java.lang.String,
	 * java.lang.String)
	 */
	public boolean deleteCardHistoryRecord(String cuid, String url) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().deleteCardHistoryRecord(account, cuid, url);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#deleteCategory(java.lang.String)
	 */
	public void deleteCategory(String categoryId) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			getRppsService().deleteCategory(account, categoryId);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#deleteICard(java.lang.String)
	 */
	public ICardUpdateResponseTO deleteICard(String cuid) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().deleteICard(account, cuid);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#deleteUrlCardRelation(java.lang.String, java.lang.String)
	 */
	public boolean deleteUrlCardRelation(String cuid, String url) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().deleteUrlCardRelation(account, cuid, url);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#deleteUrlsCardsRelations(java.lang.String)
	 */
	public boolean deleteUrlsCardsRelations(String cuid) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().deleteUrlsCardsRelations(account, cuid);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#deleteUserProfile(java.lang.String, java.lang.String)
	 */
	public ResponseMessage deleteUserProfile() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().deleteUserProfile(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#editPinCode(java.lang.String, java.lang.String,
	 * java.lang.String)
	 */
	public ICardTO editPinCode(String cuid, String oldPinCode, String newPinCode) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().editPinCode(account, cuid, oldPinCode, newPinCode);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#exportICards(java.lang.String, java.lang.String[],
	 * org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO)
	 */
	public byte[] exportICards(String formatID, String[] cards, UITokenServiceCredentialTO credential)
			throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().exportICards(account, formatID, cards, credential);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#generatePasswordResetCode(java.lang.String)
	 */
	public void generatePasswordResetCode(final String userIdentifier) throws SOAPFaultException {
		try {
			new RppsService().generatePasswordResetCode(userIdentifier);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getCardHistory(java.lang.String, int, int,
	 * java.lang.String, java.lang.String)
	 */
	public CardUsageTO[] getCardHistory(String cuid, int startFrom, int capacity, String orderBy, String orderDirection)
			throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getCardHistory(account, cuid, startFrom, capacity, orderBy, orderDirection);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getCardInformation(java.lang.String, int)
	 */
	public CardInformationTO getCardInformation(String cuid, int size) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getCardInformation(account, cuid, size);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getCategory(java.lang.String, java.lang.String)
	 */
	public CategoryTO[] getCategory() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getCategory(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getClaimValuesMRU(java.lang.String, java.lang.String)
	 */
	public MapTO[] getClaimValuesMRU() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getClaimValuesMRU(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getDefaultPCard(java.lang.String, java.lang.String)
	 */
	public ICardTO getDefaultPCard() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getDefaultPCard(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	// TODO move it to utility
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getHBXUIList(java.lang.String, java.lang.String)
	 */
	public String[] getHBXUIList(String version, String windowsName) throws SOAPFaultException {
		try {
			HttpServlet servlet = (HttpServlet) MessageContext.getCurrentContext().getProperty(
					HTTPConstants.MC_HTTP_SERVLET);

			ServletContext servletContext = (ServletContext) servlet.getServletContext();

			String schemaFile = version + "_" + windowsName + ".txt";

			InputStream stream = servletContext.getResourceAsStream("/WEB-INF/resource/" + schemaFile);

			if (stream == null)
				throw new FileNotFoundException("Schema file " + schemaFile + " does not find in resource!");

			LineNumberReader lnr = new LineNumberReader(new InputStreamReader(stream));
			ArrayList schemalist = new ArrayList();
			try {
				String s = "";

				while ((s = lnr.readLine()) != null) {
					if (s.trim().length() > 3)
						schemalist.add(s.trim());
				}
			} catch (IOException e) {

				// log.error(e);
				throw new RppsException("Schema file " + schemaFile + " can not be read!");
			} finally {
				lnr.close();
			}
			String[] rr = new String[schemalist.size()];
			schemalist.toArray(rr);

			return rr;
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	// /**
	// * Returns request header value by name .
	// *
	// * @param headerName
	// * @return header value .
	// */
	// private String getHeader(String headerName) {
	// MessageContext msgCtx = MessageContext.getCurrentContext();
	// HttpServletRequest req = (HttpServletRequest) msgCtx.getProperty(HTTPConstants.MC_HTTP_SERVLETREQUEST);
	// return req.getHeader(headerName);
	// }

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getICardById(java.lang.String)
	 */
	public ICardTO getICardById(String cuid) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getICardByCUID(account, cuid);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getICardByIdAndTokenServiceCredential(java.lang.String,
	 * org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO)
	 */
	public ICardTO getICardByIdAndTokenServiceCredential(String cuid, UITokenServiceCredentialTO userCredintial)
			throws StsFaultException, SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getICardByCUIDAndTokenServiceCredential(account, cuid, userCredintial);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getICardClaims(java.lang.String, java.lang.String,
	 * java.lang.String, java.lang.String[], java.lang.String[])
	 */
	public UIResponseCardTO getICardClaims(String cuid, String policy, String typeofCredential, String[] credentialKey,
			String[] credentialValue) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getICardClaims(account, cuid, policy, typeofCredential, credentialKey,
					credentialValue);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getICardCreationTemplate(java.lang.String,
	 * java.lang.String)
	 */
	public ICardTemplateTO[] getICardCreationTemplate() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getICardCreationTemplate(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getICards(java.lang.String, java.lang.String)
	 */
	public ICardResponseTO getICards() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getICardsByUser(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getICardsAndCategoriesByUser(java.lang.String,
	 * java.lang.String)
	 */
	public ICardsAndCategoryTO getICardsAndCategoriesByUser() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getICardsAndCategoriesByUser(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getICardsByCUIDs(java.lang.String[])
	 */
	public ICardTO[] getICardsByCUIDs(String[] cuids) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getICardsByCUIDs(account, cuids);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getICardsByFormat(java.lang.String)
	 */
	public ICardTO[] getICardsByFormat(String formatID) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getICardsByFormat(account, formatID);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getOutputFormats(java.lang.String, java.lang.String)
	 */
	public FormatDescriptorTO[] getOutputFormats() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getOutputFormats(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/**
	 * It may be overloaded for adding custom features.
	 * 
	 * @param userId
	 * @param password
	 * @return
	 * @throws AccessException
	 */
	public IRppsService getRppsService() {
		if (null == rppsService) {
			rppsService = new RppsService();
		}
		return rppsService;
	}

	/**
	 * Breaks a string into array elements separated by delimiter characters ("|") .
	 * 
	 * @param str
	 *            a string to be parsed.
	 * @return the string array.
	 */
	private String[] getStringArray(String str) {
		StringTokenizer st = new StringTokenizer(str, "|");
		String[] strs = new String[st.countTokens()];
		for (int i = 0; i < strs.length && st.hasMoreTokens(); i++) {
			strs[i] = st.nextToken();
		}
		return strs;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getTokenByUrl(java.lang.String, java.lang.String,
	 * java.lang.String, java.lang.String)
	 */
	public TokenResponseTO getTokenByUrl(String policy, String policytype, String sslCert, String url)
			throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getTokenByUrl(account, policy, policytype, sslCert, url);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getTokenExtraByUrl(java.lang.String, java.lang.String,
	 * java.lang.String, java.lang.String)
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(String policy, String policytype, String sslCert, String url)
			throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getTokenExtraByUrl(account, policy, policytype, sslCert, url);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getTokenObject(java.lang.String, java.lang.String,
	 * java.lang.String, java.lang.String[], java.lang.String, java.lang.String[], java.lang.String[])
	 */
	public TokenResponseTO getTokenObject(String policy, String policytype, String sslCert, String[] cuids,
			String typeofCredential, String[] credentialKey, String[] credentialValue) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getTokenObject(account, policy, policytype, sslCert, cuids, typeofCredential,
					credentialKey, credentialValue);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getUI(java.lang.String, java.lang.String,
	 * java.lang.String, java.lang.String, java.lang.String)
	 */
	public UIResponseTO getUI(String url, String target, String sslCert, String policytype, String policy)
			throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getUserInterface(account, url, target, sslCert, policytype, policy);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#getUserProfile(java.lang.String, java.lang.String)
	 */
	public UserProfileTO getUserProfile() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getUserProfile(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#importICardAsByteArray(byte[])
	 */
	public ICardResponseTO importICardAsByteArray(byte[] crd) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().importICards(account, crd);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#importICards(byte[], java.lang.String,
	 * org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO)
	 */
	public ICardResponseTO importICards(byte[] crd, String formatID, UITokenServiceCredentialTO credential)
			throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().importICards(account, crd, formatID, credential);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#importICardsAsString(java.lang.String)
	 */
	public void importICardsAsString(String crd) throws SOAPFaultException {
		importICardAsByteArray(crd.getBytes());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.SCRPPSService#modifyCategories(org.eclipse.higgins.rpps.core.tobj.CategoryTO
	 * [])
	 */
	public ResponseMessage modifyCategories(CategoryTO[] categories) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().modifyCategories(account, categories);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.SCRPPSService#modifyCategory(org.eclipse.higgins.rpps.core.tobj.CategoryTO)
	 */
	public CategoryTO modifyCategory(CategoryTO category) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().modifyCategory(account, category);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#modifyPasswordWithOldPassword(java.lang.String,
	 * java.lang.String, java.lang.String)
	 */
	public void modifyPasswordWithOldPassword(final String userIdentifier, final String oldPassword,
			final String newPassword) throws SOAPFaultException {
		try {
			getRppsService().modifyPasswordWithOldPassword(userIdentifier, oldPassword, newPassword);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#modifyPasswordWithPasswordResetCode(java.lang.String,
	 * java.lang.String, java.lang.String)
	 */
	public void modifyPasswordWithPasswordResetCode(final String userIdentifier, final String passwordResetCode,
			final String newPassword) throws SOAPFaultException {
		try {
			new RppsService().modifyPasswordWithPasswordResetCode(userIdentifier, passwordResetCode, newPassword);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.SCRPPSService#modifyUserProfile(org.eclipse.higgins.rpps.core.tobj.UserProfileTO
	 * )
	 */
	public UserProfileTO modifyUserProfile(UserProfileTO userProfile) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().modifyUserProfile(account, userProfile);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#removeClaimValuesMRU(java.lang.String, java.lang.String)
	 */
	public void removeClaimValuesMRU(String claimType, String claimValue) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			getRppsService().removeClaimValuesMRU(account, claimType, claimValue);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#resetPinCode(java.lang.String, java.lang.String)
	 */
	public ICardTO resetPinCode(String cuid, String oldPinCode) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().resetPinCode(account, cuid, oldPinCode);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#setCardCredential(java.lang.String,
	 * org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO)
	 */
	public boolean setCardCredential(String cuid, UITokenServiceCredentialTO tokenCredential) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().setCardCredential(account, cuid, tokenCredential);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#setDefaultPCard(java.lang.String)
	 */
	public void setDefaultPCard(String pcardId) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			getRppsService().setDefaultPCard(account, pcardId);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#setPinCode(java.lang.String, java.lang.String)
	 */
	public ICardTO setPinCode(String cuid, String pinCode) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().setPinCode(account, cuid, pinCode);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#updateClaimsAcrossCards(java.lang.String[],
	 * java.lang.String[], org.eclipse.higgins.rpps.core.tobj.ICardUpdateTO[])
	 */
	public ICardUpdateResponseTO[] updateClaimsAcrossCards(String[] claimTypes, String[] claimValues,
			ICardUpdateTO[] cardUpdate) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().updateClaimsAcrossCards(account, claimTypes, claimValues, cardUpdate);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#updateICard(org.eclipse.higgins.rpps.core.tobj.ICardTO)
	 */
	public ICardTO updateICard(ICardTO icardTO) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().updateICard(account, icardTO);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.SCRPPSService#updateICards(org.eclipse.higgins.rpps.core.tobj.ICardTO[])
	 */
	public ICardUpdateResponseTO[] updateICards(ICardTO[] icardTOs) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().updateICards(account, icardTOs);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.SCRPPSService#userIdentifierExists(java.lang.String)
	 */
	public boolean userIdentifierExists(String userIdentifier) throws SOAPFaultException {
		try {
			return getRppsService().userIdentifierExists(userIdentifier);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.ISCRPPSServiceSE#signIn(org.eclipse.higgins.rpps.core.tobj.AuthCredentialTO)
	 */
	public AccessTokenTO signIn(SelectorAuthCredentialTO authCredentialTO) throws SOAPFaultException {
		try {
			return getRppsService().signIn(authCredentialTO);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.ISCRPPSServiceSE#signOut(java.lang.String)
	 */
	public void signOut(String accessTokenId) throws SOAPFaultException {
		try {
			getRppsService().signOut(accessTokenId);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.ISCRPPSServiceSE#getOTP(org.eclipse.higgins.rpps.core.tobj.AuthCredentialTO,
	 * java.lang.String, java.lang.String, java.lang.String)
	 */
	public void getOTP(UsernamePasswordAuthCredentialTO authCredentialTO, String captchaId, String captchaResponse,
			String channel) throws SOAPFaultException {
		try {
			getRppsService().getOTP(authCredentialTO, captchaId, captchaResponse, channel);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	// /*
	// * (non-Javadoc)
	// *
	// * @see org.eclipse.higgins.rpps.webservices.ISCRPPSServiceSE#getSelectorAuthCredentialTO()
	// */
	// public SelectorAuthCredentialTO getSelectorAuthCredentialTO() throws SOAPFaultException {
	// return null;
	// }
	//
	// /*
	// * (non-Javadoc)
	// *
	// * @see org.eclipse.higgins.rpps.webservices.ISCRPPSServiceSE#getUsernamePasswordAuthCredentialTO()
	// */
	// public UsernamePasswordAuthCredentialTO getUsernamePasswordAuthCredentialTO() throws SOAPFaultException {
	// return null;
	// }
	//
	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.higgins.rpps.webservices.ISCRPPSServiceSE#addSelectorClient(org.eclipse.higgins.rpps.core.tobj.
	 * AuthCredentialTO, org.eclipse.higgins.rpps.core.tobj.SelectorClientTO, java.lang.String)
	 */
	public SelectorClientTO addSelectorClient(UsernamePasswordAuthCredentialTO authCredentialTO,
			SelectorClientTO selectorClientTO, String otp) throws SOAPFaultException {
		try {
			return getRppsService().addSelectorClient(getRppsService().authenticate(authCredentialTO),
					selectorClientTO, otp);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.ISCRPPSServiceSE#updateSelectorClient(org.eclipse.higgins.rpps.core.tobj
	 * .SelectorClientTO)
	 */
	public SelectorClientTO updateSelectorClient(SelectorClientTO selectorClientTO) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().updateSelectorClient(account, selectorClientTO);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.ISCRPPSServiceSE#deleteSelectorClient(org.eclipse.higgins.rpps.core.tobj
	 * .SelectorClientTO)
	 */
	public void deleteSelectorClient(SelectorClientTO selectorClientTO) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			getRppsService().deleteSelectorClient(account, selectorClientTO);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.ISCRPPSServiceSE#getSelectorClients()
	 */
	public SelectorClientTO[] getSelectorClients() throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getSelectorClients(account);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	public ICardResponseTO importSelectedICards(final byte[] crd, final String formatID,
			final UITokenServiceCredentialTO credential, Boolean[] importFlag) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().importICards(account, crd, formatID, credential, importFlag);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

	public ImportCardInformationTO[] getImportingCardsInformation(final byte[] crd, final String formatID,
			final UITokenServiceCredentialTO credential) throws SOAPFaultException {
		try {
			UserAccount account = getRppsService().authenticate(
					getSoapUtils().getAuthTokenId(getSoapUtils().getHeader(AUTHORIZATION_HEADER)));
			return getRppsService().getImportingCardsInformation(account, crd, formatID, credential);
		} catch (Throwable e) {
			LOG.error(e, e);
			throw getSoapUtils().buildSoapFault(e);
		}
	}

}
