package org.eclipse.higgins.rpps.core;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.higgins.rpps.core.tobj.FormTO;
import org.eclipse.higgins.rpps.core.tobj.UrlInformationAndBlackListTO;
import org.eclipse.higgins.rpps.core.tobj.UrlInformationTO;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.ac.ws.model.bean.AuditService;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IPwmCard;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.common.utils.PwmCardClaimsHelper;
import org.eclipse.higgins.icard.provider.pwm.IdASBasedPwmCardProvider;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.rpps.core.RppsException;
import org.eclipse.higgins.rpps.core.impl.RppsHelper;
import org.eclipse.higgins.rpps.core.impl.RppsService;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.utils.ResourceMetaDataHelper;
import org.eclipse.higgins.rpps.web.bean.CaptchaServiceSingleton;
import org.eclipse.higgins.sync.auth.exceptions.AuthenticationException;
import org.eclipse.higgins.sync.meta.IResourceMetaDataService;
import org.eclipse.higgins.sync.meta.entity.RevisionEntity;
import org.eclipse.higgins.sync.meta.exceptions.ResourceMetaDataException;
import org.eclipse.higgins.sync.to.RevisionTO;
import org.eclipse.higgins.sync.utilities.ConverterHelper;
import org.eclipse.higgins.user.login.UserAccount;
import org.eclipse.higgins.user.login.exception.ConfigurationException;
import org.eclipse.higgins.user.login.exception.LoginException;
import org.eclipse.higgins.user.profile.IUserProfileService;
import org.eclipse.higgins.user.profile.entity.Captcha;
import org.eclipse.higgins.user.profile.entity.UserProfile;
import org.eclipse.higgins.user.profile.exception.UserProfileException;

public class RPPSServiceCore extends RppsService {

	private static Log LOG = LogFactory.getLog(RPPSServiceCore.class);

	/**
	 * Creates an RPPSServiceCoreImpl object.
	 * 
	 * @return the RPPSServiceCoreImpl object
	 */
	public RPPSServiceCore() {
	}

	/**
	 * Authenticate user by username password and update last used date.
	 * 
	 * @param userId
	 * @param password
	 * @return
	 * @throws ConfigurationException
	 * @throws LoginException
	 */
	@Override
	public UserAccount authenticate(final String userId, final String password) throws ConfigurationException,
			LoginException {
		UserAccount res = super.authenticate(userId, password);
		new AuditService().saveUserActivity(res.getUserId());
		return res;
	}

	/**
	 * Deletes user profile. Also deletes user activity.
	 * 
	 * @see Define/Add audit handler for user authentification http://graceland.parityinc.net:8090/browse/WWM-1374
	 * 
	 * @return ResponseMessage
	 * @throws RppsException
	 */
	@Override
	public ResponseMessage deleteUserProfile(UserAccount userAccount) throws RppsException {
		ResponseMessage responseMessage = super.deleteUserProfile(userAccount);
		if (ResponseMessage.noError == responseMessage.getCode()) {
			new AuditService().deleteUserActivity(userAccount.getUserId());
		}
		return responseMessage;
	}

	// part of PWM
	public UrlInformationAndBlackListTO getUrlInformationAndBlackList(UserAccount userAccount) throws RppsException {
		UrlInformationTO[] urlInformation = new UrlInformationTO[0];
		Hashtable urlTable = new Hashtable();
		try {
			for (Iterator iter = ICardRegistry.getInstance().getICardProviders(); iter.hasNext();) {
				Object provider = iter.next();
				if (provider instanceof IdASBasedPwmCardProvider) {
					IdASBasedPwmCardProvider new_provider = (IdASBasedPwmCardProvider) provider;
					for (Iterator cardIter = new_provider.getICards(RppsHelper.getCallbackHandler(userAccount
							.getUserId())); cardIter.hasNext();) {
						IPwmCard card = (IPwmCard) cardIter.next();
						IClaim urlClaim = card.getClaim(PwmCardClaimsHelper.CLAIM_TYPE_SITE);
						String url = null;
						if (urlClaim != null && urlClaim.getType().isSimple()) {
							url = (String) ((ISimpleClaim) urlClaim).getValues().get(0);
						}
						if (url != null) {
							List formList = (List) urlTable.get(url);
							if (formList == null)
								formList = new ArrayList();

							IClaim formIdClaim = card.getClaim(PwmCardClaimsHelper.CLAIM_TYPE_FORM_ID);
							String formId = "";
							if (formIdClaim != null && formIdClaim.getType().isSimple()) {
								formId = (String) ((ISimpleClaim) formIdClaim).getValues().get(0);
							}

							IClaim formActionClaim = card.getClaim(PwmCardClaimsHelper.CLAIM_TYPE_FORM_ACTION);
							String formAction = "";
							if (formActionClaim != null && formActionClaim.getType().isSimple()) {
								formAction = (String) ((ISimpleClaim) formActionClaim).getValues().get(0);
							}

							IClaim formNameClaim = card.getClaim(PwmCardClaimsHelper.CLAIM_TYPE_FORM_NAME);
							String formName = "";
							if (formNameClaim != null && formNameClaim.getType().isSimple()) {
								formName = (String) ((ISimpleClaim) formNameClaim).getValues().get(0);
							}
							formList.add(new FormTO(formId, formName, formAction));
							urlTable.put(url, formList);
						}
					}
				}
			}
			urlInformation = new UrlInformationTO[urlTable.size()];
			int i = 0;
			for (Enumeration urlIter = urlTable.keys(); urlIter.hasMoreElements();) {
				String url = (String) urlIter.nextElement();
				List formTOList = (List) urlTable.get(url);
				urlInformation[i++] = new UrlInformationTO(url, (FormTO[]) formTOList.toArray(new FormTO[formTOList
						.size()]));
			}
			Set blackList = ((IUserProfileService) userProfileService.get()).getUserProfile(userAccount.getUserId())
					.getBlackListedSites();
			if (blackList == null)
				blackList = new HashSet();
			return new UrlInformationAndBlackListTO(urlInformation, (String[]) blackList.toArray(new String[blackList
					.size()]));
		} catch (Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	// part of PWM
	public ResponseMessage addUrlToBlackList(UserAccount userAccount, String url) throws RppsException {
		try {
			UserProfile userProfile = ((IUserProfileService) userProfileService.get()).getUserProfile(userAccount
					.getUserId());
			Set blackList = userProfile.getBlackListedSites();
			if (blackList == null)
				blackList = new HashSet();
			if (blackList.contains(url))
				return new ResponseMessage(ResponseMessage.noError, "Url added");
			blackList.add(url);

			userProfile.setBlackListedSites(blackList);
			((IUserProfileService) userProfileService.get()).modifyUserProfile(userAccount.getUserId(), userProfile);
		} catch (Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
		return new ResponseMessage(ResponseMessage.noError, "Url added");
	}

	// part of PWM
	public ResponseMessage removeUrlFromBlackList(UserAccount userAccount, String url) throws Exception {
		try {
			UserProfile userProfile = ((IUserProfileService) userProfileService.get()).getUserProfile(userAccount
					.getUserId());
			Set blackList = userProfile.getBlackListedSites();
			if (blackList == null) {
				return new ResponseMessage(ResponseMessage.noError, "Url removed");
			}

			blackList.remove(url);
			userProfile.setBlackListedSites(blackList);
			((IUserProfileService) userProfileService.get()).modifyUserProfile(userAccount.getUserId(), userProfile);

		} catch (Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
		return new ResponseMessage(ResponseMessage.noError, "Url removed");
	}

	/**
	 * Return root revision.
	 * 
	 * @see WWM-1657: AIR ICM - cache timestamp check http://graceland.parityinc.net:8090/browse/WWM-1657
	 * 
	 * @return
	 * @throws ResourceMetaDataException
	 */
	public RevisionTO getRootRevision(UserAccount userAccount) throws ResourceMetaDataException {
		// find root revision
		RevisionEntity revisionEntity = new RevisionEntity();
		revisionEntity.setUserId((String) userAccount.getUserId()).setResourceType(
				IResourceMetaDataService.ROOT_RESOURCE_TYPE).setResourceId((String) userAccount.getUserId());
		RevisionEntity rootRevisionEntity = ((IResourceMetaDataService) metaDataService.get())
				.findResourceRevisionEntity(revisionEntity);

		if (null == rootRevisionEntity) {
			// create root revision for old account
			new ResourceMetaDataHelper().updateUserProfileMD((IResourceMetaDataService) metaDataService.get(),
					(String) userAccount.getUserId(), new UserProfile().setUserIdentifier(userAccount.getUserId()));

			rootRevisionEntity = ((IResourceMetaDataService) metaDataService.get())
					.findResourceRevisionEntity(revisionEntity);
		}

		return ConverterHelper.revisionEntity2RevisionTO(rootRevisionEntity);
	}

	public void validateCaptcha(final String captchaId, final String captchaResponse) throws UserProfileException,
			AuthenticationException {
		final Captcha captcha = ((IUserProfileService) userProfileService.get()).getCaptcha(captchaId);

		if (null != captcha) {
			// captcha must be validated just one time
			((IUserProfileService) userProfileService.get()).deleteCaptcha(captchaId);
		} else {
			throw new AuthenticationException("Invalide capthcha.");
		}

		// validate captcha
		if (!CaptchaServiceSingleton.getInstance().validateResponseForID(captcha.getKey(), captchaResponse)
				.booleanValue()) {
			LOG.warn("Invalid Captcha security code");
			throw new RppsException("Sorry, you have provided an invalid security code. You might be a bot. "
					+ "This program is only open to real people.");
		}
	}

}
