/*
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 *     Artem Verkhovets - initial API and implementation
 */

package org.eclipse.higgins.rpps.webservices;

import java.rmi.AccessException;
import java.rmi.Remote;
import java.rmi.RemoteException;

import javax.xml.rpc.soap.SOAPFaultException;

import org.eclipse.higgins.rpps.core.RppsException;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.FormatDescriptorTO;
import org.eclipse.higgins.rpps.core.tobj.ICardResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ImportCardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.MapTO;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.tobj.StsFaultException;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;

/**
 * This is the service endpoint interface for the RPPSServiceweb service. Created 18.01.2007 10:34:54
 * 
 * @author Sergei Yakovlev
 * @author Artem Verkhovets
 * 
 * @deprecated Replaced by SCRPPSServiceImpl
 */

public interface RPPSServiceSEI extends Remote {

	/**
	 * Web service operation. Creates a new card's category
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param category
	 *            the category transfer object
	 * @return Returns saved card category object
	 * @throws SOAPFaultException
	 */
	public CategoryTO addCategory(String userId, String password, CategoryTO category) throws SOAPFaultException;

	/**
	 * Web service operation. Adds most recently used record
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param claimType
	 *            claim's type
	 * @param claimValue
	 *            claim's value
	 * @throws SOAPFaultException
	 */
	public void addClaimValuesMRU(String userId, String password, String claimType, String claimValue)
			throws SOAPFaultException;

	/**
	 * Web service operation. Create relation between card and url
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            the web page url
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws SOAPFaultException
	 */
	public boolean addUrlCardRelation(String userId, String password, String cuid, String url)
			throws SOAPFaultException;

	/**
	 * Web service operation. Creates a new user profile
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param userProfile
	 *            the userProfile profile information as UserProfileTO
	 * @return Returns <code>ResponseMessage</code> with <code>code</code> field equals <code>0</code> if profile was
	 *         added. Otherwise returns <code>ResponseMessage</code> with error code or throws an SOAPFaultException
	 * @throws SOAPFaultException
	 */
	public ResponseMessage addUserProfile(String userId, String password, UserProfileTO userProfile)
			throws SOAPFaultException;

	/**
	 * Web service operation. Add user profile
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @throws SOAPFaultException
	 * 
	 */
	public void addUserProfileFromHBX(String userId, String password) throws SOAPFaultException;

	/**
	 * Checks whether <code>RppsService</code> can recognize data format of the provided input stream. Some formats
	 * require authentication information in order to process input data. This method should be used prior to actual
	 * attempt to import data from the input stream in order to retrieve an information about data format and required
	 * authentication data to process the data successfully.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param crd
	 *            context of file as byte array
	 * @return the information about input stream data format if it was recognized successfully and <code>null</code>
	 *         otherwise.
	 * @throws RppsException
	 */
	public FormatDescriptorTO checkImportStream(String userId, String password, byte[] crd) throws SOAPFaultException;

	/**
	 * Web service operation. Deletes saved authentication card information of the specified card
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns updated card object
	 * @throws SOAPFaultException
	 */
	public ICardTO clearCardCredential(String userId, String password, String cuid) throws SOAPFaultException;

	/**
	 * Web service operation. Deletes card history of the specified card on all URIs unless it is checked as
	 * "Use Always".
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if card history was cleared. Otherwise <code>false</code>
	 * @throws SOAPFaultException
	 */
	public boolean clearCardHistory(String userId, String password, String cuid) throws SOAPFaultException;

	/**
	 * Web service operation. This method create duplicate of personal card only
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cardName
	 *            name of new card
	 * @param cuid
	 *            the old icard's cuid
	 * @param pinCode
	 *            pinCode for retrieve claims of personal card if that is pin protected. Otherwise <code>null</code>
	 * @return Returns updated card object
	 * @throws SOAPFaultException
	 */
	public ICardTO createDuplicateICard(String userId, String password, String cardName, String cuid, String pinCode)
			throws SOAPFaultException;

	/**
	 * Web service operation. Creates a new card by template
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param template
	 *            the icard's template
	 * @return a new card as a ICardTO
	 * @throws RemoteException
	 */
	public ICardTO createICardByTemplate(String userId, String password, ICardTemplateTO template)
			throws SOAPFaultException;

	/**
	 * Web service operation. This method creating a new card for HBX
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cardname
	 *            the card's name
	 * @return a new card's cuid
	 * @throws SOAPFaultException
	 * @deprecated
	 */
	public String createICardFromHBX(String userId, String password, String cardname) throws SOAPFaultException;

	/**
	 * Web service operation. Deletes all user's card.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @throws SOAPFaultException
	 */
	public void deleteAllICard(String userId, String password) throws SOAPFaultException;

	/**
	 * Web service operation. Delete specified record of card history.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            url of site which will be deleted from card history
	 * @return Returns <code>true</code> if card history record was deleted. Otherwise <code>false</code>
	 * @throws SOAPFaultException
	 */
	public boolean deleteCardHistoryRecord(String userId, String password, String cuid, String url)
			throws SOAPFaultException;

	/**
	 * Web service operation. Delete the card's category
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param categoryId
	 *            category identifier
	 * @throws SOAPFaultException
	 */
	public void deleteCategory(String userId, String password, String categoryId) throws SOAPFaultException;

	/**
	 * Web service operation. Deletes specified card.
	 * 
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns ICardUpdateResponseTO which contains delete status for card. If <code>code</code> field equals
	 *         <code>0</code> then card was updated correctly. Otherwise <code>code</code> field contains error code. If
	 *         deleted card had self signed type then <code>cuid</code> field contains cuid of personal card which had
	 *         been used to signed managed card.
	 * @throws SOAPFaultException
	 */
	public ICardUpdateResponseTO deleteICard(String userId, String password, String cuid) throws SOAPFaultException;

	/**
	 * Web service operation. Delete relation between card and site's url
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            the url of site
	 * @return Returns <code>true</code> if relation was deleted. Otherwise <code>false</code>
	 * @throws SOAPFaultException
	 */
	public boolean deleteUrlCardRelation(String userId, String password, String cuid, String url)
			throws SOAPFaultException;

	/**
	 * Web service operation. Delete all relations between card and site's urls
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if relations were deleted. Otherwise <code>false</code>
	 * @throws SOAPFaultException
	 */
	public boolean deleteUrlsCardsRelations(String userId, String password, String cuid) throws SOAPFaultException;

	/**
	 * Web service operation. Delete user profile with all user information
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return Returns <code>ResponseMessage</code> with <code>code</code> field equals <code>0</code> if profile was
	 *         deleted. Otherwise returns <code>ResponseMessage</code> with error code or throws an SOAPFaultException
	 * @throws Exception
	 */
	public ResponseMessage deleteUserProfile(String userId, String password) throws SOAPFaultException;

	/**
	 * Web service operation. Changes pin code for personal card
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param oldPinCode
	 *            Old pin code
	 * @param newPinCode
	 *            New pin code
	 * @return Returns updated card object
	 * @throws SOAPFaultException
	 */
	public ICardTO editPinCode(String userId, String password, String cuid, String oldPinCode, String newPinCode)
			throws SOAPFaultException;

	/**
	 * Web service operation. Returns file as byte array with special which contains select cards.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param formatID
	 *            file's format identifier
	 * @param cards
	 *            the icard's CUIDs
	 * @param credential
	 *            authentication information
	 * @return array of byte
	 * @throws SOAPFaultException
	 */
	public byte[] exportICards(String userId, String password, String formatID, String[] cards,
			UITokenServiceCredentialTO credential) throws SOAPFaultException;

	/**
	 * Web service operation. Generate and send user email with new password reset code.
	 * 
	 * This method is part of reset user password work-flow.
	 * 
	 * @see WWM-1264 RPPS Changes for password reset http://graceland.parityinc.net:8090/browse/WWM-1264
	 * 
	 * @param userIdentifier
	 * @throws SOAPFaultException
	 *             if couldn't send email or if userIdentifier is invalid.
	 */
	void generatePasswordResetCode(String userIdentifier) throws SOAPFaultException;

	/**
	 * Web service operation. Return list of card history records, beginning at the specified position in selection.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param startFrom
	 *            starting position in the selection
	 * @param capacity
	 *            count of record which will be returned
	 * @param orderBy
	 *            name of field which be used for sorting. Can take one of these values: "<code>date</code>","
	 *            <code>site</code>","<code>useAlways</code>"
	 * @param orderDirection
	 *            If value of this parameter is "<code>desc</code>" then result will be sorted in descending order.
	 *            Otherwise in ascending order.
	 * @return array of CardUsageTO
	 * @throws SOAPFaultException
	 */
	public CardUsageTO[] getCardHistory(String userId, String password, String cuid, int startFrom, int capacity,
			String orderBy, String orderDirection) throws SOAPFaultException;

	/**
	 * Web service operation. Gets card information object
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param size
	 *            count of card history element which will be returned
	 * @return Return card information object
	 * @throws SOAPFaultException
	 */
	public CardInformationTO getCardInformation(String userId, String password, String cuid, int size)
			throws SOAPFaultException;

	/**
	 * Web service operation. Gets list of card categories
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return Returns array of category transfer object
	 * @throws SOAPFaultException
	 */
	public CategoryTO[] getCategory(String userId, String password) throws SOAPFaultException;

	/**
	 * Web service operation. Gets most recently used records
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return Returns array of MapTO
	 * @throws SOAPFaultException
	 */
	public MapTO[] getClaimValuesMRU(String userId, String password) throws SOAPFaultException;

	/**
	 * Web service operation. Return default personal card.
	 * 
	 * This method is part of workflow ' Instant credential m-Cards' http://graceland.parityinc.net:8090/browse/WWM-1150
	 * 
	 * @return default pcard or null if user doesn't have default pcard
	 * @throws SOAPFaultException
	 */
	ICardTO getDefaultPCard(String userId, String password) throws SOAPFaultException;

	/**
	 * Web service operation.
	 * 
	 * @param version
	 * @param windowsName
	 * @return array of HBX messages
	 * @throws SOAPFaultException
	 * @deprecated
	 */
	public String[] getHBXUIList(String version, String windowsName) throws SOAPFaultException;

	// /**
	// * Web service operation. Gets the global i-name registered to the Higgins service (e.g. "=parity").
	// *
	// * @return the global i-name.
	// * @throws SOAPFaultException
	// * @deprecated
	// */
	// public String getHigginsServiceGlobalIName() throws SOAPFaultException;

	/**
	 * Web service operation. This method gets the card list of the specified user.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the card's cuid.
	 * @return the card list
	 * @throws SOAPFaultException
	 */
	public ICardTO getICardById(String userId, String password, String cuid) throws SOAPFaultException;

	/**
	 * Web service operation. This method gets the card with retrieved claims.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the card's cuid
	 * @param userCredential
	 *            filled card credential for retrieve claims
	 * @return the card as a ICardTO
	 * @throws SOAPFaultException
	 */
	public ICardTO getICardByIdAndTokenServiceCredential(String userId, String password, String cuid,
			UITokenServiceCredentialTO userCredential) throws StsFaultException, SOAPFaultException;

	/**
	 * Web service operation. Created for some demo.
	 * 
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the card's cuid.
	 * @param policy
	 * @param typeofCredential
	 * @param credentialKey
	 * @param credentialValue
	 * @return UIResponseCardTO
	 * @throws SOAPFaultException
	 * @deprecated
	 */
	public UIResponseCardTO getICardClaims(String userId, String password, String cuid, String policy,
			String typeofCredential, String[] credentialKey, String[] credentialValue) throws SOAPFaultException;

	/**
	 * Web service operation. This method return list of card creation template as array of ICardTemplateTO
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return a template list as array of ICardTemplateTO
	 */
	public ICardTemplateTO[] getICardCreationTemplate(String userId, String password) throws SOAPFaultException;

	/**
	 * Web service operation. This method gets ICardResponseTO which contains the card list and card relation list of
	 * the specified user.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return ICardResponseTO
	 * @throws SOAPFaultException
	 */
	public ICardResponseTO getICards(String userId, String password) throws SOAPFaultException;

	/**
	 * Web service operation. This method gets the card list and category list of the specified user.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return ICardsAndCategoryTO which contain the cards and the categorys lists
	 * @throws SOAPFaultException
	 */
	public ICardsAndCategoryTO getICardsAndCategoriesByUser(String userId, String password) throws SOAPFaultException;

	/**
	 * Web service operation. This method gets cards as array of ICardTO by CUIDs array.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuids
	 *            the card CUIDs.
	 * @return array of ICardTO.
	 * @throws SOAPFaultException
	 */
	public ICardTO[] getICardsByCUIDs(String userId, String password, String[] cuids) throws SOAPFaultException;

	/**
	 * Web service operation. Gets cards which may be export to special file format
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param formatID
	 * @return list of cards as a array of CardTO
	 * @throws SOAPFaultException
	 */
	public ICardTO[] getICardsByFormat(String userId, String password, String formatID) throws SOAPFaultException;

	/**
	 * Web service operation. Gets available file format for cards export
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return array of FormatDescriptorTO
	 * @throws SOAPFaultException
	 * @see FormatDescriptorTO
	 */
	public FormatDescriptorTO[] getOutputFormats(String userId, String password) throws SOAPFaultException;

	// /**
	// * Web service operation. Makes xmlToken usage claim list.
	// *
	// * @param sslCert
	// * the SSL certificate
	// * @param claimNameArray
	// * the array which contains claim names
	// * @param claimValueArray
	// * the array which contains claim values
	// * @return the security token as a String
	// * @throws SOAPFaultException
	// */
	// public TokenResponseTO getTokenByClaims(String sslCert, String claimNameArray, String claimValueArray)
	// throws SOAPFaultException;
	//
	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param url
	 *            the url of site which needs token
	 * @return the security token as a String
	 * @throws SOAPFaultException
	 */
	public TokenResponseTO getTokenByUrl(String userId, String password, String policy, String policytype,
			String sslCert, String url) throws SOAPFaultException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param url
	 *            the url of site which needs token
	 * @return the security token as a String with extra card's information
	 * @throws SOAPFaultException
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(String userId, String password, String policy, String policytype,
			String sslCert, String url) throws SOAPFaultException;

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the
	 * requester, and the set of one or more selections that the user has just made in the ISS Web UI (see
	 * userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security
	 * token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential
	 *            the type of selected credential
	 * @param credentialKey
	 *            the names of fields of selected credential
	 * @param credentialValue
	 *            the value of fields of selected credential
	 * @return the security token as a String
	 * @throws SOAPFaultException
	 */
	public TokenResponseTO getTokenObject(String userId, String password, String policy, String policytype,
			String sslCert, String[] cuids, String typeofCredential, String[] credentialKey, String[] credentialValue)
			throws SOAPFaultException;

	/**
	 * Web service operation. This method takes as input a policy---for example, one that has been retrieved from the
	 * Relying Party Agent (called here the 'Requester')---as well as a description of the Requester. The
	 * UserInterfaceResponse object should contain all information necessary to display both (a) a user interface which
	 * shares the same card metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b)
	 * other interface paradigms to be developed in the future
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param url
	 *            the URL to the page with <object> tag
	 * @param target
	 * @param sslCert
	 *            the SSL certificate
	 * @param policytype
	 *            the RP Security Policy type
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws SOAPFaultException
	 */
	public UIResponseTO getUI(String userId, String password, String url, String target, String sslCert,
			String policytype, String policy) throws SOAPFaultException;

	/**
	 * Web service operation. Gets user profile as a UserProfileTO
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return user profile information
	 * @throws SOAPFaultException
	 */
	public UserProfileTO getUserProfile(String userId, String password) throws AccessException, SOAPFaultException;

	/**
	 * Web service operation. Imports card from a file.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param crd
	 *            the card as a byte array.
	 * @return list of cards
	 * @throws SOAPFaultException
	 */
	public ICardResponseTO importICardAsByteArray(String userId, String password, byte[] crd) throws SOAPFaultException;

	/**
	 * Web service operation. Import cards from special format file.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param crd
	 *            file's context
	 * @param formatID
	 *            file's format identifier
	 * @param credential
	 *            authentication information
	 * @return ICardResponseTO
	 * @throws SOAPFaultException
	 */
	public ICardResponseTO importICards(String userId, String password, byte[] crd, String formatID,
			UITokenServiceCredentialTO credential) throws SOAPFaultException;

	/**
	 * Web service operation. Imports card from a file.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param crd
	 *            the card as a string.
	 * @throws SOAPFaultException
	 */
	public void importICardsAsString(String userId, String password, String crd) throws SOAPFaultException;

	/**
	 * Web service operation. Modifies list of card's category
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param categories
	 *            list of card's category
	 * @return Returns <code>ResponseMessage</code> with <code>code</code> field equals <code>0</code> if category was
	 *         updated. Otherwise returns <code>ResponseMessage</code> with error code or throws an SOAPFaultException
	 * @throws SOAPFaultException
	 */
	public ResponseMessage modifyCategories(String userId, String password, CategoryTO[] categories)
			throws SOAPFaultException;

	/**
	 * Web service operation. Modifies the card's category
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param category
	 *            category transfer object
	 * @return Returns modified category
	 * @throws SOAPFaultException
	 */
	public CategoryTO modifyCategory(String userId, String password, CategoryTO category) throws SOAPFaultException;

	/**
	 * Update user password, requires oldPassword for identify.
	 * 
	 * This method is part of reset user password work-flow.
	 * 
	 * @see WWM-1264 RPPS Changes for password reset http://graceland.parityinc.net:8090/browse/WWM-1264
	 * 
	 * @param userIdentifier
	 * @param oldPassword
	 * @param newPassword
	 * @throws RppsException
	 */
	void modifyPasswordWithOldPassword(final String userIdentifier, final String oldPassword, final String newPassword)
			throws SOAPFaultException;

	/**
	 * Update user password, requires password reset code for identify.
	 * 
	 * This method is part of reset user password work-flow.
	 * 
	 * @see WWM-1264 RPPS Changes for password reset http://graceland.parityinc.net:8090/browse/WWM-1264
	 * 
	 * @param userIdentifier
	 * @param passwordResetCode
	 * @throws RppsException
	 */

	void modifyPasswordWithPasswordResetCode(final String userIdentifier, final String passwordResetCode,
			final String newPassword) throws SOAPFaultException;

	/**
	 * Web service operation. This method update user profile information.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param userProfile
	 *            user profile information
	 * @return Return updated user profile
	 * @throws SOAPFaultException
	 */
	public UserProfileTO modifyUserProfile(String userId, String password, UserProfileTO userProfile)
			throws SOAPFaultException;

	/**
	 * Web service operation. Deletes most recently used record
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param claimType
	 *            claim's type
	 * @param claimValue
	 *            claim's value
	 * @throws SOAPFaultException
	 */
	public void removeClaimValuesMRU(String userId, String password, String claimType, String claimValue)
			throws SOAPFaultException;

	/**
	 * Web service operation. Remove pin code protection from personal card
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param oldPinCode
	 *            pin code
	 * @return Returns updated card object
	 * @throws SOAPFaultException
	 */
	public ICardTO resetPinCode(String userId, String password, String cuid, String oldPinCode)
			throws SOAPFaultException;

	/**
	 * Web service operation. Saves authentication card information of the specified card
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param tokenCredential
	 *            authentication card information
	 * @return Returns <code>true</code> if card credential was set. Otherwise <code>false</code>
	 * @throws SOAPFaultException
	 */
	public boolean setCardCredential(String userId, String password, String cuid,
			UITokenServiceCredentialTO tokenCredential) throws SOAPFaultException;

	/**
	 * Web service operation. Set default personal card.
	 * 
	 * This method is part of workflow ' Instant credential m-Cards' http://graceland.parityinc.net:8090/browse/WWM-1150
	 * if pcardId is null or empty string it just removes old default pcard.
	 * 
	 * @param pcardId
	 *            personal card id (cuid)
	 * @throws SOAPFaultException
	 *             if coudn't find pcardId throw SOAPFaultException
	 */
	void setDefaultPCard(String userId, String password, String pcardId) throws SOAPFaultException;

	/**
	 * Web service operation. Sets pin code protection on personal card
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param pinCode
	 *            pin code
	 * @return Returns updated card object
	 * @throws SOAPFaultException
	 */
	public ICardTO setPinCode(String userId, String password, String cuid, String pinCode) throws SOAPFaultException;

	// /**
	// * Web service operation. The test method that returns inputed string array.
	// *
	// * @param strArray
	// * the string array to be returned.
	// * @return the string array inputed as a parameter.
	// * @throws SOAPFaultException
	// * @deprecated
	// */
	// public String[] testStringArray(String[] strArray) throws SOAPFaultException;

	/**
	 * Web service operation. This method updates some claim values in list of cards
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param claimTypes
	 *            array of claim's types
	 * @param claimValues
	 *            array of claim's types. Must be in the same order as claimTypes
	 * @param cardUpdate
	 *            array of ICardUpdateTO
	 * @return Return array of ICardUpdateResponseTO which contains update status for every card. If <code>code</code>
	 *         field equals <code>0</code> then card was updated correctly. Otherwise <code>code</code> field contains
	 *         error code
	 * @throws SOAPFaultException
	 */
	public ICardUpdateResponseTO[] updateClaimsAcrossCards(String userId, String password, String[] claimTypes,
			String[] claimValues, ICardUpdateTO[] cardUpdate) throws SOAPFaultException;

	/**
	 * Web service operation. This method update card
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @throws SOAPFaultException
	 */
	public ICardTO updateICard(String userId, String password, ICardTO icardTO) throws SOAPFaultException;

	/**
	 * Web service operation. This method update list of cards
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @throws SOAPFaultException
	 */
	public ICardUpdateResponseTO[] updateICards(String userId, String password, ICardTO[] icardTOs)
			throws SOAPFaultException;

	/**
	 * Web service operation. Checks availability of user profile by userIdentifier
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @return Returns <code>true</code> if user identifier exists. Otherwise <code>false</code>
	 * @throws SOAPFaultException
	 */
	public boolean userIdentifierExists(String userId) throws SOAPFaultException;

	ICardResponseTO importSelectedICards(String userId, String password, final byte[] crd, final String formatID,
			final UITokenServiceCredentialTO credential, Boolean[] importFlag) throws SOAPFaultException;

	ImportCardInformationTO[] getImportingCardsInformation(String userId, String password, final byte[] crd,
			final String formatID, final UITokenServiceCredentialTO credential) throws SOAPFaultException;

}
