/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 *     Artem Verkhovets - initial API and implementation
 */

package org.eclipse.higgins.rpps.core;

import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.List;

import javax.security.auth.callback.CallbackHandler;

import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;

/**
 * The object used for executing Higgins API and returning the results it produces.
 * 
 * @author Sergei Yakovlev
 * @author Artem Verkhovets
 * 
 */
public interface IRppsService {

	public final static String ITSUsernamePasswordCredential = "ITSUsernamePasswordCredential";

	public final static String ITSSelfIssuedCredential = "ITSSelfIssuedCredential";

	public final static String ITSKerberosV5Credential = "ITSKerberosV5Credential";

	public final static String ITSX509V3Credential = "ITSX509V3Credential";

	public final static String IPinCodeCredential = "IPinCodeCredential";

	/**
	 * Acquires the i-card from RP site.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param subscribeURL the URL of the site from where i-card will be acquired
	 * @return a new card id
	 */
	public String acquireICard(String userId, String password, String subscribeURL) throws MalformedURLException;

	/**
	 * Locate the URIICard-type I-Card whose URI matches 'cref'. If no such card exists, create it. Find the Node whose unique identifier is 'cuid'. On the uniqueIdentifier attribute value itself: (1) if 'success' is true, add/update to "now" the the lastVerifiedFromSource metadata property's timestamp and (2) add/update the lastVerifiedAttempt metadata property's timestamp to "now".
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param host the host name of the RP site where user is verified
	 * @param username the user name to login to the RP site
	 * @param success the verifying result
	 */
	public void connectionNotify(String userId, String password, String host, String username, boolean success) throws MalformedURLException;

	/**
	 * The RP site invokes this method to verify that user really has an account on the specified site.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param host The verified site url.
	 * @param username The username to login onto verified site.
	 * @return The result of verification process.
	 */
	public boolean connectionVerify(String userId, String password, String host, String username) throws MalformedURLException;

	/**
	 * Gets a form filling script for the specified site&user
	 * 
	 * @param page the page url where user lands on
	 * @param username the user name
	 * @param form the HTML form name
	 * @return a form filling script
	 */
	public String getFormFillingScript(URL page, String username, String form);

	/**
	 * Gets an HTML scraping script for the specified site
	 * 
	 * @param page the page url
	 * @return an HTML scraping script
	 */
	public String getHtmlScrapingScript(String page) throws MalformedURLException, RppsException;

	/**
	 * Sets an HTML scraping script for the specified site
	 * 
	 * @param page the page url
	 * @param script the script body
	 */
	public void setHtmlScrapingScript(String page, String script) throws MalformedURLException, RppsException;

	/**
	 * Creates a new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param id the i-card id
	 * @param properties the i-card properties
	 * @return a new card id
	 */
	public ICardTO createICard(String userId, String password, String id, String[] properties) throws RppsException;

	/**
	 * Creates a new card by template
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param templete template
	 * @return a new card id
	 */
	public ICardTO createICard(String userId, String password, ICardTemplateTO template) throws RppsException;

	/**
	 * Creates the duplicate card with new card's name
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cardName new card's name
	 * @param cuid cuid of old card
	 * @param pinCode pincode if it needs
	 * @return card trasfer object
	 * @throws RppsException
	 */
	public ICardTO createDuplicateICard(String userId, String password, String cardName, String cuid, String pinCode) throws RppsException;

	/**
	 * Creates new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cardname name of card
	 * @param surname user's last name
	 * @param givenname user's first name
	 * @param emailaddress user's email
	 * @return cuid of new card
	 * @throws RppsException
	 */
	public String createICardFromHBX(String userId, String password, String cardname) throws RppsException;

	/**
	 * Gets a card by CUID.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card CUID.
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUID(String userId, String password, String cuid) throws RppsException;

	/**
	 * Gets a card by CUID and Credential.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card CUID.
	 * @param tokenCredential the tokenCredential transfer object
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUIDAndTokenServiceCredential(String userId, String password, String cuid, UITokenServiceCredentialTO tokenCredential) throws RppsException;

	/**
	 * Gets the card list of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return the card list
	 */
	public ICardTO[] getICardsByUser(String userId, String password) throws RppsException;

	/**
	 * Gets the card and category lists of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return the card list
	 */
	public ICardsAndCategoryTO getICardsAndCategoriesByUser(String userId, String password) throws RppsException;

	/**
	 * Update card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param icardTO the card transfer object.
	 * @return the card transfer object.
	 */
	public ICardTO updateICard(String userId, String password, ICardTO icardTO) throws RppsException;

	/**
	 * Imports card from a file.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd the card as a byte array.
	 */
	public ICardTO[] importICards(String userId, String password, byte[] crd) throws RppsException;

	/**
	 * Deletes specified card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the i-card CUID
	 */
	public void deleteICard(String userId, String password, String cuid) throws RppsException;

	/**
	 * Deletes all user's card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * 
	 */
	public void deleteAllICard(String userId, String password) throws RppsException;

	/**
	 * This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object should contain all information necessary to display both (a) a user interface which shares the same card metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface paradigms to be developed in the future
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param url the URL to the html page with <object> tag
	 * @param target
	 * @param sslCert the SSL certificate
	 * @param policyType the RP Security Policy type
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RemoteException
	 */

	public UIResponseTO getUserInterface(String userId, String password, String url, String target, String sslCert, String policytype, String policy) throws RppsException;

	/**
	 * Web service operation. Makes xmlToken usage claim list.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param user the username to login to the STS
	 * @param pass the password to login to the STS
	 * @param sels the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @return the security token as a String
	 * @throws RppsException
	 */
	public TokenResponseTO getTokenByClaims(String sslCert, String[] claimName, String[] claimValue) throws RppsException;

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param cuids the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential the type of selected credential
	 * @param credentialKey the names of fields of selected credential
	 * @param credentialValue the value of fields of selected credential
	 * @return the security token as a String
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenObject(String userId, String password, String policy, String policytype, String sslCert, String[] cuids, String typeofCredential, String[] credentialKey, String[] credentialValue) throws RppsException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token as a String
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenByUrl(String userId, String password, String policy, String policytype, String sslCert, String url) throws RppsException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token as a String with extra card's information  
	 * @throws RemoteException
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(String userId, String password, String policy, String policytype, String sslCert, String url) throws RppsException;
	/**
	 * Return templates for creating cards
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template list
	 */
	public ICardTemplateTO[] getICardCreationTemplate(String userId, String password) throws RppsException;

	public ICardTemplate[] getICardCreationTemplateByProvider(CallbackHandler handler, String extID) throws RppsException;

	/**
	 * Get a icard template for creating personal card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template
	 */
	public ICardTemplateTO getPersonalICardCreationTemplate(String userId, String password) throws RppsException;

	public ICardTO setPinCode(String userId, String password, String cuid, String pinCode) throws RppsException;

	public ICardTO resetPinCode(String userId, String password, String cuid, String oldPinCode) throws RppsException;

	public ICardTO editPinCode(String userId, String password, String cuid, String oldPinCode, String newPinCode) throws RppsException;

	public void addCardHistory(CallbackHandler handler, CUID cuid, URL site) throws RppsException;

	public List getCardHistory(CallbackHandler handler, CUID cuid) throws RppsException;

	public CardInformationTO clearCardHistory(String userId, String password, String cuid) throws RppsException;

	public CategoryTO[] getCategory(String userId, String password) throws RppsException;

	public CategoryTO addCategory(String userId, String password, CategoryTO categoryTO) throws RppsException;

	public void deleteCategory(String userId, String password, String categoryId) throws RppsException;

	public CategoryTO modifyCategory(String userId, String password, CategoryTO categoryTO) throws RppsException;

	public UserProfileTO getUserProfile(String userId, String password) throws RppsException;

	/**
	 * Creates new user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param firstName first name of user
	 * @param lastName last name of user
	 * @param email email of user
	 * @throws RppsException
	 */
	public void addUserProfileFromHBX(String userId, String password) throws RppsException;

	/**
	 * Creates new user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param userProfile user profile trasfer object
	 * @throws RppsException
	 */
	public void addUserProfile(String userId, String password, UserProfileTO userProfile) throws RppsException;

	/**
	 * Deletes user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @throws RppsException
	 */
	public void deleteUserProfile(String userId, String password) throws RppsException;

	/**
	 * Checks availability of user profile by userIdentifier
	 * 
	 * @param userIdentifier
	 * @return true if user profile exists otherwise false
	 * @throws RppsException
	 */
	public boolean userIdentifierExists(String userIdentifier) throws RppsException;

	/**
	 * Update user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param userProfile user profile transfer object
	 * @return user profile transfer object
	 * @throws RppsException
	 */
	public UserProfileTO modifyUserProfile(String userId, String password, UserProfileTO userProfile) throws RppsException;

	/**
	 * 
	 * @param userId
	 * @param password
	 * @param cuid
	 * @return
	 * @throws RppsException
	 */
	public CardInformationTO getCardInformation(String userId, String password, String cuid) throws RppsException;

	public CardInformationTO clearCardCredential(String userId, String password, String cuid) throws RppsException;

	public CardInformationTO addUrlCardRelation(String userId, String password, String cuid, String url) throws RppsException;

	public CardInformationTO deleteUrlCardRelation(String userId, String password, String cuid, String url) throws RppsException;

	public CardInformationTO deleteUrlsCardRelation(String userId, String password, String cuid, String[] urls) throws RppsException;

	/**
	 * Gets list of random word
	 * 
	 * @param count
	 * @return list of word
	 * @throws RppsException
	 */
	public String[] getRandomWordList(int count) throws RppsException;

	/**
	 * Gets the global i-name registered to the Higgins service (e.g. "=parity").
	 * 
	 * @return the global i-name.
	 */
	public String getHigginsServiceGlobalIName();	
}
